package cz.cuni.amis.pogamut.emohawk.agent.losChecker;

import javax.vecmath.Matrix3d;
import javax.vecmath.Vector3d;

import cz.cuni.amis.pogamut.base.communication.command.IAct;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.emohawk.agent.EhAgentInfo;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.IPureHistoricWorldView;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Trace;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.TraceResponse;
import cz.cuni.amis.pogamut.ut2004.utils.UnrealUtils;
import cz.cuni.amis.utils.listener.IListener;
import cz.cuni.amis.utils.listener.Listeners;

/** Ray tracing line of sight checker
 * 
 * Note: Since traces are not part of world snapshot and may report intermediate state between two snapshots, there can be inconsistencies.
 * For example when checked location happens to be visible only in the intermediate state, the reported result is a false positive. 
 * 
 * @author Paletz
 */
public class RayTracingLosChecker implements ILosChecker {
	
	public static String DEFAULT_TRACE_COMMAND_PREFIX = "LosChecker";
	
	protected IAct act;
	protected IPureHistoricWorldView worldView;
	protected EhAgentInfo agentInfo;
	
	protected String traceCommandPrefix;
	protected int nextTraceIndex;
	
	protected Listeners<IListener<LosReportEvent>> losReportListenrs;
	
	/** Constructor
	 * 
	 * @param act used to issue {@link Trace} commands
	 * @param worldView used to listen for trace responses 
	 * @param agentInfo used to compute agent's field of view
	 */
	public RayTracingLosChecker( 
		IAct act, 
		IPureHistoricWorldView worldView,
		EhAgentInfo agentInfo
	) {
		this.act = act;
		this.worldView = worldView;
		this.agentInfo = agentInfo;
		losReportListenrs = new Listeners<IListener<LosReportEvent>>();
		traceCommandPrefix = DEFAULT_TRACE_COMMAND_PREFIX;
		nextTraceIndex = 0;
		
		worldView.accessEvents().addEventListener( TraceResponse.class,  traceResponseListener );
	}
	
	/** Dispose
	 */
	public void dispose() {
		worldView.accessEvents().removeEventListener( TraceResponse.class, traceResponseListener );
	}
	
	/** Set name prefix of issued {@link Trace} commands
	 * 
	 * Can be used to identify responses. Default prefix is {@link #DEFAULT_TRACE_COMMAND_PREFIX}.
	 * 
	 * @param prefix prefix of issued {@link Trace} commands
	 */
	public void setTraceCommandPrefix( String prefix ) {
		this.traceCommandPrefix = prefix;
	}
	
	@Override
	public void requestLosCheck(Location location) {
		act.act( new Trace( traceCommandPrefix+nextTraceIndex, null, location, true ) );
		++nextTraceIndex;
	}

	@Override
	public void registerLosReportListener(IListener<LosReportEvent> listener) {
		losReportListenrs.addStrongListener( listener );
	}

	@Override
	public void forgetLosReportListener(IListener<LosReportEvent> listener) {
		losReportListenrs.removeListener( listener );
	}
	
	@Override
	public boolean isWithinFov( Location location ) {
	    Location locationOffset = location.getLocation().sub( agentInfo.getPawn().getLocation() ).getNormalized();
	    
		double azimuth = UnrealUtils.unrealDegreeToRad( agentInfo.getPawn().getRotation().getYaw() );
		double elevation = UnrealUtils.unrealDegreeToRad( agentInfo.getPawn().getRotation().getPitch() );
		Matrix3d matrix = new Matrix3d();
		matrix.rotY( elevation );
		matrix.rotZ( azimuth );
		Vector3d vector = new Vector3d( 1.0, 0.0, 0.0 );
		matrix.transform( vector );
		Location focusOffset = new Location( vector );
		
	    double angleBetweenFocusAndLocation = Math.acos( locationOffset.dot( focusOffset ) ); 
	    
	    // assume bot FoV is 120 degrees, so location must be less than 60 degrees from focus 
	    return ( 
	    	angleBetweenFocusAndLocation < 60.0 / 360.0 * 2 * Math.PI 
	    );
	}
	
	public void processTraceResponse( TraceResponse traceResponse ) {
		if ( !traceResponse.isTraceActors() ) {
			// ignores actors that could block vision
			return;
		}
		
		Location location = traceResponse.getTo();
		boolean isInLos = !traceResponse.isResult() && !isWithinFov( location );
		
		losReportListenrs.notify(
			new IListener.Notifier<IListener<LosReportEvent>>( new LosReportEvent( location, isInLos ) )
		);
	}
	
	protected final IWorldEventListener<TraceResponse> traceResponseListener = (
		new IWorldEventListener<TraceResponse>() {
			@Override
			public void notify( TraceResponse event ) {
				processTraceResponse(event);
			}
		}
	);
}