package cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import cz.cuni.amis.pogamut.emohawk.communication.messages.replication.IReplicationEvent;
import cz.cuni.amis.pogamut.emohawk.communication.messages.replication.ObjectReplication;
import cz.cuni.amis.pogamut.emohawk.communication.messages.replication.ObjectTearOff;
import cz.cuni.amis.pogamut.emohawk.communication.messages.replication.ObjectUpdate;
import cz.cuni.amis.pogamut.emohawk.communication.messages.replication.ReplicationCommit;
import cz.cuni.amis.pogamut.emohawk.communication.stream.DecoderStream;
import cz.cuni.amis.pogamut.emohawk.communication.stream.IEncodedObjectInputStream;
import cz.cuni.amis.pogamut.emohawk.communication.stream.IObjectInputStream;
import cz.cuni.amis.pogamut.emohawk.communication.stream.PayloadType;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.batchClock.ISimulationClock;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.event.IReplicaEvent;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.event.ReplicaCreatedEvent;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.event.ReplicaTornOffEvent;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.event.ReplicaUpdatedEvent;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.iface.object.IObjectReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.action.ActionRegistryReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.attribute.AttributeManagerReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.attribute.IAttributeReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.attribute.foggyRef.FoggyRefAttributeReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.attribute.foggyRefList.FoggyRefListAttributeReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.attribute.primitive.PrimitiveAttributeReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.attribute.primitiveList.ListAttributeReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.ds.FoggyReferenceReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.ds.ListMapEntryReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.ds.ListMapReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.ds.PrimitiveBoxReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.ds.sll.SinglyLinkedListNodeReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.ds.sll.SinglyLinkedListReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.game.GameReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.object.IGenericObjectReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.object.SpecializedClass;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.pawn.PawnReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.player.PlayerReplica;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.replication.replica.impl.worldObject.SituationReplica;
import cz.cuni.amis.utils.listener.IListener;
import cz.cuni.amis.utils.listener.Listeners;

/** Emohawk object replication client
 * 
 * Implements java side of object replication.
 * <p>
 * Implementation of some replicated objects may require additional modules. 
 * Such modules may be registered before replica is initialized via {@link #registerModule} and later obtained by {@link #getModule}.
 * <p>
 * <b>Implementation details</b>
 * <p>
 * Replication in java is a bit more complicated because Unreal classes must be mapped to Java classes.
 * A class map exists for that purpose.
 * <p>
 * This is further complicated by templated unreal classe. Because templated class can access it's type parameter in the runtime it 
 * can't be mapped to a simple generic class. However, the type parameter can be stored manually, see {@link IGenericObjectReplica}. 
 * <p>
 * Because type parameter may be such generic replicable class too, it is not sufficient to store type parameter as a Class<?>. 
 * Type parameter of type parameter woulld be lost. The {@link SpecializedClass} is defined to work around this limitation.
 * 
 * @author Paletz
 *
 */
public class ObjectReplicationClient implements IObjectReplicationClient {

	protected static String LETTER = "[a-zA-Z]";
	protected static String WORD = "(?:"+LETTER+"+)";
	protected static String RAW_PARAMETER = "(?:(?:"+LETTER+"|(?:__))+)";
	protected static Pattern GENERIC_CLASS_NAME_PATTERN = Pattern.compile( WORD+"(?:_"+RAW_PARAMETER+")*" );
	protected static Pattern GENERIC_CLASS_TYPE_PARAMETER_PATTERN = Pattern.compile(RAW_PARAMETER);
	
	protected HashMap<String,Class<?>> classMap;
	protected HashMap<Integer,IObjectReplica> objectMap;
	protected ISimulationClock simulationClock;
	
	protected LinkedList<IReplicaEvent> outgoingEventBatch;
	
	/** Object event  (added/changed/removed) listeners
	 */
	protected Listeners<IListener<IReplicaEvent>> objectEventListeners;
	
	/** Constructor
	 */
	public ObjectReplicationClient() {
		this.objectMap = new HashMap<Integer,IObjectReplica>();
		this.classMap = new HashMap<String,Class<?>>();
		this.objectEventListeners = new Listeners<IListener<IReplicaEvent>>();
		this.outgoingEventBatch = new LinkedList<IReplicaEvent>();
		
		classMap.put( "int", Integer.class );
		classMap.put( "bool", Boolean.class );
		classMap.put( "float", Float.class );
		classMap.put( "string", String.class );
		
		classMap.put( "EhActionRegistry", ActionRegistryReplica.class );
		
		classMap.put( "EhIAttribute", IAttributeReplica.class );
		classMap.put( "EhFoggyRefAttribute", FoggyRefAttributeReplica.class );
		classMap.put( "EhFoggyRefListAttribute", FoggyRefListAttributeReplica.class );
		classMap.put( "EhPrimitiveAttribute", PrimitiveAttributeReplica.class );
		classMap.put( "EhListAttribute", ListAttributeReplica.class );
		classMap.put( "EhAttributeManager", AttributeManagerReplica.class );
		
		classMap.put( "EhSinglyLinkedListNode", SinglyLinkedListNodeReplica.class );
		classMap.put( "EhSinglyLinkedList", SinglyLinkedListReplica.class );
		classMap.put( "EhFoggyRef", FoggyReferenceReplica.class );
		classMap.put( "EhListMapEntry", ListMapEntryReplica.class );
		classMap.put( "EhListMap", ListMapReplica.class );
		classMap.put( "EhPrimitiveBox", PrimitiveBoxReplica.class );
		
		classMap.put( "EhGame", GameReplica.class );
		
		classMap.put( "EhIReplicableObject", IObjectReplica.class );
		
		classMap.put( "EhPawn", PawnReplica.class );
		
		classMap.put( "EhController", PlayerReplica.class );
		
		classMap.put( "EhActorSituationMirror", SituationReplica.class );
	}
	
	@Override
	public void initSimulationClock( ISimulationClock simulationClock ) {
		this.simulationClock = simulationClock;
	}
	
	/** Get object by replication ID
	 * 
	 * It is an error to request undefined object.
	 * 
	 * @param replicationId replication ID
	 * @return object with required ID
	 */
	public IObjectReplica getObject( int replicationId )
	{
		assert( replicationId >= 0 );
		assert( objectMap.containsKey( replicationId ) );
		return objectMap.get(replicationId);
	}
	
	/** Tell if there is an replicated object with given ID
	 * 
	 * @param replicationId replication ID
	 * @return true if there is an replicated object with given ID
	 */
	public boolean exists( int replicationId )
	{
		assert( replicationId >= 0 );
		
		return objectMap.containsKey( replicationId );
	}
	
	@Override
	public void registerReplicaEventListener( IListener<IReplicaEvent> listener ) {
		objectEventListeners.addStrongListener( listener );
	}

	@Override
	public void forgetReplicaEventListener( IListener<IReplicaEvent> listener ) {
		objectEventListeners.removeListener( listener );
	}
	
	/** Find class by name.
	 * 
	 * Result of replication call originating from the server.
	 * 
	 * @param objectClass Unreal script class name.
	 * @return specialized class ( class + type parameters if it is a generic class ) 
	 */
	protected SpecializedClass<?> findClass( String objectClass )
	{
		if ( ! GENERIC_CLASS_NAME_PATTERN.matcher( objectClass ).matches() ) {
			throw new RuntimeException( "Invalid class name \""+objectClass+"\"." );
		}
		
		List<String> typeParameters = new LinkedList<String>();
		String className;
		if ( objectClass.contains("_") ) {
			className = objectClass.substring( 0, objectClass.indexOf( "_" ) );
			
			String parameterString = objectClass.substring( className.length()+1 );
			Matcher parameterMatcher = GENERIC_CLASS_TYPE_PARAMETER_PATTERN.matcher( parameterString );
			while ( parameterMatcher.find() ) {
				String parameter = parameterMatcher.group(0);
				typeParameters.add( parameter.replace( "__", "_" ) );
			}
		} else {
			className = objectClass;
		}
		
		if ( !classMap.containsKey(className) ) {
			throw new RuntimeException( "Can't replicate unknown class \""+objectClass+"\"." );
		}
		
		Class<?> genericClass = classMap.get(className);
				
		if ( genericClass.getTypeParameters().length != typeParameters.size() ) {
			throw new RuntimeException( "Incorrect number of type parameters for \""+className+"\" ( "+typeParameters+" )." );
		}
		
		LinkedList<SpecializedClass<?>> resolvedTypeParameters = new LinkedList<SpecializedClass<?>>();
		for ( String typeParameter : typeParameters ) {
			resolvedTypeParameters.add( findClass(typeParameter) );
		}
		
		@SuppressWarnings({ "unchecked", "rawtypes" })
		SpecializedClass<?> retval = new SpecializedClass<Object>( (Class) genericClass, resolvedTypeParameters );
		
		return retval;
	}
	
	@Override
	public void applyReplicationEvent( IReplicationEvent event ) {
		
		if ( event instanceof ObjectReplication ) {
			ObjectReplication objectReplicationEvent = (ObjectReplication) event;
			replicateObject( objectReplicationEvent.getReplicationIndex(), objectReplicationEvent.getObjectClass() );
		} else if ( event instanceof ObjectUpdate ) {
			ObjectUpdate objectUpdateEvent = (ObjectUpdate) event;
			updateObject( objectUpdateEvent.getReplicationIndex(), objectUpdateEvent.getPayloadStream() );
		} else if ( event instanceof ObjectTearOff ) {
			tearOffObject( ((ObjectTearOff) event).getReplicationIndex() );	
		} else if ( event instanceof ReplicationCommit ) {
			commit();
		} else {
			throw new AssertionError( "Unexpected replication event." );
		}
	}
	
	public ISimulationClock getSimulationClock() {
		return simulationClock;
	}
	
	/** Replicate object.
	 * 
	 * @param replicationIndex replication index the newly replicated object has to assume
	 * @param objectClass class of the object 
	 */
	protected void replicateObject( int replicationIndex, String objectClass ) {
		assert( replicationIndex >= 0 );
		assert( ! exists( replicationIndex ) );
		SpecializedClass<?> specializedClass = findClass( objectClass );
		assert( IObjectReplica.class.isAssignableFrom( specializedClass.getGenericClass() ) );
		
		IObjectReplica object;
		try {
			object = (IObjectReplica) specializedClass.getGenericClass().newInstance();
		} catch (InstantiationException e) {
			throw new RuntimeException( e );
		} catch (IllegalAccessException e) {
			throw new RuntimeException( e );
		}
		
		if ( specializedClass.getGenericClass().getTypeParameters().length > 0 ) {
			IGenericObjectReplica genericObject = (IGenericObjectReplica) object;
			for ( int i=0; i<specializedClass.getGenericClass().getTypeParameters().length; ++i ) {
				genericObject.setTypeParameter( i, specializedClass.getTypeParameter( i ) );
			}
		}
		
		object.initializeReplica( ObjectReplicationClient.this, replicationIndex );
		objectMap.put( replicationIndex, object );
		
		outgoingEventBatch.add(
			new ReplicaCreatedEvent( object )
		);
	}
	
	/** Update object.
	 * 
	 * @param replicationId replication ID the object to update
	 * @param encodedStream input stream with update data
	 * @param expectedClass error detection
	 */
	protected void updateObject( int replicationIndex, IEncodedObjectInputStream encodedStream ) {
		IObjectReplica object = getObject(replicationIndex);
		
		IObjectInputStream stream = new DecoderStream( encodedStream ) {
			@Override
			protected IObjectReplica decode( int objectReference ) {
				return getObject( objectReference );
			}
		};
		object.receive( stream );
		assert( stream.tellNext() == PayloadType.PAYLOAD_TYPE_EOF );
		
		outgoingEventBatch.add(
			new ReplicaUpdatedEvent( object )
		);
	}
	
	/** Tear-off object.
	 * 
	 * Internal. Result of tear-off call originating from the server.
	 * 
	 * @param replicationId replication ID the object to tear-off
	 * @param expectedClass error detection
	 */
	protected void tearOffObject( int replicationIndex ) {
		assert( exists( replicationIndex ) );
		
		IObjectReplica object = objectMap.get(replicationIndex);
		
		objectMap.remove( replicationIndex );
		
		object.finalizeReplication();
		
		outgoingEventBatch.add(
			new ReplicaTornOffEvent( object )
		);
	}
	
	/** Commit changes
	 *
	 * ATM only outgoing events are buffered; replicas are updated immediatelly.
	 * The important bit is that ReplicaCreatedEvent is sent after the replica receive()d it's value.
	 */
	protected void commit() {
		// Remove all events of objects that were replicated and subsequently destroyed.
		// Such objects were never properly initialized, and would cause an error.
		Map<IObjectReplica,ReplicaCreatedEvent> suspects = new HashMap<IObjectReplica,ReplicaCreatedEvent>();
		List<IReplicaEvent> intermediateEvents = new ArrayList<IReplicaEvent>();
		for ( IReplicaEvent event : outgoingEventBatch ) {
			if ( event instanceof ReplicaCreatedEvent) {
				suspects.put( event.getReplica(), (ReplicaCreatedEvent) event );
			} else if ( event instanceof ReplicaUpdatedEvent ) {
				// check if event applies to long-term object or a suspect
				if ( suspects.containsKey( event.getReplica() ) ) {
					// object was updated therefore properly initialized
					suspects.remove( event.getReplica() );
				}
			} else if ( event instanceof ReplicaTornOffEvent ) {
				if ( suspects.containsKey( event.getReplica() ) ) {
					// suspect was never initialized, add mark related events as intermediate
					intermediateEvents.add( suspects.get( event.getReplica() ) );
					intermediateEvents.add( event );
					suspects.remove( event.getReplica() );
				}
			}
		}
		
		outgoingEventBatch.removeAll( intermediateEvents );
		
		// send remaining events
		for ( IReplicaEvent event : outgoingEventBatch ) {
			objectEventListeners.notify(
				new IListener.Notifier<IListener<IReplicaEvent>>(
					event
				)
			);
		}
		
		// clear batch
		outgoingEventBatch.clear();
	}
}