package cz.cuni.amis.pogamut.emohawk.communication.stream;

import java.util.ArrayList;
import java.util.List;

import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.base3d.worldview.object.Rotation;
import cz.cuni.amis.pogamut.base3d.worldview.object.Velocity;

/** Serialization tools
 * 
 * NOTE: Keep serialization format in sync with unreal script (Emohawk/EhSerializationTools).
 * 
 * @author Paletz
 */
public class SerializationTools {
	
	/** Serialize location
	 * 
	 * Unreal script equivalent is Vector.
	 * Accuracy is reduced to float.
	 * 
	 * @param stream stream to serialize to
	 * @param location location to serialize
	 */
	public static void serializeLocation( IOutputStream stream, Location location )
	{
		stream.writeFloat( (float) location.getX() );
		stream.writeFloat( (float) location.getY() );
		stream.writeFloat( (float) location.getZ() );
	}
	
	/** Deserialize location
	 * 
	 * Unreal script equivalent is Vector.
	 * Actual accuracy is float.
	 * 
	 * @param stream stream to deserialize from
	 * @return deserialized location or null
	 */
	public static Location deserializeLocation( IInputStream stream )
	{
		float x,y,z;

		if ( stream.tellNext() != PayloadType.PAYLOAD_TYPE_FLOAT )
		{
			return null;
		}
		x = stream.readFloat();

		if ( stream.tellNext() != PayloadType.PAYLOAD_TYPE_FLOAT )
		{
			return null;
		}
		y = stream.readFloat();

		if ( stream.tellNext() != PayloadType.PAYLOAD_TYPE_FLOAT )
		{
			return null;
		}	
		z = stream.readFloat();

		return new Location( x, y, z );
	}
	
	/** Serialize velocity
	 * 
	 * Unreal script equivalent is Vector.
	 * Accuracy is reduced to float.
	 * 
	 * @param stream stream to serialize to
	 * @param velocity velocity to serialize
	 */
	public static void serializeVelocity( IOutputStream stream, Velocity velocity )
	{
		serializeLocation( stream, velocity.asLocation() );
	}
	
	/** Deserialize velocity
	 * 
	 * Unreal script equivalent is Vector.
	 * Actual accuracy is float.
	 * 
	 * @param stream stream to deserialize from
	 * @return deserialized velocity or null
	 */
	public static Velocity deserializeVelocity( IInputStream stream )
	{
		Location location = deserializeLocation( stream );
		if ( location == null ) {
			return null;
		} else {
			return new Velocity( location.asVector3d() );
		}
	}

	/** Serialize rotation
	 *  
	 * Accuracy of components is reduced to integer.
	 *  
	 * @param stream stream to serialize to
	 * @param serializedRotation rotation to serialize
	 */
	public static void serializeRotation( IOutputStream stream, Rotation serializedRotation )
	{
		stream.writeInt( (int) Math.round( serializedRotation.getPitch() )  );
		stream.writeInt( (int) Math.round( serializedRotation.getRoll() ) );
		stream.writeInt( (int) Math.round( serializedRotation.getYaw() ) );
	}

	/** Deserialize rotation
	 * 
	 * Actual accuracy is of components is integer.
	 * 
	 * @param stream stream to deserialize from
	 * @return deserialized rotator successful, null otherwise
	 */
	public static Rotation deserializeRotation( IInputStream stream )
	{
		if ( stream.tellNext() != PayloadType.PAYLOAD_TYPE_INT )
		{
			return null;
		}
		double pitch = stream.readInt();

		if ( stream.tellNext() != PayloadType.PAYLOAD_TYPE_INT )
		{
			return null;
		}
		double roll = stream.readInt();

		if ( stream.tellNext() != PayloadType.PAYLOAD_TYPE_INT )
		{
			return null;
		}	
		double yaw = stream.readInt();

		return new Rotation( pitch, yaw, roll );
	}
	
	/** Serialize a list
	 * 
	 * @param stream stream to serialize to
	 * @param list list to serialize
	 */
	public static <Element> void serializeList( IOutputStream stream, List<Element> list )
	{
		stream.writeInt( list.size() );
		for ( Element element : list )
		{
			PrimitiveTools.writePrimitive( element, stream );
		}
	}
	
	
	/** Deserialize a list
	 * 
	 * @param stream stream to deserialize from
	 * @param list list to append to deserialized items
	 * @return true if successful, false otherwise
	 */
	public static <Element> List<Element> deserializeList( Class<Element> elementClass, IObjectInputStream stream )
	{
		List<Element> retval = new ArrayList<Element>();
				
		if ( stream.tellNext() != PayloadType.PAYLOAD_TYPE_INT )
		{
			return null;
		}
		
		int size = stream.readInt();
	
		for ( int i=0; i<size; ++i )
		{
			if ( !PrimitiveTools.hasPrimitive( elementClass, stream ) )
			{
				return null;
			}
			retval.add( PrimitiveTools.readPrimitive( elementClass, stream ) );
		}
		
		return retval;
	}
}