package cz.cuni.amis.pogamut.emohawk.agent.action;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import cz.cuni.amis.pogamut.base.communication.command.IAct;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.emohawk.agent.EhAgentInfo;
import cz.cuni.amis.pogamut.emohawk.agent.game.EhGame;
import cz.cuni.amis.pogamut.emohawk.communication.messages.action.ActionResult;
import cz.cuni.amis.pogamut.emohawk.communication.messages.action.PerformAction;
import cz.cuni.amis.pogamut.emohawk.communication.stream.IInputStream;
import cz.cuni.amis.pogamut.emohawk.communication.stream.StreamTools;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.IPureHistoricWorldView;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.ontology.action.IAction;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.iface.action.IActionRegistryBelief;
import cz.cuni.amis.pogamut.emohawk.communication.worldView.worldObjectUpdater.historic.sightingMemory.belief.iface.game.IGameBelief;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.EndMessage;

/** Action requesting module
 * 
 * @author Paletz
 */
public class Activeness {

	public static final int ERROR_INVALID_PERFORMER = 1;
	public static final int ERROR_INVALID_ACTION = 2;
	public static final int ERROR_ACCESS_DENIED = 3;
	
	public static final double DEFAULT_ACTION_TIMEOUT_PERIOD_SECONDS = 30;
	
	protected IAct act;
	protected IPureHistoricWorldView worldView;
	protected EhAgentInfo agentInfo;
	protected EhGame game;
	protected LogCategory log;
	
	protected IActionRegistryBelief actionRegistry;
	protected Map<Integer,HandlerRecord> resultHandlers;
	protected int nextUserRequestId;
	protected double timeoutPeriodSeconds;
	  
	
	/** Constructor
	 * 
	 * @param act act
	 * @param worldView world worldView
	 * @param agentInfo agentInfo
	 * @param game game
	 * @param log log
	 * @param timeoutPeriodSeconds action request timeout, if no result is received within timeout period, handler is freed
	 */
	public Activeness( IAct act, IPureHistoricWorldView worldView, EhAgentInfo agentInfo, EhGame game, LogCategory log, double timeoutPeriodSeconds ) {
		this.act = act;
		this.worldView = worldView;
		this.agentInfo = agentInfo;
		this.game = game;
		this.timeoutPeriodSeconds = timeoutPeriodSeconds;
		this.log = log;
		this.resultHandlers = new HashMap<Integer,HandlerRecord>();
		this.nextUserRequestId = 1;
		this.timeoutPeriodSeconds = 30.0;
		worldView.accessEvents().addEventListener( ActionResult.class, onInteractionError );
		worldView.accessEvents().addEventListener( EndMessage.class, endMessageListener );
	}
	
	/** Constructor
	 * 
	 * @param act act
	 * @param worldView world worldView
	 * @param agentInfo agentInfo
	 * @param game game
	 * @param timeoutPeriodSeconds action request timeout, if no result is received within timeout period, handler is freed 
	 * @param log log
	 */
	public Activeness( IAct act, IPureHistoricWorldView worldView, EhAgentInfo agentInfo, EhGame game, LogCategory log ) {
		this( act, worldView, agentInfo, game, log, DEFAULT_ACTION_TIMEOUT_PERIOD_SECONDS );
	}
	
	/** Get action registry
	 * 
	 * @return action registry
	 */
	public IActionRegistryBelief getActionRegistry() {
		return actionRegistry;
	}
	
	/** Request action
	 * 
	 * @param action action to perform
	 * @param resultHandler result handler, can be null
	 * @param serializedCustomArguments serialized custom arguments, can be null
	 */
	public void requestAction( IAction action, IActionResultHandler resultHandler, IInputStream serializedCustomArguments ) {
		cleanHandlers();
		
		int requestId = generateRequestId();
		
		if ( resultHandler != null ) {
			resultHandlers.put( requestId, new HandlerRecord( game.getTime()+timeoutPeriodSeconds, resultHandler ) );
		}
		
		PerformAction command = new PerformAction(
			(int) agentInfo.getPawn().getId().getLongId(),
			(int) action.getId().getLongId(), 
			requestId
		);
		
		if ( serializedCustomArguments != null ) {
			StreamTools.pipeStream( serializedCustomArguments, command );
		}
		
		act.act( command );
	}
	
	protected void cleanHandlers() {
		Iterator<HandlerRecord> iterator = resultHandlers.values().iterator();
		while ( iterator.hasNext() ) {
			HandlerRecord record = iterator.next();
			if ( record.timeoutTime < game.getTime() ) {
				
				if ( log != null ) {
					log.warning( "Action result handler timed out." );
				}
				
				iterator.remove();
			}
		}
	}
	
	protected final IWorldEventListener<ActionResult> onInteractionError = new IWorldEventListener<ActionResult>() {
		@Override
		public void notify(ActionResult event) {
			cleanHandlers();
			
			HandlerRecord handlerRecord = resultHandlers.get( event.getUserRequestId() );
			if ( handlerRecord != null ) {
				handlerRecord.handler.handle( event.getMajor(), event.getMinor(), event.getMessage() );
				
				if ( 
					event.getMajor() == MajorActionResult.IR_SUCCESS
					||
					event.getMajor() == MajorActionResult.IR_FAILURE
				) {
					resultHandlers.remove( event.getUserRequestId() );
				} else {
					handlerRecord.timeoutTime = game.getTime()+timeoutPeriodSeconds;
				}
			}
		}
	};
	
	protected final IWorldEventListener<EndMessage> endMessageListener = new IWorldEventListener<EndMessage>() {
		@Override
		public void notify(EndMessage event) {
			Collection<IGameBelief> games = worldView.getPlausibleBeliefs().filterByClass( IGameBelief.class ).values();
			if ( games.size() == 0 ) {
				return;
			}
			
			assert( games.size() == 1 );
			
			actionRegistry = games.iterator().next().getActionRegistry();
		}
	};
	
	protected int generateRequestId() {
		return nextUserRequestId++;
	}
	
	static class HandlerRecord {
		
		public double timeoutTime;
		public IActionResultHandler handler;
		
		public HandlerRecord( double timeoutTime, IActionResultHandler handler ) {
			this.timeoutTime = timeoutTime;
			this.handler = handler;
		}
	}
}
