package cz.cuni.amis.pogamut.emohawk.agent.module.observationMemory;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import javax.vecmath.Matrix3d;
import javax.vecmath.Vector3d;

import cz.cuni.amis.pogamut.base.communication.command.IAct;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObject;
import cz.cuni.amis.pogamut.base3d.worldview.IVisionWorldView;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.emohawk.agent.module.essence.IEssenceMap;
import cz.cuni.amis.pogamut.emohawk.agent.module.observationMemory.memorization.PawnMemorization;
import cz.cuni.amis.pogamut.emohawk.agent.module.observationMemory.memorization.essence.IEssenceMemorization;
import cz.cuni.amis.pogamut.emohawk.agent.module.observationMemory.memorization.object.IObjectMemorization;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.PawnReplica;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.essence.IEssenceReplica;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.object.IObjectReplica;
import cz.cuni.amis.pogamut.emohawk.agent.module.replication.image.object.IUniqueGameObject;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.Trace;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.EndMessage;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.TraceResponse;
import cz.cuni.amis.pogamut.ut2004.utils.UnrealUtils;

public class ObservationMemory implements IObservationMemory {
	
	/** Trace exposing radius
	 * 
	 * Radius around trace target that is considered visible if trace is not blocked. 
	 */
	public static final double TRACE_EXPOSING_RADIUS = 1.0;
	
	protected IAct act;
	protected IVisionWorldView worldView;
	protected IEssenceMap essenceMap;
	protected AgentInfo agentInfo;
	protected Map<IEssenceMemorization<?>,SightingInfo> sightings;
	protected Map<IObjectReplica,IObjectMemorization<?>> memorizationCache;
	protected Map<IObjectMemorization<?>,IObjectReplica> reverseMemorizationCache;
	protected int tracesPerFrame;
	protected double currentEpochTime;
	
	public ObservationMemory( 
		IAct act, 
		IVisionWorldView worldView, 
		IEssenceMap essenceMap,
		AgentInfo agentInfo
	) {
		super();
		this.act = act;
		this.worldView = worldView;
		this.essenceMap = essenceMap;
		this.agentInfo = agentInfo;
		sightings = new HashMap<IEssenceMemorization<?>,SightingInfo>();
		worldView.addEventListener( EndMessage.class, endMessageListener );
		worldView.addEventListener( TraceResponse.class, traceResponseListener );
		tracesPerFrame = 1;
		memorizationCache = new HashMap<IObjectReplica,IObjectMemorization<?>>();
		reverseMemorizationCache = new HashMap<IObjectMemorization<?>,IObjectReplica>();
		currentEpochTime = 0;
	}
	
	@Override
	public List<? extends IEssenceMemorization<?>> getAll() {
		List<IEssenceMemorization<?>> retval = new ArrayList<IEssenceMemorization<?>>();
		for ( IEssenceMemorization<?> memorization : getAllByMemorization( IEssenceMemorization.class ) ) {
			retval.add( memorization );
		}
		return retval;
	}
		
	@Override
	public <TMemorization extends IEssenceMemorization<?>> List<TMemorization> getAllByMemorization( Class<? extends TMemorization> memorizationClass ) {
		List<TMemorization> retval = new ArrayList<TMemorization>();
		for ( IEssenceMemorization<?> memorization : sightings.keySet() ) {
			if ( memorizationClass.isInstance( memorization ) ) {
				@SuppressWarnings("unchecked")
				TMemorization added = (TMemorization) memorization;
				retval.add( added );
			}
		}
		return retval;
	}

	@Override
	public void setImportance( IEssenceMemorization<?> memorizedEssence, double importance ) {
		assert( sightings.containsKey( memorizedEssence ) );
		assert( importance >= 0.0 );
		sightings.get( memorizedEssence ).importance = importance;
	}
	
	@Override
	public <TReplica extends IObjectReplica> IObjectMemorization<TReplica> getMemorization( TReplica object ) {
		if ( object == null ) {
			return null;
		}
		assert( object.isLive() );
		
		IObjectMemorization<?> memorization = memorizationCache.get( object );
		if ( memorization == null ) {
			SemifinishedMemorization semifinishedMemorization = makeMemorization(object);
			memorization = semifinishedMemorization.memorization;
			memorizationCache.put( object, memorization );
			reverseMemorizationCache.put( memorization, object);
			semifinishedMemorization.finishConstruction();
			
			if ( memorization instanceof IUniqueGameObject ) {
				int newMemorizationGameObjectId = ((IUniqueGameObject) memorization).getGameObjectId();
				IEssenceMemorization<?> contradictedSighting = null;
				for ( IEssenceMemorization<?> sighting : sightings.keySet() ) {
					if ( sighting.getGameObjectId() == newMemorizationGameObjectId ) {
						contradictedSighting = sighting;
					}
				}
				if ( contradictedSighting != null ) {
					contradict( contradictedSighting );
				}
			}
		}
		@SuppressWarnings("unchecked")
		IObjectMemorization<TReplica> retval = (IObjectMemorization<TReplica>) memorization;
		return retval;
	}
	
	@Override
	public void contradict(IEssenceMemorization<?> contradictedSighting) {
		sightings.remove( contradictedSighting );
	}

	@Override
	public <MemorizedClass extends IObjectReplica> MemorizedClass getPreimage( IObjectMemorization<MemorizedClass> freshMemorization ) {
		@SuppressWarnings("unchecked")
		MemorizedClass retval = (MemorizedClass) reverseMemorizationCache.get( freshMemorization );
		return retval;
	}
	
	@Override
	public double getEpochTime() {
		return currentEpochTime;
	}

	
	/** Make a new memorization of an observed object
	 * 
	 * @param object observed object
	 * @return memorization of the object
	 */
	protected SemifinishedMemorization makeMemorization( IObjectReplica object ) {
		SemifinishedMemorization retval = new SemifinishedMemorization();
		if ( object instanceof PawnReplica ) {
			retval.memorization = new PawnMemorization<PawnReplica>( (PawnReplica) object, this );
			return retval;
		}
		throw new AssertionError( "Unexpected object");
	}
	
	protected void observe() {
		memorizationCache.clear(); // without cache reset most memorizations wouldn't be updated 
		reverseMemorizationCache.clear();
		assert( currentEpochTime < agentInfo.getTime() );
		currentEpochTime = agentInfo.getTime();
		
		// remember sighted essences
		for ( Entry<IWorldObject,IEssenceReplica> entry : essenceMap.getAllVisible( IEssenceReplica.class ).entrySet() ) {
			IWorldObject seenWorldObject = entry.getKey();
			if ( ! ( seenWorldObject instanceof ILocated ) ) {
				continue;
			}
			sightings.put(
				(IEssenceMemorization<?>) getMemorization( entry.getValue() ),
				new SightingInfo( agentInfo.getTime() ) 
			);
		}
		
		for ( int i=0; i < tracesPerFrame; i++ ) {
			IEssenceMemorization<?> tracedSighting = findMemoryRecordWithTopTracePriority();
			if ( tracedSighting == null ) {
				break;
			}
			act.act( new Trace( "EmoMmr"+i, null, tracedSighting.getActorLocation(), true ) );
			sightings.get( tracedSighting ).lastTraceAttemptTime = agentInfo.getTime();
		}
	}
		
	protected IWorldEventListener<EndMessage> endMessageListener = ( 
		new IWorldEventListener<EndMessage>() {
			@Override
			public void notify( EndMessage event ) {
				observe();
			}
		}
	);
	
	protected IWorldEventListener<TraceResponse> traceResponseListener = (
		new IWorldEventListener<TraceResponse>() {
			@Override
			public void notify( TraceResponse event ) {
				if ( !event.isTraceActors() ) {
					// ignores actors that could block vision
					return;
				}
				
				if ( event.isResult() ) {
					// hit something, location is not visible
					return;
				}
				
				Location traceEnd = event.getTo();
				
				if ( !isWithinFov( traceEnd ) ) {
					// bot is not facing the location 
					return;
				}
				
				Iterator<IEssenceMemorization<?>> sightingIterator = sightings.keySet().iterator();
				while ( sightingIterator.hasNext() ) {
					IEssenceMemorization<?> sighting = sightingIterator.next();
					
					if ( 
						sighting.getMemorizationEpochTime() != currentEpochTime
						&&
						sighting.getActorLocation().getDistance( traceEnd ) < TRACE_EXPOSING_RADIUS
					) {
						// entity is not visible but entity location is visible -> contradicting observation
						sightingIterator.remove();
					}
				}
			}
		}
	);
	
	/** Find sighting with top trace priority
	 * 
	 * @return sighting with top trace priority or null if no record needs to be traced
	 */
	protected IEssenceMemorization<?> findMemoryRecordWithTopTracePriority() {
		IEssenceMemorization<?> sightingWithTopTracePriority = null;
		double topTracePriority = 0.0;
		for ( IEssenceMemorization<?> sighting : sightings.keySet() ) {
			double tracePriority = computeTracePriority( sighting );
			if ( topTracePriority < tracePriority ) {
				sightingWithTopTracePriority = sighting;
				topTracePriority = tracePriority;
			}
		}
		
		return sightingWithTopTracePriority;
	}
	
	protected double computeTracePriority( IEssenceMemorization<?> sighting ) {
		if ( !isWithinFov( sighting.getActorLocation() ) ) {
			return 0.0;
		}
		
		double timeSinceSeen = currentEpochTime - sighting.getMemorizationEpochTime();
		double timeSinceTraceAttempt = currentEpochTime - sightings.get( sighting ).lastTraceAttemptTime;
		double importance = sightings.get( sighting ).importance;
		double distance = Math.max( 50.0, agentInfo.getDistance( sighting.getActorLocation() ) );
		
		return Math.max( 0.0, importance*timeSinceSeen*timeSinceTraceAttempt/distance );
	}
	
	protected boolean isWithinFov( Location location ) {
	    Location locationOffset = location.getLocation().sub( agentInfo.getLocation() ).getNormalized();
	    
		double azimuth = UnrealUtils.unrealDegreeToRad( agentInfo.getRotation().getYaw() );
		double elevation = UnrealUtils.unrealDegreeToRad( agentInfo.getRotation().getPitch() );
		Matrix3d matrix = new Matrix3d();
		matrix.rotY( elevation );
		matrix.rotZ( azimuth );
		Vector3d vector = new Vector3d( 1.0, 0.0, 0.0 );
		matrix.transform( vector );
		Location focusOffset = new Location( vector );
		
	    double angleBetweenFocusAndLocation = Math.acos( locationOffset.dot( focusOffset ) ); 
	    
	    // assume bot FoV is 120 degrees, so location must be less than 60 degrees from focus 
	    return ( 
	    	angleBetweenFocusAndLocation < 60.0 / 360.0 * 2 * Math.PI 
	    );
	}
		
	protected class SightingInfo {
		public double importance;
		public double lastTraceAttemptTime;
		
		public SightingInfo( double lastTraceAttemptTime ) {
			importance = 1.0;
			this.lastTraceAttemptTime = lastTraceAttemptTime;
		}
	}
}
