package cz.cuni.amis.pogamut.defcon.communication.worldview.polygons;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.SortedMap;
import java.util.TreeMap;
import java.util.logging.Logger;

import javabot.PogamutJBotSupport;
import cz.cuni.amis.pogamut.base.agent.module.SensorModule;
import cz.cuni.amis.pogamut.base.component.controller.ComponentDependencies;
import cz.cuni.amis.pogamut.defcon.base3d.worldview.object.DefConLocation;
import cz.cuni.amis.pogamut.defcon.agent.DefConAgent;
import cz.cuni.amis.pogamut.defcon.communication.worldview.DefConWorldView;
import cz.cuni.amis.pogamut.defcon.communication.worldview.modules.grid.flags.BasicFlag;
import cz.cuni.amis.pogamut.defcon.communication.worldview.modules.grid.flags.IFlagChecker;
import cz.cuni.amis.pogamut.defcon.communication.worldview.polygons.loaders.borders.FilePrecomputedBordersLoader;
import cz.cuni.amis.pogamut.defcon.communication.worldview.polygons.loaders.borders.IPrecomputedBordersLoader;
import cz.cuni.amis.pogamut.defcon.communication.worldview.polygons.loaders.borders.IPrecomputedBordersSaver;
import cz.cuni.amis.pogamut.defcon.utils.Pair;
import cz.cuni.amis.utils.iterators.CircularListIterator;

/**
 * Module containing polygonal description of certain features of map.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class GameMapInfoPolygons extends SensorModule<DefConAgent> {

	/**
	 * Size of a sampling step.
	 */
	protected final float STEP_SIZE = 1f;
	/**
	 * Factor of polygon's area to vertices count. If the factor of count
	 * vertices to the area is greater than SIMPLIFICATION_FACTOR, then it gets
	 * simplified.
	 */
	protected final float SIMPLIFICATION_FACTOR = 0.1f;


	/**
	 * Territories. First member is sea, the second is land.
	 */
	protected ArrayList<Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>> territories =
			new ArrayList<Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>>();
	/**
	 * List of own territories.
	 */
	protected TreeMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>> ownTerritories =
			new TreeMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>>();
	/**
	 * List of enemy territories.
	 */
	protected TreeMap<Integer, SortedMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>>> enemyTerritories =
			new TreeMap<Integer, SortedMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>>>();
	/**
	 * Land.
	 */
	protected List<List<DefConLocation>> land;
	/**
	 * Sea.
	 */
	protected List<List<DefConLocation>> sea;

	/**
	 * Flag checker for queries to map features
	 */
	protected IFlagChecker flagChecker;

	/**
	 * Currently processed flag
	 */
	protected BasicFlag currentFlag;

	/**
	 * Current territory id in case of TERRITORY flags
	 */
	protected int currentTerritoryId;
	/**
	 * Current enemy id in case based on territory.
	 */
	protected int currentEnemyId;

	protected int lastAssignedTerritory;

	protected double minX;
	protected double minY;

	public GameMapInfoPolygons(DefConAgent<?> agent,
			IFlagChecker flagChecker) {
		this(agent, flagChecker, null, null);
	}

	public GameMapInfoPolygons(DefConAgent<?> agent,
			IFlagChecker flagChecker, String preprocessedPath) {
		this(agent, flagChecker, preprocessedPath, null);
	}

	public GameMapInfoPolygons(DefConAgent<?> agent,
			IFlagChecker flagChecker, String preprocessedPath, Logger log) {
		this(agent, flagChecker, preprocessedPath, log, null);
	}

	public GameMapInfoPolygons(DefConAgent<?> agent,
			IFlagChecker flagChecker, String preprocessedPath, Logger log,
			ComponentDependencies dependencies) {
		super(agent, log, dependencies);

		this.flagChecker = flagChecker;

		minX = agent.getWorldView().getMinX();
		minY = agent.getWorldView().getMaxX();

		for (int i = 0; i < agent.getWorldView().getGameInfo()
				.getTerritoriesCount(); ++i) {
			territories
					.add(new Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>());
		}

		for (int teamId : agent.getWorldView().getGameInfo().getTeamIds()) {
			for (int territoryId : agent.getWorldView().getGameInfo()
					.getTeamTerritories(teamId)) {

				Pair<List<List<DefConLocation>>, List<List<DefConLocation>>> territory =
						territories.get(territoryId);

				if (lastAssignedTerritory < territoryId)
					lastAssignedTerritory = territoryId;

				if (agent.getWorldView().getGameInfo().getOwnTeamId() == teamId) {
					ownTerritories.put(territoryId, territory);
				} else {
					SortedMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>> setOfEnemiesTerritories =
							new TreeMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>>();
					enemyTerritories.put(teamId, setOfEnemiesTerritories);
					setOfEnemiesTerritories.put(territoryId, territory);
				}
			}
		}

		if (preprocessedPath == null)
			processMap();
		else
			loadBorders(preprocessedPath);
		/*
		 * int i = 0; for (Pair<List<List<DefConLocation>>,
		 * List<List<DefConLocation>>> terri : territories) { if (terri.first !=
		 * null) { PogamutJBotSupport.writeToConsole("Terr sea " + i + ": " +
		 * terri.first); } if (terri.second != null) {
		 * PogamutJBotSupport.writeToConsole("Terr land " + i + ": " +
		 * terri.second); } ++i; }
		 * 
		 * for (int enemyId : enemyTerritories.keySet()) {
		 * 
		 * for (int territoryId : enemyTerritories.get(enemyId).keySet()) {
		 * 
		 * Pair<List<List<DefConLocation>>, List<List<DefConLocation>>> terri =
		 * enemyTerritories.get(enemyId).get(territoryId);
		 * 
		 * if (terri.first != null) {
		 * PogamutJBotSupport.writeToConsole("EnemyTerr " + enemyId + " sea " +
		 * territoryId + ": " + terri.first); } if (terri.second != null) {
		 * PogamutJBotSupport.writeToConsole("EnemyTerr " + enemyId + " land " +
		 * territoryId + ": " + terri.second); } } }
		 */

	}


	public void saveBorders(IPrecomputedBordersSaver saver) {
		log.info("Saving borders");

		for (BasicFlag flag : new FlagColl()) {

			log.info("Saving: " + flag + " "
					+ currentTerritoryId);
			String type = null;
			List<List<DefConLocation>> territory = null;

			try {
				switch (flag) {
					case ENEMY_PLACEABLE_LAND:
					case OWN_PLACEABLE_LAND:
						type = "LAND";
						territory = territories.get(currentTerritoryId).second;
					case ENEMY_PLACEABLE_SEA:
					case OWN_PLACEABLE_SEA:
						if (type == null) {
							type = "SEA";
							territory = territories.get(currentTerritoryId).first;
						}
						saver.saveBorderOfTerritory("TERRITORY_" + type
									+ "_"
									+ currentTerritoryId
									+ "_border.def", territory);
						break;
					case SEA:
						territory = sea;
					case LAND:
						if (territory == null)
							territory = land;
						saver.saveBorderOfTerritory(
								flag + "_border.def",
								territory);
						break;
				}
			} catch (IOException e) {
				e.printStackTrace();
			}
		}
	}

	/**
	 * Loads borders from a given path.
	 * 
	 * @param preprocessedPath
	 */
	private void loadBorders(String preprocessedPath) {
		log.info("Loading borders");

		IPrecomputedBordersLoader loader = new FilePrecomputedBordersLoader(preprocessedPath);
		for (BasicFlag flag : new FlagColl()) {
			
			log.info("Loading: " + flag + " "
					+ currentTerritoryId);

			List<List<DefConLocation>> borders;
			
			try {
				String type = null;
				switch (flag) {
					case ENEMY_PLACEABLE_LAND:
					case OWN_PLACEABLE_LAND:
						type = "LAND";
					case ENEMY_PLACEABLE_SEA:
					case OWN_PLACEABLE_SEA:
						if (type == null)
							type = "SEA";

						borders = loader
								.loadBordersForTerritory("TERRITORY_" + type
										+ "_"
										+ currentTerritoryId
										+ "_border.def");
						break;
					default: {
						borders = loader
								.loadBordersForTerritory(flag + "_border.def");
						break;
					}
				}

				switch (flag) {
					case ENEMY_PLACEABLE_SEA:
					case OWN_PLACEABLE_SEA: {
						territories.get(currentTerritoryId).first = borders;
						break;
					}
					case OWN_PLACEABLE_LAND:
					case ENEMY_PLACEABLE_LAND: {
						territories.get(currentTerritoryId).second = borders;
						break;
					}
					case SEA: {
						sea = borders;
						break;
					}
					case LAND: {
						land = borders;
						break;
					}
				}

			} catch (NumberFormatException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (FileNotFoundException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
	}

	public DefConWorldView getWorldView() {
		return (DefConWorldView) worldView;
	}

	/**
	 * Converts coords to an index into linearize 2D array. In this case it is
	 * used for getting index into visited BitSet in @see #processMap().
	 * 
	 * @param coords
	 * @return
	 */
	private final int convertCoordsToVisitedIndex(Coords coords) {
		return coords.getX() * coords.getYSize() + coords.getY();
	}

	private class FlagColl implements Iterable<BasicFlag> {

		@Override
		public Iterator<BasicFlag> iterator() {
			currentFlag = null;
			currentTerritoryId = -1;
			currentEnemyId = -1;
			return new FlagIterator();
		}

		private class FlagIterator implements Iterator<BasicFlag> {

			Iterator<Integer> territoryIterator;
			ArrayList<BasicFlag> acceptedFlags = new ArrayList<BasicFlag>();
			Iterator<BasicFlag> flagIterator;

			public FlagIterator() {
				territoryIterator = null;
				currentFlag = null;
				currentEnemyId = -1;
				currentTerritoryId = -1;

				if (enemyTerritories.keySet().size() == 0) {
					for (BasicFlag flag : BasicFlag.values()) {
						switch (flag) {
							case ENEMY_PLACEABLE_LAND:
							case ENEMY_PLACEABLE_SEA:
							case ENEMY_TERRITORY:
							case OWN_TERRITORY:
								continue;
							default:
								acceptedFlags.add(flag);
						}
					}
				} else {
					for (BasicFlag flag : BasicFlag.values()) {
						switch (flag) {
							case ENEMY_TERRITORY:
							case OWN_TERRITORY:
								continue;
							default:
								acceptedFlags.add(flag);
						}
					}
				}

				flagIterator = acceptedFlags.iterator();
			}

			@Override
			public boolean hasNext() {

				if (!flagIterator.hasNext())
					switch (currentFlag) {
						case ENEMY_PLACEABLE_LAND:
						case ENEMY_PLACEABLE_SEA:
						case OWN_PLACEABLE_LAND:
						case OWN_PLACEABLE_SEA:
							return (territoryIterator != null && territoryIterator
									.hasNext());
						default:
							return false;
					}
				else {
					return true;
				}
			}

			@Override
			public BasicFlag next() {

				if (currentFlag == null) {
					return putFlagIntoEffect(flagIterator.next());
				}

				switch (currentFlag) {
					case ENEMY_PLACEABLE_LAND:
					case ENEMY_PLACEABLE_SEA:
					case OWN_PLACEABLE_LAND:
					case OWN_PLACEABLE_SEA:
						if (territoryIterator != null &&
								territoryIterator.hasNext()) {

							currentTerritoryId = -1;

							territoryIterator: while (territoryIterator
									.hasNext()
									&& (currentTerritoryId < lastAssignedTerritory)) {

								currentTerritoryId = territoryIterator.next();

								currentEnemyId = agent.getWorldView()
										.getGameInfo()
										.getTerritoryOwner(currentTerritoryId);


								switch (currentFlag) {
									case ENEMY_PLACEABLE_LAND:
									case ENEMY_PLACEABLE_SEA:
										if (agent.getWorldView()
												.getGameInfo()
												.getOwnTeamId() != currentEnemyId) {
											break territoryIterator;
										}
										break;
									default:
										break territoryIterator;
								}

							}

							if (currentTerritoryId != -1) {
								return currentFlag;
							}
						}
					default:
						territoryIterator = null;
						if (flagIterator.hasNext()) {
							return putFlagIntoEffect(flagIterator.next());
						} else {
							return null;
						}
				}
			}

			@Override
			public void remove() {
				throw new UnsupportedOperationException(
						"Flag iterator does not support remove operation.");
			}

			private BasicFlag putFlagIntoEffect(BasicFlag flag) {
				switch (flag) {
					case OWN_PLACEABLE_LAND:
					case OWN_PLACEABLE_SEA:
						territoryIterator = ownTerritories
								.keySet().iterator();
						currentFlag = flag;
						return next();
					case ENEMY_PLACEABLE_LAND:
					case ENEMY_PLACEABLE_SEA:
						territoryIterator = agent
								.getWorldView()
								.getGameInfo()
								.getAllEnemyTerritories()
								.iterator();
						currentFlag = flag;
						return next();
					default:
						return currentFlag = flag;
				}
			}

		}
	}

	/**
	 * Processes the map from scratch.
	 */
	protected void processMap() {

		BitSet visited = new BitSet();
		for (BasicFlag flag : new FlagColl()) {

			visited.clear();

			ArrayList<List<DefConLocation>> borders = new ArrayList<List<DefConLocation>>();

			Coords coords = new Coords(0, 0);
			// find polygons

			for (coords.setX(0); coords.getX() < coords.getXSize(); coords
						.setX(coords.getX() + 1)) {

				for (coords.setY(0); coords.getY() < coords.getYSize(); coords
							.setY(coords.getY() + 1)) {

					int visited_index = convertCoordsToVisitedIndex(coords);
					if (!visited.get(visited_index)
									&& isBorder(coords, flag)) {

						LinkedList<DefConLocation> poly =
									getFloodFillBorder(coords, flag, visited);

						if (poly != null && poly.size() > 0) {
							borders.add(Collections
										.unmodifiableList(smoothPoly(poly)));
						}
					}
					visited.set(visited_index);

				}
			}

			log.info("Finished: " + flag);

			switch (flag) {
				case OWN_PLACEABLE_LAND:
				case OWN_PLACEABLE_SEA:
				case ENEMY_PLACEABLE_LAND:
				case ENEMY_PLACEABLE_SEA: {
					
					Pair<List<List<DefConLocation>>, List<List<DefConLocation>>> pair = territories
							.get(currentTerritoryId);

					switch (flag) {
						case OWN_PLACEABLE_SEA:
						case ENEMY_PLACEABLE_SEA:
							pair.first = borders;
							break;
						case OWN_PLACEABLE_LAND:
						case ENEMY_PLACEABLE_LAND:
							pair.second = borders;
							break;
					}
				}
				break;
				case LAND:
					land = borders;
					break;
				case SEA:
					sea = borders;
					break;
			}
		}
		PogamutJBotSupport.writeToConsole("mapInfo done");
		System.gc();
	}

	/**
	 * Smooths the poly using diagonal borders.
	 * 
	 * @param poly
	 * @return
	 */
	private LinkedList<DefConLocation> smoothPoly(
			LinkedList<DefConLocation> poly) {
		if (poly.size() <= 3)
			return poly;
		
		ListIterator<DefConLocation> iter = poly.listIterator(1);
		DefConLocation first = iter.next(), second = null;
		DefConLocation third = poly.peekFirst();
		boolean last = false;
		DefConLocation last_dir = null;
		
		while (iter.hasNext()) {
			second = first;
			first = iter.next();
			
			if (first.equals(third))
				continue;
			
			if (areNeighbours(first, third) &&
					(!last || last_dir.equals(third.sub(first).getNormalized(), 0.1f))) {
				iter.previous();
				iter.previous();
				iter.remove();
				
				if (!last)
					last_dir = new DefConLocation(third.sub(first)
							.getNormalized());
				
				first = iter.next();
				last = true;
			} else {
				third = second;
				last = false;
			}
		}
		/*
		first = poly.peekFirst();
		third = second;
		second = poly.peekLast();
		
		if (areNeighbours(first, third)) {
			poly.removeLast();
		}
		
		first = poly.get(1);
		second = poly.getFirst();
		third = poly.getLast();
		
		if (areNeighbours(first, third)) {
			poly.removeFirst();
		}
		*/		
		return poly;
	}

	/**
	 * Expects Locations normed to fit into (<-180, 180>; <-100 100>).
	 * 
	 * @param first
	 * @param second
	 * @return
	 */
	private boolean areNeighbours(DefConLocation first, DefConLocation second) {
		
		return ((Math.abs(first.getX() - second.getX()) <= 1) ||
				(Math.abs(first.getX() - second.getX())
						- getWorldView().getMinX() <= 1))
				&&
				(Math.abs(first.getY() - second.getY()) <= 1);
	}

	private boolean isPossibleBorderStartPoint(Coords coords, BasicFlag flag) {
		if (coords == null)
			return false;

		if (!flagCheck(coords.getLocation()))
			return false;

		DefConLocation neighbour_location = Direction.LEFT
				.getLocationInThisDirection(coords);

		if (neighbour_location != null
				&& flagCheck(neighbour_location))
			return false;

		neighbour_location = Direction.LOWER.getLocationInThisDirection(coords);
		// LOWER because -100 -> 100 means from bottom up

		if (neighbour_location != null
				&& flagCheck(neighbour_location))
			return false;

		return true;
	}

	private boolean isBorder(Coords coords, BasicFlag flag) {

		if (coords == null)
			return false;

		if (!flagCheck(coords.getLocation()))
			return false;

		DefConLocation neighbour_location;

		// cross

		neighbour_location = Direction.RIGHT.getLocationInThisDirection(coords);

		if (neighbour_location == null
				|| !flagCheck(neighbour_location))
			return true;

		neighbour_location = Direction.LEFT.getLocationInThisDirection(coords);

		if (neighbour_location == null
				|| !flagCheck(neighbour_location))
			return true;

		neighbour_location = Direction.UPPER.getLocationInThisDirection(coords);

		if (neighbour_location == null
				|| !flagCheck(neighbour_location))
			return true;

		neighbour_location = Direction.LOWER.getLocationInThisDirection(coords);

		if (neighbour_location == null
				|| !flagCheck(neighbour_location))
			return true;

		// diagonal

		neighbour_location = Direction.UPPER_RIGHT
				.getLocationInThisDirection(coords);

		if (neighbour_location == null
				|| !flagCheck(neighbour_location))
			return true;

		neighbour_location = Direction.UPPER_LEFT
				.getLocationInThisDirection(coords);

		if (neighbour_location == null
				|| !flagCheck(neighbour_location))
			return true;

		neighbour_location = Direction.LOWER_RIGHT
				.getLocationInThisDirection(coords);

		if (neighbour_location == null
				|| !flagCheck(neighbour_location))
			return true;

		neighbour_location = Direction.LOWER_LEFT
				.getLocationInThisDirection(coords);

		if (neighbour_location == null
				|| !flagCheck(neighbour_location))
			return true;

		return false;
	}

	private boolean isStandAloneCell(Coords coords, BasicFlag flag) {
		if (coords == null)
			return false;

		if (!isPossibleBorderStartPoint(coords, flag))
			return false;

		DefConLocation neighbour_location = coords.getLocation();

		neighbour_location = Direction.RIGHT.getLocationInThisDirection(coords);

		if (neighbour_location != null
				&& flagCheck(neighbour_location))
			return false;

		neighbour_location = Direction.UPPER.getLocationInThisDirection(coords);

		if (neighbour_location != null
				&& flagCheck(neighbour_location))
			return false;

		return true;
	}

	private class Coords {
		private int x;
		private int y;

		public Coords(int x, int y) {
			this.x = x;
			this.y = y;
			filterRestrictedCoords();
		}

		public Coords(Coords source) {
			this.x = source.x;
			this.y = source.y;
			filterRestrictedCoords();
		}

		public int getX() {
			return x;
		}

		public int getY() {
			return y;
		}

		public int setX(int x) {
			this.x = x;
			filterRestrictedCoords();

			return this.x;
		}

		public int setY(int y) {
			this.y = y;
			filterRestrictedCoords();

			return this.y;
		}

		public DefConLocation getLocation() {
			DefConLocation ret = new DefConLocation(x * STEP_SIZE
					+ getWorldView().getMinX(), -(y * STEP_SIZE
					+ getWorldView().getMinY()));

			return ret;
		}

		public final int getXSize() {
			return (int) Math.floor(getWorldView().getXSize() / STEP_SIZE);
		}

		public final int getYSize() {
			return (int) Math.floor(getWorldView().getYSize() / STEP_SIZE);
		}

		private final void filterRestrictedCoords() {
			if (this.y < 0 || this.y > getYSize())
				throw new IllegalArgumentException(
						"Coords cant have y out of bounds");

			if (this.x < 0 || this.x > getXSize())
				throw new IllegalArgumentException(
						"Coords cant have x out of bounds");
		}

		private final void filterCoords() {
			if (this.y < 0 || this.y > getYSize())
				throw new IllegalArgumentException(
						"Coords cant have y out of bounds");

			this.x = (int) (this.x % getXSize());
		}

		public final GameMapInfoPolygons getOwner() {
			return GameMapInfoPolygons.this;
		}

		@Override
		public boolean equals(Object o) {
			return (((Coords) o).x == x) && (((Coords) o).y == y);
		}

		public void setCoords(Coords coords) {
			this.x = coords.x;
			this.y = coords.y;

			filterRestrictedCoords();
		}

		@Override
		public String toString() {
			return "Coords: [ " + x + " ; " + y + " ] ";
		}
	}

	private LinkedList<DefConLocation> getFloodFillBorder(Coords coords,
			BasicFlag flag, BitSet visited) {

		int visited_index = convertCoordsToVisitedIndex(coords);
		if (!isBorder(coords, flag) || visited.get(visited_index)) {
			return null;
		}

		LinkedList<DefConLocation> polygon = new LinkedList<DefConLocation>();

		Coords first_coords = new Coords(coords);
		Coords old_coords = new Coords(coords);
		coords = new Coords(coords);

		Direction direction = Direction.RIGHT;
		Direction old_direction = Direction.LEFT;

		visited.set(visited_index);
		
		if (!isStandAloneCell(first_coords, flag)) {

			do {
				direction = turnAsLeftAsPossible(direction, coords, flag);

				coords = direction.getPointInThisDirection(old_coords);

				visited_index = convertCoordsToVisitedIndex(coords);
				visited.set(visited_index);

				if (direction != old_direction) {
					polygon.add(old_coords.getLocation());
				}
				old_coords.setCoords(coords);
				old_direction = direction;

				// PogamutJBotSupport.writeToConsole("Checking: " +
				// coords.toString());
				
				if (coords.equals(first_coords)) {
					direction = turnAsLeftAsPossible(direction, coords, flag);

					coords = direction.getPointInThisDirection(old_coords);

					visited_index = convertCoordsToVisitedIndex(coords);
					
					if (visited.get(visited_index)) {
						break;
					}
					
					visited.set(visited_index);

					if (direction != old_direction) {
						polygon.add(old_coords.getLocation());
					}
					old_coords.setCoords(coords);
					old_direction = direction;
				}
				
			} while (!coords.equals(first_coords));

		} else {
			polygon.add(first_coords.getLocation());
		}

		return polygon;
	}

	private Direction turnAsLeftAsPossible(Direction direction, Coords coords,
			BasicFlag flag) {
		direction = direction.turnLeft4Way();

		while (!isBorder(direction.getPointInThisDirection(coords), flag)) {
			direction = direction.turnRight4Way();
		}

		return direction;
	}

	private Coords getFilteredCoords(int x, int y) {
		try {
			return new Coords(x, y);
		} catch (IllegalArgumentException e) {
			return null;
		}
	}

	private enum Direction {
		RIGHT(1, 0), UPPER_RIGHT(1, -1), UPPER(0, -1), UPPER_LEFT(-1, -1), LEFT(
				-1, 0), LOWER_LEFT(-1, 1), LOWER(0, 1), LOWER_RIGHT(1, 1);

		public final int x;
		public final int y;

		private Direction(int x, int y) {
			this.x = x;
			this.y = y;
		}

		public Direction reverse() {
			switch (this) {
			case RIGHT:
				return LEFT;
			case UPPER_RIGHT:
				return LOWER_LEFT;
			case UPPER:
				return LOWER;
			case UPPER_LEFT:
				return LOWER_RIGHT;
			case LEFT:
				return RIGHT;
			case LOWER_LEFT:
				return UPPER_RIGHT;
			case LOWER:
				return UPPER;
			case LOWER_RIGHT:
				return UPPER_LEFT;
			}
			return null;
		}

		public Direction turnLeft() {
			switch (this) {
			case RIGHT:
				return UPPER_RIGHT;
			case UPPER_RIGHT:
				return UPPER;
			case UPPER:
				return UPPER_LEFT;
			case UPPER_LEFT:
				return LEFT;
			case LEFT:
				return LOWER_LEFT;
			case LOWER_LEFT:
				return LOWER;
			case LOWER:
				return LOWER_RIGHT;
			case LOWER_RIGHT:
				return RIGHT;
			}
			return null;
		}

		public Direction turnRight() {
			switch (this) {
			case RIGHT:
				return LOWER_RIGHT;
			case LOWER_RIGHT:
				return LOWER;
			case LOWER:
				return LOWER_LEFT;
			case LOWER_LEFT:
				return LEFT;
			case LEFT:
				return UPPER_LEFT;
			case UPPER_LEFT:
				return UPPER;
			case UPPER:
				return UPPER_RIGHT;
			case UPPER_RIGHT:
				return RIGHT;
			}
			return null;
		}

		public Direction turnRight4Way() {
			switch (this) {
			case LOWER_RIGHT:
			case UPPER_RIGHT:
			case LOWER_LEFT:
			case UPPER_LEFT:
				return null;
			}
			return turnRight().turnRight();
		}

		public Direction turnLeft4Way() {
			switch (this) {
			case LOWER_RIGHT:
			case UPPER_RIGHT:
			case LOWER_LEFT:
			case UPPER_LEFT:
				return null;
			}
			return turnLeft().turnLeft();
		}

		public DefConLocation getLocationInThisDirection(Coords coords) {

			try {
				switch (this) {
				case RIGHT:
					return coords.getOwner().getFilteredCoords(
								coords.getX() + 1, coords.getY()).getLocation();
				case UPPER:
					return coords.getOwner().getFilteredCoords(coords.getX(),
										coords.getY() + 1).getLocation();
				case LEFT:
					return coords.getOwner().getFilteredCoords(
								coords.getX() - 1, coords.getY()).getLocation();
				case LOWER:
					return coords.getOwner().getFilteredCoords(coords.getX(),
										coords.getY() - 1).getLocation();
				case UPPER_RIGHT:
					return coords.getOwner().getFilteredCoords(
								coords.getX() + 1, coords.getY() + 1)
								.getLocation();
				case UPPER_LEFT:
					return coords.getOwner().getFilteredCoords(
								coords.getX() - 1, coords.getY() + 1)
								.getLocation();
				case LOWER_LEFT:
					return coords.getOwner().getFilteredCoords(
								coords.getX() - 1, coords.getY() - 1)
								.getLocation();
				case LOWER_RIGHT:
					return coords.getOwner().getFilteredCoords(
								coords.getX() + 1, coords.getY() - 1)
								.getLocation();
				}
			} catch (IllegalArgumentException e) {
				return null;
			} catch (NullPointerException e) {
				return null;
			}
			return null;
		}

		public Coords getPointInThisDirection(Coords coords) {
			switch (this) {
			case RIGHT:
				return coords.getOwner().getFilteredCoords(coords.getX() + 1,
						coords.getY());
			case UPPER:
				return coords.getOwner().getFilteredCoords(coords.getX(),
						coords.getY() + 1);
			case LEFT:
				return coords.getOwner().getFilteredCoords(coords.getX() - 1,
						coords.getY());
			case LOWER:
				return coords.getOwner().getFilteredCoords(coords.getX(),
						coords.getY() - 1);
			case UPPER_RIGHT:
				return coords.getOwner().getFilteredCoords(coords.getX() + 1,
						coords.getY() + 1);
			case UPPER_LEFT:
				return coords.getOwner().getFilteredCoords(coords.getX() - 1,
						coords.getY() + 1);
			case LOWER_LEFT:
				return coords.getOwner().getFilteredCoords(coords.getX() - 1,
						coords.getY() - 1);
			case LOWER_RIGHT:
				return coords.getOwner().getFilteredCoords(coords.getX() + 1,
						coords.getY() - 1);
			}
			return null;
		}

	};

	/**
	 * Removes enough vertices to achieve the given simplification factor (poly.size()^2 / area).
	 * Should we REALLY simplify a poly? Afterall what we area really interested in are the extremes,
	 * not the deep points.
	 * @param poly
	 * @return
	 */
	private LinkedList<DefConLocation> simplifyPoly(
			LinkedList<DefConLocation> poly) {
		
		double area = approximatePolyArea(poly);
		
		if (poly.size() <= 2)
			return poly;

		while (poly.size() * poly.size() / area > SIMPLIFICATION_FACTOR) {
			Edge lightest_edge = getLightestEdge(poly);

			area -= lightest_edge.weighEdge();
			lightest_edge.removeIntermediateVertices();
		}

		return null;
	}

	private final double approximatePolyArea(LinkedList<DefConLocation> poly) {

		if (poly.size() <= 1)
			return 0d;

		double area = 0d;

		Iterator<DefConLocation> iter = poly.iterator();
		DefConLocation look_back = iter.next();

		while (iter.hasNext()) {
			DefConLocation current = new DefConLocation(iter.next());
			// TODO: yeho9o
			current.sub(new DefConLocation(minX, minY));
			area += look_back.getX() * current.getY();
			area -= look_back.getY() * current.getX();
		}

		area += look_back.getX() * poly.getFirst().getY();
		area -= look_back.getY() * poly.getFirst().getX();

		return 0.5f * area;
	}
	
	private boolean flagCheck(DefConLocation location) {

		switch (currentFlag) {
			case ENEMY_TERRITORY:
				return flagChecker.hasEnemyTerritoryFlag(
						location,
						currentEnemyId);
			case ENEMY_PLACEABLE_LAND:
				return flagChecker.hasEnemyTerritoryFlag(
						location,
						currentEnemyId, false);
			case ENEMY_PLACEABLE_SEA:
				return flagChecker.hasEnemyTerritoryFlag(
						location,
						currentEnemyId, true);
			default:
				return flagChecker.hasFlag(location, currentFlag);
		}

	}

	private Edge getLightestEdge(LinkedList<DefConLocation> poly) {
		
		if (poly.size() <= 2)
			return null;
		
		CircularListIterator<DefConLocation> tmp = new CircularListIterator<DefConLocation>(
				poly);
		tmp.next();
		Edge lightest_edge = new Edge(new CircularListIterator<DefConLocation>(
				poly), tmp, poly);
		double lightest_weight = Double.POSITIVE_INFINITY;
		
		CircularListIterator<DefConLocation> first_iter = new CircularListIterator<DefConLocation>(
				poly);
		
		while (!first_iter.hasPassedBeginning()) {
			DefConLocation first = first_iter.next();
			
			CircularListIterator<DefConLocation> second_iter =
					new CircularListIterator<DefConLocation>(first_iter);
			
			while (!second_iter.hasPassedEnd()) {
				DefConLocation second = second_iter.next();
				double weight;
				
				if (second == first)
					continue;
				
				if (lightest_weight > (weight = lightest_edge.weighEdge(
						first_iter.previousIter(),
						second_iter.previousIter(),
						poly))) {
					
					lightest_edge = new Edge(first_iter, second_iter, poly);
					lightest_weight = weight;
				}
				first_iter.next(); second_iter.next();
			}
		}
		
		return lightest_edge;
	}
	
	private class Edge {
		public CircularListIterator<DefConLocation> first, second;
		public LinkedList<DefConLocation> poly;
		
		public Edge(CircularListIterator<DefConLocation> first,
				CircularListIterator<DefConLocation> second,
				LinkedList<DefConLocation> poly) {
			
			if (first == null || second == null || poly == null)
				throw new NullPointerException(
						"None of the parameters of Edge construction should be null.");
				
			this.first = new CircularListIterator<DefConLocation>(first);
			this.second = new CircularListIterator<DefConLocation>(second);
			this.poly = poly;
		}
		
		public void removeIntermediateVertices() {
			while (first.next() != second.next()) {
				first.remove();
				first.previous();
				second.previous();
			}
		}

		public final double weighEdge() {
			return weighEdge(this);
		}
		
		public final double weighEdge(
				CircularListIterator<DefConLocation> first,
				CircularListIterator<DefConLocation> second,
				LinkedList<DefConLocation> poly) {
			LinkedList<DefConLocation> subPoly = new LinkedList<DefConLocation>();
			CircularListIterator<DefConLocation> first_iter = new CircularListIterator<DefConLocation>(
					first);
			CircularListIterator<DefConLocation> second_iter = new CircularListIterator<DefConLocation>(
					second);
			
			DefConLocation current = first_iter.next();
			DefConLocation last = second_iter.next();
			
			while (current != last) {
				subPoly.add(current);
				current = first_iter.next();
			}
			return approximatePolyArea(subPoly);
		}
		
		public final double weighEdge(Edge edge) {
			LinkedList<DefConLocation> subPoly = new LinkedList<DefConLocation>();
			CircularListIterator<DefConLocation> first_iter = edge.getFirst();
			CircularListIterator<DefConLocation> second_iter = edge.getSecond();
			
			DefConLocation current = first_iter.next();
			DefConLocation last = second_iter.next();
			
			while (current != last) {
				subPoly.add(current);
				current = first_iter.next();
			}
			return approximatePolyArea(subPoly);
		}
		
		public CircularListIterator<DefConLocation> getFirst() {
			return new CircularListIterator<DefConLocation>(first);
		}
		
		public CircularListIterator<DefConLocation> getSecond() {
			return new CircularListIterator<DefConLocation>(second);
		}		
	}

	private static final float mathModulus(float a, float b) {
		return (a % b + b) % b;
	}

	private static final List<DefConLocation> checkAndJoinBordersOnSeam(
			List<DefConLocation> borderA, List<DefConLocation> borderB) {

		// TODO: :(((

		return null;
	}

	public SortedMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>> getOwnTerritories() {
		return Collections.unmodifiableSortedMap(ownTerritories);
	}

	public SortedMap<Integer, SortedMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>>>
			getEnemiesTerritories() {
		return Collections.unmodifiableSortedMap(enemyTerritories);
	}

	/**
	 * Returns a single territory.
	 * 
	 * @param territoryId
	 * @return
	 */
	public Pair<List<List<DefConLocation>>, List<List<DefConLocation>>> getTerritory(
			int territoryId) {
		return territories.get(territoryId);
	}

	/**
	 * Returns a list of territories the enemy has.
	 * 
	 * @param enemyId
	 * @return
	 */
	public SortedMap<Integer, Pair<List<List<DefConLocation>>, List<List<DefConLocation>>>> getEnemyTerritories(
			int enemyId) {
		if (!enemyTerritories.containsKey(enemyId)) {
			throw new IllegalArgumentException(
					"getEnemyTerritories does not contain " + enemyId +
							" only contains "
							+ enemyTerritories.keySet().toString());
		}
		return Collections.unmodifiableSortedMap(enemyTerritories
				.get(enemyId));
	}

	/**
	 * Returns contours of negation of sea.
	 * 
	 * @return
	 */
	public List<List<DefConLocation>> getLand() {
		return Collections.unmodifiableList(land);
	}

	/**
	 * Returns contours of sea.
	 * 
	 * @return
	 */
	public List<List<DefConLocation>> getSea() {
		return Collections.unmodifiableList(sea);
	}
}
