package cz.cuni.amis.pogamut.defcon.communication.worldview.modules.managers.fleets;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.SortedMap;
import java.util.TreeMap;

import com.google.inject.Inject;

import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectDestroyedEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.defcon.agent.impl.DefConAgentLogicController;
import cz.cuni.amis.pogamut.defcon.agent.impl.ILogicUpdateListener;
import cz.cuni.amis.pogamut.defcon.ai.fleetai.IFleetAI;
import cz.cuni.amis.pogamut.defcon.base3d.worldview.object.DefConLocation;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.PlaceFleet;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.Fleet;
import cz.cuni.amis.pogamut.defcon.consts.UnitType;

/**
 * Manages all known fleets. Be they either own or enemies. Also takes care of
 * updating of all you fleets AIs.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class FleetsManager {

	protected final DefConAgentLogicController<?> logic;

	protected final SortedMap<Integer, List<Fleet>> enemyFleets = new TreeMap<Integer, List<Fleet>>();
	protected final ArrayList<FleetWithAI> ownFleets = new ArrayList<FleetWithAI>();
	protected final int ownTeamId;

	protected final LinkedList<QueuedPlacing> queuedPlacings = new LinkedList<QueuedPlacing>();
	protected final LinkedList<QueuedPlacing> oldPlacings = new LinkedList<QueuedPlacing>();

	/**
	 * Fleet and AI pair.
	 * 
	 * @author Radek 'Black_Hand' Pibil
	 * 
	 */
	public class FleetWithAI {
		Fleet fleet;
		IFleetAI ai;

		public FleetWithAI(Fleet fleet, IFleetAI ai) {
			this.fleet = fleet;
			this.ai = ai;
		}

		public final IFleetAI getAI() {
			return ai;
		}

		public final void setAI(IFleetAI ai) {
			this.ai = ai;
		}

		public final Fleet getFleet() {
			return fleet;
		}
	}

	protected class QueuedPlacing {
		protected final Queue<DefConLocation> placementOptions;
		protected final IPlacingFinishedListener callback;
		protected final int expectedCount;
		protected final Object data;
		protected final ArrayList<Fleet> succeeded = new ArrayList<Fleet>();
		protected final List<DefConLocation> last = new ArrayList<DefConLocation>();
		protected final UnitType[] fleetComposition;
		protected double lastPlacementTime;
		protected static final double LAST_PLACEMENT_MS_LIMIT = 2500d;

		public QueuedPlacing(int expectedCount,
				UnitType[] fleetComposition,
				Queue<DefConLocation> placementOptions,
				Object data,
				IPlacingFinishedListener callback) {

			if (expectedCount <= 0 || fleetComposition == null
					|| placementOptions == null
					|| placementOptions.size() < expectedCount) {
				throw new IllegalArgumentException(
						"Invalid parameters for QueuedPlacing.");
			}

			for (int teamId : logic.getGameInfo().getEnemyTeamIds()) {
				enemyFleets
						.put(teamId, new ArrayList<Fleet>());
			}

			this.expectedCount = expectedCount;
			this.placementOptions = placementOptions;
			this.callback = callback;
			this.data = data;
			this.fleetComposition = fleetComposition;
		}

		public final Queue<DefConLocation> getPlacementOptions() {
			return placementOptions;
		}

		public final int getExpectedCount() {
			return expectedCount;
		}

		public final List<DefConLocation> getLast() {
			return last;
		}

		public final void doPlace() {
			if (succeeded.size() + placementOptions.size() < expectedCount ||
					!logic.getGameInfo().canCreateFleet(fleetComposition)) {
				queuedPlacings.remove(this);
				oldPlacings.add(this);
				if (callback != null) {
					callback.placingFinished(succeeded, data, false);
				}
				return;
			}

			int lastCount = last.size();
			DefConLocation location = null;
			while (!placementOptions.isEmpty() && lastCount == last.size()) {
				location = placementOptions.poll();

				if (queuePlacingsContain(location))
					continue;

				if (!isValidPlacement(location, fleetComposition.length))
					continue;

				last.add(location);
				break;
			}

			if (lastCount == last.size()) {
				return;
			}

			lastPlacementTime = System.currentTimeMillis();
			if (!placeFleetWorker(location, fleetComposition)) {
				oldPlacings.add(this);
				queuedPlacings.remove(this);
				if (callback != null) {
					callback.placingFinished(succeeded, data, false);
				}
			}
		}

		public final void successfulPlacement(Fleet fleet) {
			succeeded.add(fleet);

			if (succeeded.size() == expectedCount) {
				queuedPlacings.remove(this);
				oldPlacings.remove(this);
				if (callback != null) {
					callback.placingFinished(succeeded, data, true);
				}
			} else {
				doPlace();
			}
		}

		public void refresh() {
			if (System.currentTimeMillis() - lastPlacementTime > LAST_PLACEMENT_MS_LIMIT) {
				doPlace();
			}
		}
	}

	protected boolean queuePlacingsContain(Location location) {
		for (QueuedPlacing placing : queuedPlacings) {
			if (placing.getLast() != null && placing.getLast().equals(location)) {
				return true;
			}
		}
		return false;
	}

	@Inject
	public FleetsManager(DefConAgentLogicController<?> logic) {
		ownTeamId = logic.getWorldView().getGameInfo()
				.getOwnTeamId();
		this.logic = logic;
		logic.getWorldView().addObjectListener(Fleet.class, listener);
		logic.addGameLogicListener(logicUpdateListener);
	}

	protected final IWorldObjectListener<Fleet> listener =
			new IWorldObjectListener<Fleet>() {

				@Override
				public void notify(IWorldObjectEvent<Fleet> event) {

					if (event instanceof WorldObjectFirstEncounteredEvent<?>) {
						addFleet(event.getObject());
					}
					if (event instanceof WorldObjectDestroyedEvent<?>) {
						removeFleet(event.getObject());
					}
				}
			};

	protected final ILogicUpdateListener logicUpdateListener = new ILogicUpdateListener() {

		@Override
		public void update() {
			if (!queuedPlacings.isEmpty())
				queuedPlacings.peek().refresh();

			for (FleetWithAI fleet : ownFleets) {
				IFleetAI ai = fleet.getAI();

				if (ai != null)
					ai.update();
			}
		}

	};

	protected void addFleet(Fleet fleet) {
		if (fleet.getTeamId() == ownTeamId) {
			addOwnFleet(fleet);
		} else {
			addEnemyFleet(fleet);
		}
	}

	protected void removeFleet(Fleet fleet) {
		if (fleet.getTeamId() == ownTeamId) {
			removeOwnFleet(fleet);
		} else {
			removeEnemyFleet(fleet);
		}
	}

	protected void addEnemyFleet(Fleet fleet) {

		int enemyId = fleet.getTeamId();
		List<Fleet> singleEnemyFleets = enemyFleets
				.get(enemyId);

		if (!singleEnemyFleets.contains(fleet))
			singleEnemyFleets.add(fleet);

	}

	protected void removeEnemyFleet(Fleet fleet) {

		int enemyId = fleet.getTeamId();
		List<Fleet> singleEnemyFleets = enemyFleets
				.get(enemyId);

		if (singleEnemyFleets.contains(fleet)) {
			logic.getLog().info("Removing enemy fleet: " + fleet.getId());
			singleEnemyFleets.remove(fleet);
		}
	}

	protected void addOwnFleet(Fleet fleet) {
		if (!ownFleets.contains(fleet)) {
			ownFleets.add(new FleetWithAI(fleet, null));

			logic.getLog().info(
					"Placed fleet to: " + fleet.getLocation() + " "
							+ fleet.getId());
		} else {
			logic.getLog().info(
					"WUT?! Placed fleet to: " + fleet.getLocation() + " "
							+ fleet.getId());
		}
		

		for (QueuedPlacing placings : queuedPlacings) {

			for (Location loc : placings.getLast()) {
				if (loc.equals(fleet.getLocation())) {
					placings.successfulPlacement(fleet);
					return;
				}
			}
		}

		for (QueuedPlacing placings : oldPlacings) {

			for (Location loc : placings.getLast()) {
				if (loc.equals(fleet.getLocation())) {
					placings.successfulPlacement(fleet);
					return;
				}
			}
		}

	}

	protected void removeOwnFleet(Fleet fleet) {
		
		Iterator<FleetWithAI> ownFleetsIterator = ownFleets.iterator();

		while (ownFleetsIterator.hasNext()) {
			FleetWithAI ownFleet = ownFleetsIterator.next();
			
			if (ownFleet.fleet.getId().getLongId() != fleet.getId().getLongId())
				continue;

			logic.getLog().info("Removing own fleet: " + fleet.getId());

			ownFleetsIterator.remove();

			if (ownFleet.getAI() != null) {
				ownFleet.getAI().dispose();
				ownFleet.setAI(null);
				return;
			}

		}
	}

	/**
	 * Returns the list of enemy fleets.
	 * 
	 * @return
	 */
	public SortedMap<Integer, List<Fleet>> getEnemyFleets() {
		return Collections.unmodifiableSortedMap(enemyFleets);
	}

	/**
	 * Returns the list of enemy fleets.
	 * 
	 * @return
	 */
	public List<Fleet> getEnemyFleets(int enemyId) {
		// logic.getLog().info(
		// "getting enemy fleets: " + enemyId + " "
		// + enemyFleets.toString());
		return Collections.unmodifiableList(enemyFleets.get(enemyId));
	}

	/**
	 * Returns the list of own fleets.
	 * 
	 * @return
	 */
	public List<FleetWithAI> getOwnFleets() {
		return Collections.unmodifiableList(ownFleets);
	}

	/**
	 * Assigns AI to the given fleet.
	 * 
	 * @param fleet
	 * @param ai
	 * @return
	 */
	public boolean assignAI(Fleet fleet, IFleetAI ai) {
		for (FleetWithAI fleetWithAI : ownFleets) {
			if (fleetWithAI.getFleet().equals(fleet)) {
				fleetWithAI.setAI(ai);
				return true;
			}
		}

		return false;
	}

	/**
	 * This wont work most likely, because of the quirky WorldObjectId
	 * 
	 * @param fleetId
	 * @param ai
	 * @return
	 */
	public boolean assignAI(int fleetId, IFleetAI ai) {
		for (FleetWithAI fleetWithAI : ownFleets) {
			if (fleetWithAI.getFleet().getId().getLongId() == fleetId) {
				fleetWithAI.setAI(ai);
				return true;
			}
		}

		return false;
	}


	/**
	 * Tries to place fleet on a given location (queues the command).
	 * 
	 * @param location
	 * @param ships
	 * @return tentative success in placing of the fleet. if true then check the
	 *         event for definitive confirmation.
	 */
	public boolean placeFleet(DefConLocation location, UnitType[] ships,
			Object initData,
			IPlacingFinishedListener finishedListener) {

		LinkedList<DefConLocation> tmp = new LinkedList<DefConLocation>();
		tmp.add(location);
		
		return placeFleet(tmp, ships, 1, initData, finishedListener);
	}

	protected boolean isValidPlacement(Location location, int count) {
		if (logic.getGameInfo()
				.isValidFleetPlacement(location, count)) {
			return true;
		} else {
			return false;
		}
	}

	protected boolean placeFleetWorker(DefConLocation location, UnitType[] ships) {


		logic.getGameInfo().canCreateFleet(ships);

		logic.getLog().info(
				String.format("Placing fleet to: %s.", location.toString()));


		logic.getAct().act(new PlaceFleet(location, ships));

		return true;
	}

	/**
	 * Tries to place the fleet on one of the given positions (queues the
	 * command, if it finds suitable place), starting from the head.
	 * 
	 * @param orderedPlacements
	 *            given positions with preference
	 * @param fleetComposition
	 *            ship types
	 * @return true if at least one fleet might be placeable (listener is going
	 *         to be called)
	 */
	public boolean placeFleet(
			List<DefConLocation> orderedPlacements,
			UnitType[] fleetComposition, int fleetsCount,
			Object initData,
			IPlacingFinishedListener finishedListener) {

		if (fleetsCount <= 0 || orderedPlacements.size() < fleetsCount)
			return false;

		LinkedList<DefConLocation> placements = new LinkedList<DefConLocation>();

		for (DefConLocation placement : orderedPlacements) {
			placements.addLast(placement);
		}

		Iterator<DefConLocation> placementsIterator = placements.listIterator();

		while (placementsIterator.hasNext()) {

			Location location = placementsIterator.next();
			if (!isValidPlacement(location, fleetComposition.length)) {
				placementsIterator.remove();
				if (placements.size() < fleetsCount) {
					break;
				}
			}
		}

		if (placements.size() < fleetsCount) {
			return false;
		}

		QueuedPlacing placing = new QueuedPlacing(fleetsCount,
				fleetComposition, placements, initData, finishedListener);

		queuedPlacings.add(placing);
		
		if (queuedPlacings.size() == 1) {
			placing.doPlace();
		}

		return true;
	}
}
