package cz.cuni.amis.pogamut.multi.communication.worldview.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldObjectUpdateResult;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.pogamut.base.component.controller.ComponentDependencies;
import cz.cuni.amis.pogamut.base.component.lifecyclebus.ILifecycleBus;
import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base3d.worldview.object.IViewable;
import cz.cuni.amis.pogamut.multi.agent.ITeamedAgentId;
import cz.cuni.amis.pogamut.multi.communication.translator.event.ICompositeWorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.multi.communication.translator.event.ILocalWorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.multi.communication.worldview.ISharedWorldView;
import cz.cuni.amis.pogamut.multi.communication.worldview.IVisionLocalWorldView;
import cz.cuni.amis.pogamut.multi.communication.worldview.object.ILocalViewable;
import cz.cuni.amis.pogamut.multi.communication.worldview.object.ILocalWorldObject;
import cz.cuni.amis.utils.ClassUtils;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * VisionLocalWorldView manages information about all objects currently in the bot's FOV (field-of-view)
 * by implementing methods from {@link IVisionLocalWorldView} interface.
 * 
 * @author srlok
 *
 */
@AgentScoped
public abstract class VisionLocalWorldView extends EventDrivenLocalWorldView implements IVisionLocalWorldView {

	public VisionLocalWorldView(
			ComponentDependencies dependencies,
			ILifecycleBus bus, IAgentLogger logger,
			ISharedWorldView parentWorldView, ITeamedAgentId agentId
	) {
		super(dependencies, bus, logger, parentWorldView, agentId);
	}

	@Override
	public void notify(IWorldChangeEvent event)
	{
		if ( event instanceof ILocalWorldObjectUpdatedEvent)
		{
			objectUpdatedEvent( (ILocalWorldObjectUpdatedEvent)event );
		}
		else if ( event instanceof ICompositeWorldObjectUpdatedEvent)
		{
			objectUpdatedEvent( ((ICompositeWorldObjectUpdatedEvent)event).getLocalEvent() );
		}
		else
		{
			super.notify(event);
		}
	}
	
	/**
	 * Map of all currently visible objects.
	 */
	protected LazyCompositeObjectMap<IViewable> visibleMap =
		new LazyCompositeObjectMap<IViewable>(currentTimeKey);
	
	/**
	 * Synchronized version of visible objects.
	 */
	protected Map<WorldObjectId, IViewable> syncVisibleMap = 
		Collections.synchronizedMap(visibleMap);
	
	/**
	 * Map of all currently visible objects, sorted according to their classes.
	 */
	@SuppressWarnings("rawtypes")
	protected Map< Class, Map<WorldObjectId, IViewable>> visibleClassMap =
		new HashMap<Class, Map<WorldObjectId, IViewable>>();

	/**
	 * Synchronized version of visible objects sorted according to class.
	 */
	@SuppressWarnings("rawtypes")
	protected Map< Class, Map<WorldObjectId,IViewable>> syncVisibleClassMap =
		Collections.synchronizedMap(visibleClassMap);
	
	@Override
	protected void objectUpdatedEvent(ILocalWorldObjectUpdatedEvent updateEvent) {
        ILocalWorldObject obj = getMostRecentLocalWorldObject( updateEvent.getId() );
        
        boolean oldVisible = false;
        boolean isViewable = false;
        
        ILocalWorldObject copy = null;
        //if old timeKeys are held, store the original value, create a new copy, which will be updated
        if ( obj != null)
        {
        	if ( obj instanceof ILocalViewable) 
        	{
        		oldVisible = ((ILocalViewable)obj).isVisible();
        		isViewable = true;
        	}
        	copy = obj.clone();
        }
        else //may be created event
        {
        	copy = null;
        }
       
        IWorldObjectUpdateResult<ILocalWorldObject> updateResult = updateEvent.update(copy);
        switch (updateResult.getResult()) {
        case CREATED:            	
            objectCreated(updateResult.getObject());
            return;
        case UPDATED:
        	if (updateResult.getObject() != copy) {
        		throw new PogamutException("Update event " + updateEvent + " does not returned the same instance of the object (result UPDATED).", this);
        	}
        	
        	super.addOldLocalWorldObject(obj, updateEvent.getSimTime());        	
        	
        	if (isViewable)
        	{
        		boolean visible = ((ILocalViewable)copy).isVisible();
        		if ( visible != oldVisible)
        		{
        			if ( visible )
        			{
        				objectAppeared((ILocalViewable)copy);
        			}
        			else
        			{
        				objectDisappeared((ILocalViewable)copy);
        			}
        		}
        	}
        	
        	actLocalWorldObjects.put(copy.getId(), copy);    	
        	objectUpdated(copy);
        	
        	return;
        case SAME:
        	return;
        case DESTROYED:
        	
        	super.addOldLocalWorldObject(obj, updateEvent.getSimTime());
        	objectDestroyed(copy);
        	
            return;
        default:
        	throw new PogamutException("Unhandled object update result " + updateResult.getResult() + " for the object " + obj + ".", this);
        }
    }
	
	@Override
	public void objectCreated( ILocalWorldObject obj )
	{
		if ( obj instanceof ILocalViewable)
		{
			if ( ((ILocalViewable)obj).isVisible() )
			{
				objectAppeared( (ILocalViewable)obj );
			}
		}
		
		super.objectCreated(obj);
	}
	
	@Override
	public void objectDestroyed( ILocalWorldObject obj )
	{
		if ( obj instanceof ILocalViewable)
		{
			removeVisible((ILocalViewable)obj);
		}
		super.objectDestroyed(obj);
	}
	
	/**
	 * Handles events for making the object visible.
	 * @param obj
	 */
	protected void objectAppeared( ILocalViewable obj )
	{
		addVisible(obj);
		//TODO eventRaise
	}
	
	/**
	 * Handles events for making the object not visible.
	 * @param obj
	 */
	protected void objectDisappeared( ILocalViewable obj )
	{
		removeVisible(obj);
		//TODO eventRaise
	}
	
	/**
	 * Adds the provided object as visible into all visibleMaps int the worldView.
	 * Note that since the cached visible objects are composite and the parameter for this method is a local object,
	 * only the id and the getCompositeClass of the object are actually used.
	 * @param obj
	 */
	protected synchronized void addVisible( ILocalViewable obj )
	{
		synchronized(visibleMap)
		{
			visibleMap.addKey(obj.getId());
		}
		synchronized( syncVisibleClassMap )
		{
			for ( Class cls : ClassUtils.getSubclasses(obj.getCompositeClass()) )
			{
				LazyCompositeObjectMap<IViewable> map = (LazyCompositeObjectMap<IViewable>)visibleClassMap.get(cls);
				if ( map == null )
				{
					map = new LazyCompositeObjectMap<IViewable>( getCurrentTimeKey() );
					visibleClassMap.put(cls, map);
				}
				map.addKey(obj.getId());
			}
		}
	}
	
	/**
	 * Removes object of the same objectId as the provided localObject from visible maps.
	 * Note that the provided ILocalViewable object has to implement the getCompositeClass() method to return the correct composite object class.
	 * @param obj
	 */
	protected synchronized void removeVisible( ILocalViewable obj )
	{
		syncVisibleMap.remove(obj.getId());
		synchronized( visibleClassMap )
		{
			for ( Class cls : ClassUtils.getSubclasses(obj.getCompositeClass()) )
			{
				LazyCompositeObjectMap<IViewable> map = (LazyCompositeObjectMap<IViewable>)visibleClassMap.get(cls);
				if ( map != null )
				{
					map.remove(obj.getId());
				}
			}
		}
	}
		
	@Override
	public Map<Class, Map<WorldObjectId, IViewable>> getAllVisible() {
		synchronized(syncVisibleClassMap)
		{
			for ( Class cls : syncVisibleClassMap.keySet() )
			{
				LazyCompositeObjectMap<IViewable> map = (LazyCompositeObjectMap<IViewable>)syncVisibleClassMap.get(cls);
				map.setTimeKey( getCurrentTimeKey() );
			}
		}
		return visibleClassMap;
	}

	@Override
	public <T extends IViewable> Map<WorldObjectId, T> getAllVisible(
			Class<T> type) {
		LazyCompositeObjectMap<IViewable> map = (LazyCompositeObjectMap<IViewable>)syncVisibleClassMap.get(type);
		map.setTimeKey( getCurrentTimeKey() );
		return (Map<WorldObjectId, T>) map;
	}

	@Override
	public Map<WorldObjectId, IViewable> getVisible() {
		LazyCompositeObjectMap<IViewable> map = (LazyCompositeObjectMap<IViewable>) syncVisibleMap;
		map.setTimeKey( getCurrentTimeKey() );
		return syncVisibleMap;
	}

	@Override
	public IViewable getVisible(WorldObjectId id) {
		LazyCompositeObjectMap<IViewable> map = (LazyCompositeObjectMap<IViewable>) syncVisibleMap;
		map.setTimeKey( getCurrentTimeKey() );
		return visibleMap.get(id);
	}
	
}
