package cz.cuni.amis.pogamut.episodic.episodes;

import cz.cuni.amis.pogamut.episodic.decisions.DecisionTree;
import cz.cuni.amis.pogamut.episodic.decisions.DecisionTreeTest;
import cz.cuni.amis.pogamut.episodic.memory.AffordanceUsed;
import cz.cuni.amis.pogamut.episodic.memory.AgentMemory;
import cz.cuni.amis.pogamut.episodic.memory.Parameters;
import java.util.ArrayList;
import java.util.Iterator;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import static org.junit.Assert.*;

/**
 *
 * @author Michal Cermak
 */
public class ChronobagTest {

    AgentMemory mem = null;
    Chronobag instance = null;
    
    public ChronobagTest() {
    }

    @Before
    public void setUp() {
        DecisionTree dTree = DecisionTreeTest.sampleTree(10, 2, 3);
        mem = new AgentMemory();
        mem.initialize(dTree.topLevelGoals.values());
    }

    @After
    public void tearDown() {
    }

    @Test
    public void testChronobag01() {
        instance = new Chronobag(mem.getIdGenerator(), mem);
        assertSame(instance.getMemory(), mem);
        assertEquals(instance.level, 0);
        assertFalse(instance.landmark);
        assertEquals(instance.numberOfEpisodeNodes, 0);
        assertTrue(instance.getEpisodes().isEmpty());
        assertFalse(instance.newNodeAdded);
        assertTrue(instance.objectNodes.isEmpty());
        assertEquals(instance.getAge().getMinAge(), 0);
        assertEquals(instance.getAge().getMaxAge(), 0);

        ArrayList<String> trace = new ArrayList<String>();
        trace.add("intention.2.0");
        trace.add("action.1.0");
        trace.add("intention.1.0");
        trace.add("action.0.0");
        ArrayList<AffordanceUsed> affs = new ArrayList<AffordanceUsed>();

        instance.addNewNode("atomic.0.0", trace, affs, "");

        assertEquals(instance.currentEpisodes.size(), 1);
        assertNotNull(instance.currentEpisodes.get("intention.2.0"));
        assertEquals(instance.numberOfEpisodeNodes, 5);

        affs.add(new AffordanceUsed("action.1.0", "type", "obj1"));
        instance.addNewNode("atomic.0.0", trace, affs, "");

        assertEquals(instance.objectNodes.size(), 1);
        assertNotNull(instance.objectNodes.get("obj1"));

        assertEquals(instance.getFirstEpisode().getRoot().getName(), "intention.2.0");
        assertTrue(instance.getEpisodes().contains(instance.getFirstEpisode()));
        assertTrue(instance.finishedEpisodes.isEmpty());

        trace.clear();
        trace.add("intention.2.0");
        instance.finishNode("intention.2.0", trace, true);

        assertTrue(instance.currentEpisodes.isEmpty());
        assertEquals(instance.getEpisodes().size(), 1);
        assertTrue(instance.getEpisodes().contains(instance.getFirstEpisode()));
        assertEquals(instance.finishedEpisodes.size(), 1);
        assertEquals(instance.finishedEpisodes.iterator().next().getRoot().getName(), "intention.2.0");

        trace.clear();
        trace.add("intention.2.1");
        affs.clear();
        affs.add(new AffordanceUsed("intention.2.1", "type", "obj2"));
        instance.addNewNode("atomic.2.0", trace, affs, "");

        trace.clear();
        trace.add("intention.2.2");
        affs.clear();
        affs.add(new AffordanceUsed("intention.2.2", "type", "obj2"));
        instance.addNewNode("atomic.2.0", trace, affs, "");

        trace.clear();
        trace.add("intention.2.3");
        affs.clear();
        affs.add(new AffordanceUsed("intention.2.3", "type", "obj2"));
        instance.addNewNode("atomic.2.0", trace, affs, "");

        assertEquals(instance.currentEpisodes.size(), 3);
        assertEquals(instance.getEpisodes().size(), 4);
        assertNotNull(instance.currentEpisodes.get("intention.2.1"));
        assertNotNull(instance.currentEpisodes.get("intention.2.2"));
        assertNotNull(instance.currentEpisodes.get("intention.2.3"));

        Episode e = instance.getFirstEpisode();
        assertEquals(e.getRoot().getName(), "intention.2.0");
        assertEquals(e.next.size(), 1);
        e = e.next.iterator().next();
        assertEquals(e.getRoot().getName(), "intention.2.1");
        assertEquals(e.next.size(), 1);
        e = e.next.iterator().next();
        assertEquals(e.getRoot().getName(), "intention.2.2");
        assertEquals(e.next.size(), 1);
        e = e.next.iterator().next();
        assertEquals(e.getRoot().getName(), "intention.2.3");
        assertTrue(e.next.isEmpty());

        assertEquals(e.previous.size(), 1);
        e = e.previous.iterator().next();
        assertEquals(e.getRoot().getName(), "intention.2.2");
        assertEquals(e.previous.size(), 1);
        e = e.previous.iterator().next();
        assertEquals(e.getRoot().getName(), "intention.2.1");
        assertEquals(e.previous.size(), 1);
        e = e.previous.iterator().next();
        assertEquals(e.getRoot().getName(), "intention.2.0");
        assertTrue(e.previous.isEmpty());

        instance.finish();

        assertTrue(instance.currentEpisodes.isEmpty());
        assertEquals(instance.finishedEpisodes.size(), 4);
        
        System.out.println("---/// TEST CHRONOBAG 01 OK ///---");
    }

    @Test
    public void testChronobag02() {
        instance = new Chronobag(mem.getIdGenerator(), mem);

        assertFalse(instance.isLandmark());
        for (int i = 0; i < Parameters.LANDMARK_AGE_THRESHOLDS[0] - 1; i++) {
            instance.increaseDay();
        }
        assertFalse(instance.isLandmark());
        instance.increaseDay();
        assertTrue(instance.isLandmark());

        instance = new Chronobag(mem.getIdGenerator(), mem);
        instance.increaseDay();
        assertEquals(instance.getAge().getMinAge(), 1);
        assertEquals(instance.getAge().getMaxAge(), 1);

        assertNull(instance.getMoreAbstractChronobag());
        Chronobag abs = instance.createMoreAbstractChronobag();
        assertNotNull(abs);
        assertEquals(abs.getLevel(), 1);
        assertSame(abs, instance.getMoreAbstractChronobag());
        assertEquals(abs.getAge().getMinAge(), instance.getAge().getMinAge() - (Parameters.CHRONOBAG_INTERVAL_LENGHTS[1] - Parameters.CHRONOBAG_INTERVAL_LENGHTS[0]) / 2);
        assertEquals(abs.getAge().getMaxAge(), instance.getAge().getMaxAge() + (Parameters.CHRONOBAG_INTERVAL_LENGHTS[1] - Parameters.CHRONOBAG_INTERVAL_LENGHTS[0]) / 2);
        assertEquals(abs.getAge().getMaxAge() - abs.getAge().getMinAge() + 1, Parameters.CHRONOBAG_INTERVAL_LENGHTS[abs.level]);

        instance = new Chronobag(mem.getIdGenerator(), mem, instance);
        assertSame(instance.getMoreAbstractChronobag(), abs);

        instance.createMoreAbstractChronobag();
        assertSame(instance.getMoreAbstractChronobag(), abs);

        assertFalse(abs.isLandmark());
        for (int i = 0; i < Parameters.LANDMARK_AGE_THRESHOLDS[abs.getLevel()] - 1; i++) {
            abs.increaseDay();
        }
        assertFalse(abs.isLandmark());
        abs.increaseDay();
        assertTrue(abs.isLandmark());
        
        System.out.println("---/// TEST CHRONOBAG 02 OK ///---");
    }

    @Test
    public void testChronobag03() {
        instance = new Chronobag(mem.getIdGenerator(), mem);
        ArrayList<String> trace = new ArrayList<String>();
        ArrayList<AffordanceUsed> affs = new ArrayList<AffordanceUsed>();

        //episode unite
        boolean result;
        trace.add("intention.2.0");
        instance.addNewNode("atomic", trace, affs, "");
        result = instance.finishNode("intention.2.0", trace, true);
        assertTrue(result);
        trace.add("intention.2.0");
        instance.addNewNode("atomic", trace, affs, "");
        result = instance.finishNode("intention.2.0", trace, true);
        assertTrue(result);
        trace.add("intention.2.0");
        instance.addNewNode("atomic", trace, affs, "");
        result = instance.finishNode("intention.2.0", trace, true);
        assertTrue(result);

        assertEquals(instance.getEpisodes().size(), 1);

        trace.add("intention.2.0");
        instance.addNewNode("atomic", trace, affs, "");
        instance.finishNode("intention.2.0", trace, true);
        trace.add("intention.2.1");
        instance.addNewNode("atomic", trace, affs, "");
        instance.finishNode("intention.2.1", trace, true);
        trace.add("intention.2.0");
        instance.addNewNode("atomic", trace, affs, "");
        instance.finishNode("intention.2.0", trace, true);

        assertEquals(instance.getEpisodes().size(), 3);

        //suspended episode
        Episode e;
        trace.add("intention.2.2");
        instance.addNewNode("atomic", trace, affs, "");
        trace.clear();
        trace.add("intention.2.3");
        instance.addNewNode("atomic", trace, affs, "");
        e = instance.currentEpisodes.get("intention.2.2");
        assertEquals(e.next.size(), 1);
        assertTrue(e.next.contains(instance.currentEpisodes.get("intention.2.3")));
        trace.clear();
        trace.add("intention.2.2");
        instance.addNewNode("atomic", trace, affs, "");
        trace.clear();
        trace.add("intention.2.4");
        instance.addNewNode("atomic", trace, affs, "");
        assertEquals(e.next.size(), 2);
        assertTrue(e.next.contains(instance.currentEpisodes.get("intention.2.3")));
        assertTrue(e.next.contains(instance.currentEpisodes.get("intention.2.4")));

        assertTrue(instance.currentEpisodes.get("intention.2.3").previous.contains(e));
        assertTrue(instance.currentEpisodes.get("intention.2.4").previous.contains(e));

        trace.clear();
        trace.add("intention.2.2");
        instance.addNewNode("atomic", trace, affs, "");
        trace.clear();
        trace.add("intention.2.3");
        instance.addNewNode("atomic", trace, affs, "");

        assertEquals(e.next.size(), 3);
        assertSame(e.next.get(0), instance.currentEpisodes.get("intention.2.3"));
        assertSame(e.next.get(1), instance.currentEpisodes.get("intention.2.4"));
        assertSame(e.next.get(2), instance.currentEpisodes.get("intention.2.3"));

        System.out.println("---/// TEST CHRONOBAG 03 OK ///---");
    }

    @Test
    public void testChronobag04() {
        //episode merging
        instance = new Chronobag(mem.getIdGenerator(), mem);
        ArrayList<String> trace = new ArrayList<String>();
        ArrayList<AffordanceUsed> affs = new ArrayList<AffordanceUsed>();

        trace.add("intention.2.0");
        trace.add("action.1.0");
        trace.add("intention.1.0");
        trace.add("action.0.0");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        
        Episode e = instance.getFirstEpisode();
        EpisodeNode n = e.getRoot().getChild("action.1.0").getChild("intention.1.0").getChild("action.0.0").getChild("atomic.0.0");
        assertNotNull(n.getAssociatedNode());

        affs.add(new AffordanceUsed("intention.2.0", "type", "obj1"));
        affs.add(new AffordanceUsed("action.0.0", "type", "obj2"));
        instance.addNewNode("atomic.0.0", trace, affs, "");

        trace.clear();
        trace.add("intention.2.0");
        trace.add("action.1.0");
        trace.add("intention.1.1");
        trace.add("action.0.1");
        affs.clear();
        affs.add(new AffordanceUsed("intention.2.0", "type", "obj3"));
        affs.add(new AffordanceUsed("action.0.1", "type", "obj4"));
        instance.addNewNode("atomic.0.1", trace, affs, "");

        e = instance.currentEpisodes.get("intention.2.0");
        assertEquals(e.getRoot().numberOfSubNodes, 7);
        assertEquals(e.getRoot().numberOfSubNodesWithObjects, 11);
        assertEquals(instance.numberOfEpisodeNodes, 8);

        instance.finish();
        trace.clear();
        trace.add("intention.2.2");
        affs.clear();
        instance.addNewNode("atomic.x", trace, affs, "");
        instance.finish();
        assertEquals(instance.numberOfEpisodeNodes, 10);

        trace.clear();
        trace.add("intention.2.0");
        trace.add("action.1.0");
        trace.add("intention.1.1");
        trace.add("action.0.1");
        affs.clear();
        affs.add(new AffordanceUsed("intention.2.0", "type", "obj5"));
        affs.add(new AffordanceUsed("action.0.1", "type", "obj6"));
        instance.addNewNode("atomic.0.1", trace, affs, "");

        assertEquals(instance.getEpisodes().size(), 3);

        trace.clear();
        trace.add("intention.2.0");
        trace.add("action.1.2");
        trace.add("intention.1.2");
        trace.add("action.0.2");
        affs.clear();
        affs.add(new AffordanceUsed("intention.2.0", "type", "obj1"));
        affs.add(new AffordanceUsed("action.0.2", "type", "obj2"));
        instance.addNewNode("atomic.0.2", trace, affs, "");

        Episode e2 = instance.currentEpisodes.get("intention.2.0");
        instance.finish();

        assertEquals(e2.getRoot().numberOfSubNodes, 8);
        assertEquals(e2.getRoot().numberOfSubNodesWithObjects, 12);
        assertEquals(instance.numberOfEpisodeNodes, 19);

        assertEquals(instance.getEpisodes().size(), 3);
        assertEquals(instance.finishedEpisodes.size(), 3);

        assertEquals(e.getRoot().getName(), "intention.2.0");
        assertEquals(e2.getRoot().getName(), "intention.2.0");
        assertNotSame(e, e2);
        e.mergeWith(e2);

        assertEquals(instance.getEpisodes().size(), 2);
        assertTrue(instance.getEpisodes().contains(e));
        assertFalse(instance.getEpisodes().contains(e2));

        assertEquals(e.getRoot().numberOfSubNodes, 11);
        assertEquals(e.getRoot().numberOfSubNodesWithObjects, 18);

        n = e.getRoot().getChild("action.1.0");
        assertEquals(n.numberOfSubNodes, 6);
        assertEquals(n.numberOfSubNodesWithObjects, 9);

        n = e.getRoot().getChild("action.1.2");
        assertEquals(n.numberOfSubNodes, 3);
        assertEquals(n.numberOfSubNodesWithObjects, 4);

        assertEquals(instance.numberOfEpisodeNodes, 14);

        System.out.println("---/// TEST CHRONOBAG 04 OK ///---");
    }
    
    @Test
    public void testChronobag05() {
        //deleteEpisode test
        instance = new Chronobag(mem.getIdGenerator(), mem);
        ArrayList<String> trace = new ArrayList<String>();
        ArrayList<AffordanceUsed> affs = new ArrayList<AffordanceUsed>();

        trace.add("intention.2.1");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.2");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.2");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.4");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.1");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.5");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.4");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.5");
        instance.addNewNode("atomic.0.0", trace, affs, "");

        Iterator<Episode> it;
        Episode e1 = instance.currentEpisodes.get("intention.2.1");
        Episode e2 = instance.currentEpisodes.get("intention.2.2");
        Episode e3 = instance.currentEpisodes.get("intention.2.3");
        Episode e4 = instance.currentEpisodes.get("intention.2.4");
        Episode e5 = instance.currentEpisodes.get("intention.2.5");
        instance.finish();
        assertEquals(e3.next.size(), 5);
        assertEquals(e3.previous.size(), 5);
        assertEquals(e3.previous.get(0).getRoot().getName(), "intention.2.5");
        assertEquals(e3.previous.get(1).getRoot().getName(), "intention.2.1");
        assertEquals(e3.previous.get(2).getRoot().getName(), "intention.2.4");
        assertEquals(e3.previous.get(3).getRoot().getName(), "intention.2.2");
        assertEquals(e3.previous.get(4).getRoot().getName(), "intention.2.2");
        assertSame(e3.previous.get(3), e3.previous.get(4));
        assertEquals(e3.next.get(0).getRoot().getName(), "intention.2.2");
        assertEquals(e3.next.get(1).getRoot().getName(), "intention.2.4");
        assertEquals(e3.next.get(2).getRoot().getName(), "intention.2.1");
        assertEquals(e3.next.get(3).getRoot().getName(), "intention.2.5");
        assertEquals(e3.next.get(4).getRoot().getName(), "intention.2.4");
        assertSame(e3.next.get(1), e3.next.get(4));

        instance.deleteEpisode(e3);
        
        assertEquals(e1.next.size(), 2);
        it = e1.next.iterator();
        assertSame(it.next(), e2);
        assertSame(it.next(), e5);
        assertEquals(e2.next.size(), 1);
        it = e2.next.iterator();
        assertSame(it.next(), e4);
        assertEquals(e4.next.size(), 2);
        it = e4.next.iterator();
        assertSame(it.next(), e1);
        assertSame(it.next(), e5);
        assertEquals(e5.next.size(), 1);
        it = e5.next.iterator();
        assertSame(it.next(), e4);

        assertEquals(e1.previous.size(), 1);
        it = e1.previous.iterator();
        assertSame(it.next(), e4);
        assertEquals(e2.previous.size(), 1);
        it = e2.previous.iterator();
        assertSame(it.next(), e1);
        assertEquals(e4.previous.size(), 2);
        it = e4.previous.iterator();
        assertSame(it.next(), e5);
        assertSame(it.next(), e2);
        assertEquals(e5.previous.size(), 2);
        it = e5.previous.iterator();
        assertSame(it.next(), e4);
        assertSame(it.next(), e1);

        instance.deleteEpisode(e1);

        assertEquals(e2.next.size(), 1);
        it = e2.next.iterator();
        assertSame(it.next(), e4);
        assertEquals(e4.next.size(), 2);
        it = e4.next.iterator();
        assertSame(it.next(), e5);
        assertSame(it.next(), e5);
        assertEquals(e5.next.size(), 1);
        it = e5.next.iterator();
        assertSame(it.next(), e4);

        assertEquals(e2.previous.size(), 0);
        assertEquals(e4.previous.size(), 2);
        it = e4.previous.iterator();
        assertSame(it.next(), e5);
        assertSame(it.next(), e2);
        assertEquals(e5.previous.size(), 2);
        it = e5.previous.iterator();
        assertSame(it.next(), e4);
        assertSame(it.next(), e4);

        instance.deleteEpisode(e5);

        assertEquals(e2.next.size(), 1);
        it = e2.next.iterator();
        assertSame(it.next(), e4);
        assertEquals(e4.next.size(), 0);
        
        assertEquals(e2.previous.size(), 0);
        assertEquals(e4.previous.size(), 1);
        it = e4.previous.iterator();
        assertSame(it.next(), e2);
                
        instance.deleteEpisode(e4);
        assertEquals(e2.previous.size(), 0);
        assertEquals(e2.next.size(), 0);

        //-------------------------

        instance = new Chronobag(mem.getIdGenerator(), mem);
        trace.clear();
        assertEquals(instance.getEpisodes().size(), 0);

        trace.add("intention.2.1");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.2");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.2");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.4");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.1");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.5");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.3");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.4");
        instance.addNewNode("atomic.0.0", trace, affs, "");
        trace.set(0, "intention.2.5");
        instance.addNewNode("atomic.0.0", trace, affs, "");

        assertEquals(instance.getEpisodes().size(), 5);
        e1 = instance.currentEpisodes.get("intention.2.1");
        e2 = instance.currentEpisodes.get("intention.2.2");
        e3 = instance.currentEpisodes.get("intention.2.3");
        e4 = instance.currentEpisodes.get("intention.2.4");
        e5 = instance.currentEpisodes.get("intention.2.5");
        instance.finish();

        instance.deleteEpisode(e3);

        instance.deleteEpisode(e5);

        assertEquals(e1.next.size(), 2);
        it = e1.next.iterator();
        assertSame(it.next(), e2);
        assertSame(it.next(), e4);
        assertEquals(e2.next.size(), 1);
        it = e2.next.iterator();
        assertSame(it.next(), e4);
        assertEquals(e4.next.size(), 1);
        it = e4.next.iterator();
        assertSame(it.next(), e1);

        assertEquals(e1.previous.size(), 1);
        it = e1.previous.iterator();
        assertSame(it.next(), e4);
        assertEquals(e2.previous.size(), 1);
        it = e2.previous.iterator();
        assertSame(it.next(), e1);
        assertEquals(e4.previous.size(), 2);
        it = e4.previous.iterator();
        assertSame(it.next(), e1);
        assertSame(it.next(), e2);

        instance.deleteEpisode(e1);

        assertEquals(e2.next.size(), 1);
        it = e2.next.iterator();
        assertSame(it.next(), e4);
        assertEquals(e4.next.size(), 0);

        assertEquals(e2.previous.size(), 0);
        assertEquals(e4.previous.size(), 1);
        it = e4.previous.iterator();
        assertSame(it.next(), e2);

        instance.deleteEpisode(e2);
        assertEquals(e4.previous.size(), 0);
        assertEquals(e4.next.size(), 0);

        System.out.println("---/// TEST CHRONOBAG 05 OK ///---");
    }
}