package cz.cuni.amis.pogamut.episodic.memory;

import cz.cuni.amis.pogamut.episodic.decisions.Action;
import cz.cuni.amis.pogamut.episodic.decisions.AffordanceSlot;
import cz.cuni.amis.pogamut.episodic.decisions.AtomicAction;
import cz.cuni.amis.pogamut.episodic.decisions.DecisionTree;
import cz.cuni.amis.pogamut.episodic.decisions.Intention;
import cz.cuni.amis.pogamut.episodic.episodes.Chronobag;
import cz.cuni.amis.pogamut.episodic.episodes.Episode;
import cz.cuni.amis.pogamut.episodic.episodes.ObjectNode;
import cz.cuni.amis.pogamut.episodic.schemas.SchemaBag;
import cz.cuni.amis.pogamut.episodic.visualizer.IVisualizationListener;
import cz.cuni.amis.pogamut.episodic.visualizer.VisualizationEvent;
import cz.cuni.amis.pogamut.episodic.visualizer.VisualizationEventType;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.Semaphore;
import javax.swing.event.EventListenerList;

/**
 * <code>AgentMemory</code> is the main class of this project and integrates
 * all the information agent remembers. Its main parts are: decision tree
 * mirroring the decision nodes from pogamut end project, chronobags containing
 * memories of episodes group together on temporal basis and schemas integrating
 * all the episodes experienced by an agent into the average episodes.
 * <p>
 * Before agent memory can be used, it needs to be initialized by providing
 * the content of the decision tree. Then each time agent performs an action
 * new nodes are added into the memory structures. Memory nodes are forgotten
 * during the reorganization process that is triggered once a day when the agent
 * goes to sleep.
 *
 * @author Michal Cermak 
 */
public class AgentMemory implements IAgentMemory, Serializable {
    /**
     * Determines if a de-serialized file is compatible with this class.
     *
     * Maintainers must change this value if and only if the new version
     * of this class is not compatible with old versions. See Sun docs
     * for <a href=http://java.sun.com/products/jdk/1.1/docs/guide
     * /serialization/spec/version.doc.html> details. </a>
     *
     * Not necessary to include in first version of the class, but
     * included here as a reminder of its importance.
     */
    private static final long serialVersionUID = 1L;

    public Semaphore sem = new Semaphore(1, true);

    EventListenerList listeners = new EventListenerList();

    public void addVisualizationListener(IVisualizationListener listener) {
        listeners.add(IVisualizationListener.class, listener);
    }

    public void removeVisualizationListener(IVisualizationListener listener) {
        listeners.remove(IVisualizationListener.class, listener);
    }

    public void fireVisualizationEvent(VisualizationEvent event) {
        IVisualizationListener listenerList[] = listeners.getListeners(IVisualizationListener.class);
        for (int i = 0, n = listenerList.length; i < n; i++) {
            ((IVisualizationListener) listenerList[i]).handleVisualizationEvent(event);
        }
    }

    /**
     * Instance of <code>IdGenerator</code> class providing unique IDs to
     * all nodes in agent memory. Unique IDs are needed mainly for visualizing
     * purposes.
     */
    private IdGenerator idGenerator = new IdGenerator();

    /**
     * Variable storing the number of day when agent performed last action.
     * Used to determine if new day has already started when agent begins
     * sleeping, so that memory reorganization process can be triggered.
     * <p>
     * If agent did not perform any action yet, its value is -1.
     */
    private int lastDay = -1;

    /**
     * Memory structure mirroring decision nodes from decision making module.
     * Needs to be initialized before the memory can work correctly. Most of
     * the nodes in agent's memory store a reference to the associated node
     * in decision trees.
     *
     * @see DecisionTree
     */
    DecisionTree decisionTree = new DecisionTree();

    /**
     * Current chronobag storing episodes from the actual day. Each time a new
     * action is performed by the agent, it is added to one episode in this
     * chronobag. During memory reorganization process it is replaced by a new
     * empty chronobag and added into the list of past chronobags.
     */
    Chronobag present = new Chronobag(idGenerator, this);

    /**
     * List of all the past chronobags in agent's memory. Chronobag can get
     * here during memory reorganization process when the current chronobag
     * is moved to the set of past chronobags or when new chronobags with
     * extended temporal coverage are created. Chronobags are sorted by age
     * in several parallel lists with each list containing chronobags of one
     * level of time abstraction (daily, weekly...).
     */
    HashSet<Chronobag> past = new HashSet<Chronobag>();

    //indexed by chronobag level
    Map<Integer ,Chronobag> pastSequenceEnds = new HashMap<Integer, Chronobag>();

    /**
     * Schema bag accumullates all different experiences instances of episodes
     * into aggregated schemas. Apart from the capability to retrieve universal
     * episodes it gives us a way how to tell what sub-episodes happen more
     * often that others and it helps us to more efficiently store the concrete
     * episodes by providing a way to derive some of the nodes from what is
     * only a part of complete episode tree.
     */
    SchemaBag schemas = new SchemaBag(idGenerator, this);

    /**
     * Instance of a visualizer class providing a vizualization of decision
     * tree, individual chronobags, schemas and chronobag overview.
     */
//    VisualizationCreator  viz;

    /**
     * Parameter determining if the visualization components should be
     * displayed while agent's memory is running.
     */
    public static boolean visualize = true;

    /**
     * Getter method for the <code>decisionTree</code>.
     *
     * @return  Returns the reference to <code>DecisionTree</code> structure
     * in agent's memory.
     */
    public DecisionTree getDecisionTree() {
        return decisionTree;
    }

    public IdGenerator getIdGenerator() {
        return idGenerator;
    }

    /**
     * Private method used to recursively generate unique IDs to specified
     * <code>Intention</code> node in decision tree and its subtree.
     * This is needed because the trees are received during memory
     * initialization lacking these unique IDs.
     *
     * @param i     Reference to the <code>Intention</code> that needs new ID.
     * @param idGen Reference to IdGenerator structure in agent's memory.
     */
    private void generateIntentionIds(Intention i, IdGenerator idGen) {
        i.setId(idGen.getNewId());
        for (AffordanceSlot slot : i.getAffordances()) {
            slot.setId(idGen.getNewId());
        }
        for (Action a : i.getSubNodes()) {
            generateActionIds(a, idGen);
        }
        getDecisionTree().numberOfNodes++;
     //   getDecisionTree().branchFactor.add(i.getSubNodes().size());
    }

    /**
     * Private method used to recursively generate unique IDs to specified
     * <code>Action</code> node in decision tree and its subtree.
     * This is needed because the trees are received during memory
     * initialization lacking these unique IDs.
     *
     * @param i     Reference to the <code>Action</code> that needs new ID.
     * @param idGen Reference to IdGenerator structure in agent's memory.
     */
    private void generateActionIds(Action a, IdGenerator idGen) {
        a.setId(idGen.getNewId());
        for (AffordanceSlot slot : a.getAffordances()) {
            slot.setId(idGen.getNewId());
        }
        for (AtomicAction aa : a.getAtomicActions()) {
            aa.setId(idGen.getNewId());
        }
        for (Intention i : a.getSubNodes()) {
            generateIntentionIds(i, idGen);
        }
        getDecisionTree().numberOfAtomicActions += a.getAtomicActions().size();
        getDecisionTree().numberOfNodes++;
    //    getDecisionTree().branchFactor.add(a.getAllChildrenNodes().size());
    }

    /**
     * Method used to recursively generate unique IDs to all nodes in
     * decision trees. This is needed because the trees are received
     * during memory initialization lacking these unique IDs. This method
     * should therefore be called near the end of <code>initialize</code>
     * method.
     *
     * @param i     Reference to the decision tree without set IDs.
     * @param idGen Reference to IdGenerator structure in agent's memory.
     */
    private void generateDecisionTreeIds(DecisionTree tree, IdGenerator idGen) {
        for (Intention i : tree.topLevelGoals.values()) {
            generateIntentionIds(i, idGen);
        }
    }

    /**
     * Creates the decision tree structure based on the trees representing
     * decision plans for top-level goals received in a parameter.
     * IDs for all the nodes in new decision trees are created and visualization
     * structures are also initialized.
     * 
     * @param topLevelGoals List of <code>Intention</code> structures
     * representing possible top-level goals of agent. They are roots of
     * already complete trees consisting of intentions, actions and
     * atomic actions.
     */
    @Override
    public void initialize(Collection<Intention> topLevelGoals) {
        for (Intention i : topLevelGoals) {
            decisionTree.topLevelGoals.put(i.getName(), i);
        }
        generateDecisionTreeIds(decisionTree, idGenerator);

        present.objectNodes = new HashMap<String, ObjectNode>();
        
    }

    public void initVisualizers() {
        if (visualize) {
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.INITIALIZATION, this));
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_PAST_CHRONOBAGS, this));
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_PRESENT_CHRONOBAG, this));
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_CHRONOBAG_VIEW, this));
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_SCHEMA_BAG, this));
        }
    }

    /**
     * This method should be called each time agent executes an atomic action.
     * It is the only way to add new <code>EpisodeNode</code>s into the <code>Episode</code>.
     * It invokes the equivalent method in current chronobag and pass it all the
     * parameters, then it finds (or creates) the current <code>Episode</code>
     * according to the top level goal in trace. And invokes the <code>addNewNode</code>
     * method on that episode while passing it the same parameters.
     *
     * @param   atomicAction    Name of an atomic action that was executed.
     * @param   trace   List of names of decision nodes that lead to
     * the atomic action executed. The top-level goal is first in the list and
     * parent of atomic action is last item in the list.
     * @param   affordances     List of <code>AffordanceUsed</code> objects.
     * Each object represent one filled affordance slot that was
     * needed for the execution of current atomic action.
     */
    @Override
    public boolean addNewNode(String atomicAction, ArrayList<String> trace, ArrayList<AffordanceUsed> affordances) {
        return addNewNode(atomicAction, trace, "", affordances);
    }

    /**
     * This method should be called each time agent executes an atomic action.
     * It is the only way to add new <code>EpisodeNode</code>s into the <code>Episode</code>s.
     * It invokes the equivalent method in current chronobag and pass it all the
     * parameters, then it finds (or creates) the current <code>Episode</code>
     * according to the top level goal in trace. And invokes the <code>addNewNode</code>
     * method on that episode while passing it the same parameters.
     * <p>
     * In case agent has started the sleep action for the first time in a day
     * memory reorganization process is also triggered.
     * <p>
     * All visualizations are refreshed according to new contents of agent's memory.
     *
     * @param   atomicAction    Name of an atomic action that was executed.
     * @param   trace   List of names of decision nodes that lead to
     * the atomic action executed. <strong>Parent of atomic action is first in the list and
     * top-level goal is last item in the list</strong>.
     * @param   time    String containing information about the time of execution
     * of the atomic action. Its format should be: "Fr, 03:44, day 12."
     * @param   affordances     List of <code>AffordanceUsed</code> objects.
     * Each object represent one filled affordance slot that was
     * needed for the execution of current atomic action.
     */
    public boolean addNewNode(String atomicAction, ArrayList<String> trace, String time, ArrayList<AffordanceUsed> affordances) {
        sem.acquireUninterruptibly();
        Collections.reverse(trace);
        
        Iterator<AffordanceUsed> itr = affordances.iterator();
        AffordanceUsed aff = null;
        while (itr.hasNext()){
            aff = itr.next();
            if (aff.type.equals("perceived") && !Parameters.REMEMBER_SEEN_ITEMS) {
                itr.remove();
            }
        }


        boolean success = present.addNewNode(atomicAction, trace, affordances, time);

        schemas.updateSchema(atomicAction, trace, affordances);

        Collections.reverse(trace);

        if (checkNewDay(atomicAction, time)) reorganizeMemory();
        sem.release();

        //this will cause current atomic action to be displayed with current time
        time = time.concat(" | ");
        time = time.concat(atomicAction);

        if (visualize) {
            if (present.newNodeAdded) {
                fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_PRESENT_CHRONOBAG, this));
            }
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.UPDATE_TIME, time, this));
            present.newNodeAdded = false;
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_SCHEMA_BAG, this));
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_CHRONOBAG_VIEW, this));
         }
        return success;
    }

    /**
     * This method checks whether the memory reorganization should be triggered.
     *
     * @param action    Name of the atomic action executed by an agent.
     * @param time  Time of execution of the action.
     * @return  Returns true is the agent has started the sleep action
     * for the first time in a day. Returns false otherwise or if the time
     * information cannot be parsed correctly.
     */
    private boolean checkNewDay(String action, String time) {
        //format of time is: "Fr, 03:44, day 12."
        if (!time.contains("day")) return false;
        time = time.substring(15, time.length() - 1);
        Integer day;
        try {
            day = Integer.parseInt(time);
        } catch (Exception e) {
            System.err.print("Cannot check new day - error parsing time information.");
            System.err.print(e.toString());
            return false;
        }
        if (lastDay == -1) {
            lastDay = day;
            return false;
        }
        boolean newDay = (day > lastDay);
        if ((action.equals("Look for _TO_SLEEP_IN") || action.equals("SLEEP")) && newDay) {
   //     if (action.equals("SLEEP") && newDay) {
            lastDay = day;
            return true;
        }
        return false;
    }

    /**
     * This method is responsible for memory reorganization during agent's
     * sleep. It creates the new current chronobag, shifts other chronobags
     * into past and triggers the forgetting process.
     * <p>
     * It should be called each time the agent has started the sleep action
     * for the first time in a day.
     * 
     * @return  Returns true.
     */
    private boolean reorganizeMemory() {
        HashMap<String, ObjectNode> objNodes = present.objectNodes;
        Chronobag last = present;
        present.finish();
        past.add(present);
        present = new Chronobag(idGenerator, this, last);
        pastSequenceEnds.put(0, present);
        present.objectNodes = objNodes;

        for (Chronobag c : past) {
            c.increaseDay();
        }

        if (!Parameters.NO_ABSTRACT_CHRONOBAGS) {
            copyEpisodesToAbstractChronobags();
        }

        if (!Parameters.NO_FORGETTING) {
            Chronobag c;
            for (Iterator<Chronobag> it = past.iterator(); it.hasNext();) {
                c = it.next();
                if (c.isLandmark()) continue;
                c.decideToForgetNodes();
                if (c.getEpisodes().isEmpty()) {
                    c.deleteChronobag();
                    if (c == pastSequenceEnds.get(c.getLevel())) {
                        pastSequenceEnds.put(c.getLevel(), c.getOlderChronobag());
                    }
                    it.remove();
                }
            }
        }

        if (!Parameters.NO_EPISODE_MERGING) {
            consolidateEpisodes();
        }

        if (visualize) {
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_PAST_CHRONOBAGS, this));
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_PRESENT_CHRONOBAG, this));
            fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_CHRONOBAG_VIEW, this));
        }
        return true;
    }

    /**
     * Marks specified episodic node as finished. If the root node of
     * an episode is marked as finished, whole episode is moved from
     * <code>currentEpisodes</code> list of current chronobags
     * to <code>finishedEpisodes</code> list.
     *
     * @param   node    Name of a node that was finished.
     * @param   trace   Names of traces nodes that lead to the finished node,
     * beginning with top-level goal and ending with node's parent.
     * Should be empty if node itself is top-level goal.
     * @param   succeeded   True if node was finished successfully.
     * @return  Returns true node was successfully located and marked,
     * false otherwise.
     */
    public boolean nodeFinished(String node, ArrayList<String> trace, boolean succeeded) {
        sem.acquireUninterruptibly();
        Boolean rv;
        Collections.reverse(trace);
        rv = present.finishNode(node, trace, succeeded);
        sem.release();
        return rv;
    }

    /**
     * Each time new past chronobag is created it has to be added to the set
     * of all past chronobags. This method does so and it is also responsible
     * for updating the <code>pastSequenceEnds</code> references.
     *
     * @param c Reference to the newly created chronobag.
     */
    public void registerNewPastChronobag(Chronobag c) {
        past.add(c);
        Chronobag first = pastSequenceEnds.get(c.getLevel());
        if (first == null) {
            pastSequenceEnds.put(c.getLevel(), c);
            return;
        }
        if (c.getAge().getMaxAge() < first.getAge().getMaxAge()) {
            pastSequenceEnds.put(c.getLevel(), c);
        }
    }

    /**
     * Copies all episodes from daily chronobag to more abstract chronobags
     * and updates the maxNumberOfNodes variable for chronobags.
     */
    private void copyEpisodesToAbstractChronobags() {
     // this was original version that went over all the chronobags and copied
     // the episodes passing specified threshold.
     /*   Chronobag c;
        for (int i = 0; i < Parameters.MAX_CHRONOBAG_LEVELS - 1; i++) {
            c = pastSequenceEnds.get(i);
            while (c != null) {
                if (c.isLandmark()) break;

                for (Episode e : c.getEpisodes()) {
                    if (e.calculateCopyScore() > Parameters.EPISODE_COPY_THRESHOLD_SCORE) {
                        c.copyEpisodeToAbstractChronobag(e);
                    }
                }

                c = c.getOlderChronobag();
            }
        }*/

        Chronobag c = present;
        assert (c.getEpisodes().isEmpty());
        c = c.getOlderChronobag();
        for (int i = 0; i < Parameters.MAX_CHRONOBAG_LEVELS - 1; i++) {
            for (Episode e : c.getEpisodes()) {
                c.copyEpisodeToAbstractChronobag(e);
            }
            c = c.getMoreAbstractChronobag();
        }

        c = present.getOlderChronobag();
        Chronobag ch = c;
        while (c != null) {
            c.increaseMaxNumberOfNodes(ch);
            c = c.getMoreAbstractChronobag();
        }
    }

    private void consolidateEpisodes() {
        for (Chronobag c : past) {
            for (Episode e1 : c.getEpisodes()) {
                for (Episode e2 : c.getEpisodes()) {
                 //   if (e1.getIdEpisode() < e2.getIdEpisode() && !e1.deleted && !e2.deleted) {
                    if (e1.getIdEpisode() < e2.getIdEpisode()) {
                        double similarity = e1.episodeSimilarity(e2);
                        if (decideEpisodeMerging(similarity, c.getAge().getMinAge())) {
                            e1.mergeWith(e2);
                        }
                    }
                }
            }
        }
    }

    private boolean decideEpisodeMerging(double similarity, int age) {
        if (similarity >= 1 - age * Parameters.DECIDE_EPISODE_MERGING_COEFFICIENT) return true;
        return false;
    }

    public SchemaBag getSchemaBag() {
        return schemas;
    }

    public void reviewActiveIntentions(ArrayList<String> list) {
        present.reviewActiveEpisodes(list);
    }

    public Collection<Chronobag> getChronobags() {
        Collection<Chronobag> col = new HashSet<Chronobag>();
        col.add(present);
        col.addAll(past);
        return col;
    }

    public Map<Integer ,Chronobag> getChronobagSequenceEnds() {
        return pastSequenceEnds;
    }

    public Chronobag getPresentChronobag() {
        return present;
    }

    public HashSet<Chronobag> getPastChrononags() {
        return past;
    }

    /**
     * Generates a csv file with information how different goals were satisfied.
     * Generated data is based on the actual events from the simulation.
     * All actions performed to satisfy certain goal are assigned percentages
     * providing distribution how often the goal was satisfied with each action.
     */
    public void generateStatistics(String fileName) {
        schemas.generateStatistics(fileName);
    }

    public Integer getLastEpisodeId() {
        return present.getLastEpisodeId();
    }
}
