package cz.cuni.amis.pogamut.episodic.episodes;

import cz.cuni.amis.pogamut.episodic.memory.AffordanceUsed;
import cz.cuni.amis.pogamut.episodic.memory.AgentMemory;
import cz.cuni.amis.pogamut.episodic.memory.IdGenerator;
import cz.cuni.amis.pogamut.episodic.memory.Parameters;
import cz.cuni.amis.pogamut.episodic.visualizer.VisualizationEvent;
import cz.cuni.amis.pogamut.episodic.visualizer.VisualizationEventType;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;

/**
 * The <code>Chronobag</code> class is one of the most significant class
 * in this project. It is a collection of episode memories
 * from one time period. Chronobags can consist of episode from one
 * particular day or from a longer time period.
 * <p>
 * One special instance of this class is a chronobag of episodes
 * from the current day. Any time some action is executed, it is
 * added to this chronobag and then into the current episode.
 * <p>
 * Each night a new present day chronobag is created and other
 * chronobags are shifted by one day. It is clear, the episodes
 * in chronobags are subjects to forgetting, so as new episodes are
 * created in current day chronobag, episodes in all chronobags
 * can be forgotten each night when the process of forgetting
 * is triggered.
 *
 * @author  Michal Cermak
 */
public class Chronobag implements Serializable {
    /**
     * Determines if a de-serialized file is compatible with this class.
     *
     * Maintainers must change this value if and only if the new version
     * of this class is not compatible with old versions. See Sun docs
     * for <a href=http://java.sun.com/products/jdk/1.1/docs/guide
     * /serialization/spec/version.doc.html> details. </a>
     *
     * Not necessary to include in first version of the class, but
     * included here as a reminder of its importance.
     */
    private static final long serialVersionUID = 1L;

    /**
     * A reference to ID generator structure, that will provide
     * the unique IDs to all inside node. This unique IDs are mostly
     * needed when identifying the visualized vertices (because nodes
     * can have identical names).
     */
    public final IdGenerator idGenerator;

    /**
     * A reference to the parent structure. It should be possible
     * to access decision tree, schemas or other parts of memory via
     * this reference.
     */
    private AgentMemory mem;

    /**
     * ID of this chronobag. Used as an ID of vertex representing
     * this chronobag when visualizing Chronobag View.
     */
    public final int id;

    /**
     * Determines whether this chronobag is old enough to pass a landmark
     * threshold. No episodes are forgotten from landmark episodes.
     */
    boolean landmark = false;

    /**
     * Specifies the level of abstraction of temporal location of chronobag's
     * episodes. Chronobag can contain episodes from single day or from longer
     * time frame (e.g. a week). Level 0 zero means the chronobag is daily
     * chronobag. Higher level means the chronobag contains episodes from
     * longer time period.
     * <p>
     * Default value is 0: daily chronobag.
     */
    int level = 0;

    /**
     * List of finished episodes. Most episodes remembered
     * will be in this list. An episode gets here from <code>currentEpisodes</code>
     * once it is finished. In past chronobags (all except
     * the current day) all episodes are stored here, because
     * all episodes are already finished.
     */
    ArrayList<Episode> finishedEpisodes = new ArrayList<Episode>();

    /**
     * List of current episodes. This is a list of episodes that
     * were not finished yet. Each time a new action is performed,
     * it has to be added to an episode in this list. Only current
     * chronobag will be using this list (it will be empty for past
     * chronobags).
     * <p>
     * An episode will be moved to <code>finishedEpisodes</code> once its
     * root node is finished, its goal disappears from short-term memory
     * or during agent's sleep.
     */
    HashMap<String, Episode> currentEpisodes = new HashMap<String, Episode>();

    /**
     * Link to the <code>episode<code> that was performed last - parent episode
     * of last performed action. Used to create the temporal sequence of
     * episodes.
     */
    private Episode lastEpisode = null;

    /**
     * Link to the first episode in this chronobag. Used to display episodes
     * in correct sequence when visualizing the chronobag.
     */
    private Episode firstEpisode = null;

    /**
     * Age contains information about the age of a chronobag.
     * Age of all episodes in chronobag should fit inside this interval.
     */
    private AgeInterval age = new AgeInterval(0,0);

    /**
     * Older chronobag in the list when arranged by their age.
     * Episodes in older chronobag were experienced just before episodes
     * in this chronobag.
     * <p>
     * Chronobags can appear on different levels of abstraction,
     * e.g. daily chronobags, week chronobags... This points to this next
     * chronobag in the same level of abstraction. Chronobags
     * of other levels forms separate chain of chronobags.
     */
    private Chronobag older = null;

    /**
     * Yonger chronobag in the list when arranged by their age.
     * Episodes in younger chronobag were experienced just after episodes
     * in this chronobag.
     * <p>
     * Chronobags can appear on different levels of abstraction,
     * e.g. daily chronobags, week chronobags... This points to this next
     * chronobag in the same level of abstraction. Chronobags
     * of other levels forms separate chain of chronobags.
     */
    Chronobag younger = null;

    /**
     * Reference to the youngest chronobag in the next higher level
     * of abstraction. Chronobags can be devided into levels
     * according the lenght of their age interval (daily chronobags,
     * weekly chronobags). Higher levels of abstraction are those
     * where the lenght of this interval is higher.
     */
    Chronobag nextLevel = null;

    /**
     * An indicator whether new node was added into this chronobags
     * since its visualization was last refreshed. Set to true when
     * new node is added. Set to false when the chronobag is redrawed.
     */
    public boolean newNodeAdded = false;

    /**
     * Count the total number of <code>EpisodeNode</code> nodes
     * in the chronobag.
     */
    int numberOfEpisodeNodes = 0;

    /**
     * Count the total number of <code>ObjectNode</code>s connections to
     * the <code>ObjectSlot</code>s in this chronobag. Each object attached
     * to a node in chronobag is counted as one node when it comes to
     * calculating how big is the chronobag capacity.
     */
    int numberOfUsedObjects = 0;

    /**
     * Maximum number of nodes the chronobag could contain.
     * For daily chronobags it is calculated before first forgetting process
     * is triggered. For abstract chronobags it is sum of daily values from
     * daily chronobags covering the same time period
     */
    private int maxNumberOfNodes = 0;
    /**
     * Score of a chronobag. It is used to determine how many episode nodes
     * can be kept in the chronobag after the "forgetting" process.
     */
    private double score = Parameters.MAX_NODE_SCORE;

    /**
     * Map of <code>ObjectNode</code>s remembered by the agent.
     * This map is shared among the chronobags so it is possible
     * to find all usages of specified object. Object are indexed
     * by their unique name.
     * <p>
     * In episodes object nodes are connected with the affordance
     * slots they were used in.
     */
    public HashMap<String, ObjectNode> objectNodes = new HashMap<String, ObjectNode>();;

    /** 
     * Instantiate the class by providing references to a common ID generator
     * and parent <code>AgentMemory</code> structure.
     * <p>
     * The ID of a chronobag is also generated in constructor.
     *
     * @param   idGen Reference to common <code>IdGenerator</code>.
     * @param   mem Reference to the <code>AgentMemory</code> structure.
     */
    public Chronobag(IdGenerator idGen, AgentMemory mem) {
        idGenerator = idGen;
        this.mem = mem;
        id = idGenerator.getNewId();
    }

    /**
     * Instantiate the class by providing references
     * to a common ID generator, parent <code>AgentMemory</code> structure
     * and older chronobag of the same level that will be added
     * behind the new chronobag in temporal chain.
     * <p>
     * The ID of a chronobag is also generated in constructor.
     *
     * @param   idGen refernce to common <code>IdGenerator</code>.
     * @param   mem reference to the <code>AgentMemory</code> structure.
     * @param   last    chronobag of the same level containing episodes
     * preceding those added to this chronobag.
     */
    public Chronobag(IdGenerator idGen, AgentMemory _mem, Chronobag last) {
        idGenerator = idGen;
        mem = _mem;
        id = idGenerator.getNewId();
        older = last;
        last.younger = this;
        nextLevel = last.nextLevel;
    }

    /**
     * Returns a <code>String</code> object representing this
     * <code>Chronobag</code>'s info. It is used to provide detailed
     * information about chronobag when it is invoked from
     * the visualization structures.
     *
     * @return  a string representation of the value of this object.
     */
    @Override
    public String toString() {
        String newline = System.getProperty("line.separator");
        String s = "";
        s += ((Integer)id).toString() + " Age:" + age.toString() + newline;
        s += "Number of Episode Nodes: " + numberOfEpisodeNodes + newline;
        return s;
    }

    /**
     * Getter method for the <code>younger</code> variable.
     *
     * @return  Return younger chronobag in the list when arranged by their age.
     * Episodes younger chronobag were experienced just after episodes
     * in this chronobag.
     * <p>
     * Chronobags can appear on different levels of abstraction,
     * e.g. daily chronobags, week chronobags... This points to this next
     * chronobag in the same level of abstraction. Chronobags
     * of other levels forms separate chain of chronobags.
     */
    public Chronobag getYoungerChronobag() {
        return younger;
    }

    /**
     * Getter method for the <code>older</code> variable.
     *
     * @return  Return older chronobag in the list when arranged by their age.
     * Episodes older chronobag were experienced just before episodes
     * in this chronobag.
     * <p>
     * Chronobags can appear on different levels of abstraction,
     * e.g. daily chronobags, week chronobags... This points to this next
     * chronobag in the same level of abstraction. Chronobags
     * of other levels forms separate chain of chronobags.
     */
    public Chronobag getOlderChronobag() {
        return older;
    }

    /**
     * Getter method for the <code>mem</code> variable.
     *
     * @return Returns a reference to the parent <code>AgentMemory</code> structure.
     * It should be possible to access decision tree, schemas or other
     * parts of memory via this reference.
     */
    public AgentMemory getMemory() {
        return mem;
    }

    /**
     * Getter method for the <code>numberOfEpisodeNodes</code> variable.
     *
     * @return Count the total number of <code>EpisodeNode</code> nodes
     * in the chronobag.
     */
    public int getNumberOfEpisodeNodes() {
        return numberOfEpisodeNodes;
    }

    /**
     * Returns a chronobag belonging to a higher level of time abstraction
     * chronobags that contains episode from time interval that includes
     * whole time interval of this chronobag
     * @return  Returns a reference to a chronobag or <code>null</code> if
     * no such chronobag exists.
     */
    public Chronobag getMoreAbstractChronobag() {
        if (nextLevel == null) return null;
        Chronobag c = nextLevel;
        while (c.getAge().getMaxAge() < age.getMaxAge()) {
        //while (c.getAge().getMinAge() > age.getMinAge()) {
            //c = c.getNextChronobag();
            c = c.getOlderChronobag();
            if (c == null) return null;
        }
        if (c.getAge().getMinAge() <= age.getMinAge() && c.getAge().getMaxAge() >= age.getMaxAge()) {
            return c;
        }
        return null;
    }

    /**
     * Creates a chronobag on higher level of time abstraction whose interval
     * will include whole interval of this chronobag. If such interval already
     * exists, this method only returns it (same as <code>getMoreAbstractChronobag</code>
     * method.
     *
     * @return  Returns a newly created chronobag or the one satisfying
     * requirements if it already existed.
     */
    public Chronobag createMoreAbstractChronobag() {
        Chronobag res = null;
        if (nextLevel == null) {
            res = new Chronobag(idGenerator, mem);

            res.level = level + 1;
            int ageMin = age.getMinAge();
            int ageMax = age.getMaxAge();
            int diff = Parameters.CHRONOBAG_INTERVAL_LENGHTS[level+1] - Parameters.CHRONOBAG_INTERVAL_LENGHTS[level];
            ageMin -= diff / 2;
            ageMax += diff / 2;
            if (diff % 2 == 1) ageMax++;
            res.age = new AgeInterval(ageMin, ageMax);
            mem.registerNewPastChronobag(res);

            Chronobag c = this;
            nextLevel = res;
            while (c.getYoungerChronobag() != null) {
                c = c.younger;
                c.nextLevel = res;
            }
            c = this;
            while (c.getOlderChronobag() != null) {
                c = c.older;
                c.nextLevel = res;
            }
            return res;
        }
        res = nextLevel;
        Chronobag temp = res;
        while (res != null && res.getAge().getMaxAge() < age.getMaxAge()) {
            //res = res.getNextChronobag();
            temp = res;
            res = res.getOlderChronobag();
        }
        if (res != null && res.getAge().getMinAge() <= age.getMinAge() && res.getAge().getMaxAge() >= age.getMaxAge()) {
            return res;
        }
        
        res = new Chronobag(idGenerator, mem);
        if (temp.getAge().getMaxAge() < age.getMaxAge()) {
            res.younger = temp;
            res.older = temp.older;
            temp.older = res;
            if (res.older != null) {
                res.older.younger = res;
            }
        } else {
            res.older = temp;
            temp.younger = res;
        }
    
     /*   res.previous = temp.previous;
        if (temp.previous != null) {
            temp.previous.next = res;
        }
        res.next = temp.next;
        if (temp.next != null) {
            temp.next.previous = res;
        }*/
        
        res.level = level + 1;
        int ageMin = age.getMinAge();
        int ageMax = age.getMaxAge();
        int diff = Parameters.CHRONOBAG_INTERVAL_LENGHTS[level+1] - Parameters.CHRONOBAG_INTERVAL_LENGHTS[level];
        ageMin -= diff / 2;
        ageMax += diff / 2;
        if (diff % 2 == 1) ageMax++;
        res.age = new AgeInterval(ageMin, ageMax);
        mem.registerNewPastChronobag(res);

        Chronobag c = this;
        nextLevel = mem.getChronobagSequenceEnds().get(level + 1);
        while (c.getYoungerChronobag() != null) {
            c = c.younger;
            c.nextLevel = mem.getChronobagSequenceEnds().get(level + 1);
        }
        c = this;
        while (c.getOlderChronobag() != null) {
            c = c.older;
            c.nextLevel = mem.getChronobagSequenceEnds().get(level + 1);
        }

        res.nextLevel = mem.getChronobagSequenceEnds().get(res.level + 1);
        return res;
    }

    /**
     * Method used to <strong>increase the age</strong> of a chronobag by <strong>one day</strong>.
     * Should be called each night for all the chronobags.
     * <p>
     * There is <strong>no reversal</strong> to calling this method - age of chronobags
     * or episodes cannot be decreased.
     * <p>
     * If age passes landmark threshold, chronobag is also marked as landmark.
     */
    public void increaseDay() {
        age.increase();
        if (age.getMinAge() >= Parameters.LANDMARK_AGE_THRESHOLDS[level]) {
            landmark = true;
        }
    }

    /**
     * Getter method for the <code>age</code> structure.
     *
     * @return  Class containing information about the age of a chronobag.
     * Age of all episodes in chronobag should fit inside this interval.
     */
    public AgeInterval getAge() {
        return age;
    }

    /**
     * Getter method for the <code>score</code> structure.
     *
     * @return   Score of a chronobag. It is used to determine how many
     * nodes can be kept in the chronobag after the "forgetting" process.
     */
    public double getScore() {
        return score;
    }

    /**
     * This method should be called each time agent executes an atomic action.
     * It is the only way to add new <code>EpisodeNode</code>s into the <code>Episode</code>s.
     * It finds (or creates) the current <code>Episode</code> according to
     * the top level goal in trace. And invokes the <code>addNewNode</code>
     * method on that episode while passing it the same parameters.
     * <p>
     * It is also responsible for maintaining the correct episode
     * sequence within the chronobag.
     *
     * @param   atomicAction    Name of an atomic action that was executed.
     * @param   trace   List of names of decision nodes that lead to
     * the atomic action executed. The top-level goal is first in the list and
     * parent of atomic action is last item in the list.
     * @param   affordances     List of <code>AffordanceUsed</code> objects.
     * Each object represent one filled affordance slot that was
     * needed for the execution of current atomic action.
     */
    public boolean addNewNode(String atomicAction, ArrayList<String> trace, ArrayList<AffordanceUsed> affordances) {
        return addNewNode(atomicAction, trace, affordances, "");
    }

    public boolean addNewNode(String atomicAction, ArrayList<String> trace, ArrayList<AffordanceUsed> affordances, String time) {
        if (trace.size() == 0) return false;
        String topLevelGoal = trace.get(0);
        //TODO: check if it is really a top-level goal (could be spontaneous action)
        if (!currentEpisodes.containsKey(topLevelGoal)) {
            Episode episode = new Episode(this);
            if (firstEpisode == null) firstEpisode = episode;
            currentEpisodes.put(topLevelGoal, episode);
        }

        Episode current = currentEpisodes.get(topLevelGoal);

        //update episode sequence - link current and last episode if they are instances of different episodes
        if (lastEpisode == null) lastEpisode = current;
        if (current != lastEpisode) {
            lastEpisode.next.add(current);
            current.previous.add(0, lastEpisode);
        }
        lastEpisode = current;
        
        return current.addNewNode(atomicAction, trace, affordances, time);
    }

    /** Adds new node to the map of <code>ObjectNode</code>s remembered
     * by the agent. This map is shared among the chronobags so
     * it is possible to find all usages of specified object.
     * Object are indexed by their unique name.
     * <p>
     * New node is not created if it already exists.
     *
     * @param   item    Unique name of an object.
     * @return  Returns a reference to the <code>ObjectNode</code> representing
     * the item in the parameter.
     */
    public ObjectNode createObjectNode(String item) {
        if (objectNodes.containsKey(item)) {
            return objectNodes.get(item);
        }
        objectNodes.put(item, new ObjectNode(item, idGenerator.getNewId()));
        return objectNodes.get(item);
    }

    /**
     * Marks specified episodic node as finished. If the root node of
     * an episode is marked as finished, whole episode is moved from
     * <code>currentEpisodes</code> list to <code>finishedEpisodes</code> list.
     *
     * @param   node    Name of a node that was finished.
     * @param   trace   Names of traces nodes that lead to the finished node,
     * beginning with top-level goal and ending with node's parent.
     * Should be empty if node itself is top-level goal.
     * @param   succeeded   True if node was finished successfully.
     * @return  Returns true node was successfully located and marked,
     * false otherwise.
     */
    public boolean finishNode(String node, ArrayList<String> trace, boolean succeeded) {
        if (trace.size() == 0) return false;
        String topLevelGoal = trace.get(0);
        if (!currentEpisodes.containsKey(topLevelGoal)) return false;
        boolean rv = currentEpisodes.get(topLevelGoal).finishNode(node, trace, succeeded);
        if (currentEpisodes.get(topLevelGoal).finished) {
            Episode e = currentEpisodes.remove(topLevelGoal);
            finishedEpisodes.add(e);

            uniteEpisode(e);
        }
        return rv;
    }

    /**
     * Creates and returns a collection of all episodes in a chronobag.
     * Collection will contain both finished and unfinished episodes.
     *
     * @return  Returns collection of all episodes in this chronobag.
     */
    public Collection<Episode> getEpisodes() {
        Collection<Episode> col = new ArrayList<Episode>();
        col.addAll(finishedEpisodes);
        col.addAll(currentEpisodes.values());
        return col;
    }

    /**
     * A getter method for the <code>firstEpisode</code> variable.
     *
     * @return  Returns a link to the first episode in this chronobag. Used to
     * display episodes in correct sequence when visualizing the chronobag.
     */
    public Episode getFirstEpisode() {
        return firstEpisode;
    }

    /**
     * Getter method for the <code>level</code> variable.
     * 
     * @return Returns the level of abstraction of temporal location of chronobag's
     * episodes. Chronobag can contain episodes from single day or from longer
     * time frame (e.g. a week). Level 0 zero means the chronobag is daily
     * chronobag. Higher level means the chronobag contains episodes from
     * longer time period.
     */
    public int getLevel() {
        return level;
    }

    /**
     * Method used to compute the score of this chronobag based on its
     * current properties and node it consists. It represents overall
     * attractiveness of the chronobag.
     *
     * @return  Returns the new score value.
     */
    public double calculateChronobagScore() {
        int attractive = numberOfNodesWithAttractivity(Parameters.CHRONOBAG_SCORE_NODE_THRESHOLD);
        int all = numberOfEpisodeNodes;
        if (all == 0) {
            score = 0;
            return 0;
        }
        score = attractive / all * 100;

        assert(score <= 100);
        return score;
    }

    /**
     * Creates a copy of specified episode from this chronobag and copies
     * it to the more abstract chronobag. Episode can be copied to more
     * abstract chronobag only once. Order of episodes in more abstract
     * chronobag is maintained based on episode IDs that are generated
     * in an ascending way.
     *
     * @param e Episode to be copied.
     * @return  Returns true if episode was copied successfully, returns
     * false otherwise.
     */
    public boolean copyEpisodeToAbstractChronobag(Episode e) {
        if (e.getParentChronobag() != this) return false;
        if (e.getCopied()) return false;
        Chronobag abs = getMoreAbstractChronobag();
        if (abs == null) abs = createMoreAbstractChronobag();
        Episode copy = e.createCopy(abs);
        //copy.setParentChronobag(abs);
        abs.finishedEpisodes.add(copy);
        abs.newNodeAdded = true;
        abs.numberOfEpisodeNodes += copy.getRoot().numberOfSubNodes;

        Episode predecessor = null;
        Episode successor = null;
        for (Episode ep : abs.finishedEpisodes) {
            if (ep.idEpisode < e.idEpisode) {
                if (predecessor == null || predecessor.idEpisode < ep.idEpisode) {
                    predecessor = ep;
                }
            }
            if (ep.idEpisode > e.idEpisode) {
                if (successor == null || successor.idEpisode > ep.idEpisode) {
                    successor = ep;
                }
            }
        }
        if (successor != null) {
            successor.previous.remove(predecessor);
            successor.previous.add(copy);
        }
        if (predecessor != null) {
            predecessor.next.remove(successor);
            predecessor.next.add(copy);
        }
        return true;
    }

    /**
     * Getter method for the <code>landmark</code> variable.
     * 
     * @return  Returns true if chronobag is old enough to pass a landmark
     * threshold. No episodes are forgotten from landmark episodes.
     */
    public boolean isLandmark() {
        return landmark;
    }

    /**
     * Moves all the episodes from current episode list to finished episodes
     * list. Called at the end the day for present chronobag.
     */
    public void finish() {
        for (Episode e : currentEpisodes.values()) {
            finishedEpisodes.add(e);
        }
        currentEpisodes.clear();
    }

    /**
     * Returns the number of episode nodes with at least given attractivity in this
     * chronobag.
     *
     * @param min Minimum node attractivity for a node to be added into the count.
     * @return  Returns sum of nodes with at least specified score.
     */
    private int numberOfNodesWithAttractivity(double min) {
        int count = 0;
        Queue<EpisodeNode> q = new LinkedList<EpisodeNode>();
        for (Episode e : getEpisodes()) {
            q.add(e.getRoot());
        }

        EpisodeNode n;
        while (!q.isEmpty()) {
            n = q.poll();
            if (n.getAssociatedNode() == null) continue;
            if (n.getAssociatedNode().getAttractivity() >= min) count++;
            for (EpisodeNode child : n.getChildrenNodes()) {
                q.add(child);
            }
        }
        return count;
    }

    /**
     * Calculates the k-th highers score in the list of scores of episode nodes.
     * K is calculated in this method. It is based on chronobag's age and score.
     *
     * @param scores    List of the scores of chronobag's episode nodes.
     * @return  Returns k-th highest node score of chronobag's episode nodes.
     */
    private double calculateKScore(List<Double> scores) {
        calculateChronobagScore();
        int k = calculateK(age.getMinAge(), score);
        if (k >= scores.size()) return 0;
        Collections.sort(scores);
        Collections.reverse(scores);
        if (k == 0) {
            return Parameters.MAX_NODE_SCORE;
        }
        k--;
        return scores.get(k);
    }

    /**
     * Calculates how many nodes should the chronobag have based on its age
     * and score. Number of nodes decreases with age and increases with
     * chronobag overall score.
     *
     * @param minAge    Lower bound for age of episodes in a chronobag.
     * @param chronobagScore    Overall score of chronobag based on its nodes.
     * @return  Returns K - number of episode nodes this chronobag is capable
     * to contain. If it contains more than K nodes, nodes with lowest score
     * will be deleted.
     */
    private int calculateK (int minAge, double chronobagScore) {
        double k = maxNumberOfNodes;
        if (k > Parameters.MAX_CHRONOBAG_NODES)
            k = Parameters.MAX_CHRONOBAG_NODES;
        double limit = Parameters.LANDMARK_AGE_THRESHOLDS[level] - 1;
        if (limit < minAge) limit = minAge;

     //   double k2 = (limit - minAge) / limit * k;
     //   k = (int) Math.round(k2);
       // k = k * (limit - minAge) / limit;
        
        if (Parameters.FORGETTING_CURVE_COEFFICIENT > -1) {
            double exponent = Parameters.FORGETTING_CURVE_COEFFICIENT;
            k *= Math.pow((limit - minAge)/limit,exponent);
        }
        if (Parameters.FORGETTING_CURVE_COEFFICIENT == -1) {
            k *= Math.pow(Math.E, -1 * (age.getMinAge() + age.getMaxAge()) / 2);
        }

        k += chronobagScore / 2;
        return (int) Math.round(k);
    }

    public void decideToForgetNodes() {
        ArrayList<Double> nodeScores = new ArrayList<Double>();
        
        Queue<EpisodeNode> q = new LinkedList<EpisodeNode>();
        for (Episode e : getEpisodes()) {
            q.add(e.getRoot());
        }
        EpisodeNode n;
        while (!q.isEmpty()) {
            n = q.poll();
            assert n.validateNode(n) : n.getId();
            for (ObjectSlot s : n.getObjectSlots()) {
                nodeScores.addAll(s.computeScore());
            }
            // IMPORTANT: node calculation also uses slot scores, so they should be calculated first
            nodeScores.add(n.calculateScore());
            for (EpisodeNode child : n.getChildrenNodes()) {
                q.add(child);
            }
        }

        double kScore = calculateKScore(nodeScores);

        for (Episode e : getEpisodes()) {
            q.add(e.getRoot());
            assert e.getRoot().validateNode(e.getRoot()) : e.getRoot().getId();
        }
        while (!q.isEmpty()) {
            n = q.poll();
       //     System.out.print(n.getId() + " ");
            if (n.consumed) continue;

            Collection<EpisodeNode> pred = n.getPredecessor().values();
            Collection<EpisodeNode> suc = n.getSuccessor().values();
            EpisodeNode par = n.getParent();
            Collection<EpisodeNode> ch = n.getChildrenNodes();

            assert n.validateNode(n) : n.getId();
            assert n.validateNode(par) : n.getId();
            assert n.validateNode(ch) : n.getId();
            assert n.validateNode(pred) : n.getId();
            assert n.validateNode(suc) : n.getId();

            for (EpisodeNode child : n.getChildrenNodes()) {
                if (!q.contains(child)) {
                //assert !q.contains(child);
                    q.add(child);
                }
            }
            Collection<ObjectSlot> col = new HashSet<ObjectSlot>();
            col.addAll(n.getObjectSlots());
            for (ObjectSlot s : col) {
                s.forgetConnections(kScore);
                if (s.getUsedObjects().isEmpty()) s.deleteSlot();
            }
            
            assert n.validateNode(par) : n.getId();
            assert n.validateNode(ch) : n.getId();
            assert n.validateNode(pred) : n.getId();
            assert n.validateNode(suc) : n.getId();

            boolean deleted = false;
            if (n.getScore() < kScore)  {
                n.deleteNode();
                deleted = true;
            }

            if (!deleted) {
                assert n.validateNode(par) : n.getId();
                assert n.validateNode(ch) : n.getId();
                assert n.validateNode(pred) : n.getId();
                assert n.validateNode(suc) : n.getId();
            }
        }
    }

    void episodesMerged(Episode merged, Episode absorbed, int removedNodes, int removedNodesObjects) {
        if (currentEpisodes.containsValue(absorbed)) {
            currentEpisodes.remove(absorbed.getRoot().getName());
        }
        finishedEpisodes.remove(absorbed);
        if (absorbed == firstEpisode) firstEpisode = merged;
        if (absorbed == lastEpisode) lastEpisode = merged;
        int temp = numberOfEpisodeNodes; //because not
        deleteEpisode(absorbed);
        numberOfEpisodeNodes = temp;
        newNodeAdded = true;
        numberOfEpisodeNodes -= removedNodes;
    }

    public void reviewActiveEpisodes(ArrayList<String> list) {
        boolean updated = false;
        for (Map.Entry<String, Episode> entry : currentEpisodes.entrySet()) {
            if (!list.contains(entry.getKey())) {
                Episode e = entry.getValue();
                e.finished = true;
                updated = true;
            }
        }
        if (updated) {
            mem.fireVisualizationEvent(new VisualizationEvent(this, VisualizationEventType.REFRESH_PRESENT_CHRONOBAG, mem));
        }
    }

    public boolean uniteEpisode(Episode e) {
        for (Episode prev : e.previous) {
            if (e.episodeSimilarity(prev) == 1) {
                prev.mergeWith(e);
                return true;
            }
        }
        return false;
    }

    public void deleteEpisode(Episode e) {
        //removes from finished episodes only, episodes should not be deleted unless chronobag is finished
        finishedEpisodes.remove(e);
        if (currentEpisodes.containsValue(e)) {
            System.err.println ("Deleting unfinished episode: " + e);
        }
     //   numberOfEpisodeNodes -= 1 + e.getRoot().numberOfSubNodes;
        //not needed because the nodes are deleted in fullDelete at the end of this method
        //number of EpisodeNodes is updated there
        Episode e2;
        ArrayList<Episode> tempNext = new ArrayList<Episode>(e.next);
        Collections.reverse(e.previous);
        Iterator<Episode> it;// = e.next.iterator();

        if (e == firstEpisode) {
            if (!e.next.isEmpty()) {
                e.next.remove(0);
            }
        }
        for (Episode prev : e.previous) {
            for (int i = 0; i < prev.next.size();) {
                if (prev.next.get(i) == e) {
                    it = e.next.iterator();
                    if (!it.hasNext()) {
                        prev.next.remove(i);
               //         assert(prev.next.size() == i);
                        continue;
                    }
                    e2 = it.next();
                    it.remove();
                    if (e2 == prev) {
                        prev.next.remove(i);
                        continue;
                    }
                    prev.next.set(i, e2);
                    break;
                } else {
                    i++;
                }
            }
        }
        e.next = tempNext;
        if (e == lastEpisode) {
            if (!e.previous.isEmpty()) {
        //        e.previous.remove(e.previous.size() - 1);
            }
        }
        for (Episode n : e.next) {
            for (int i = n.previous.size() - 1; i >= 0;) {
                if (n.previous.get(i) == e) {
                    if (n.previous.size() < e.next.size() && e == firstEpisode) {
                        n.previous.remove(i);
                        break;
                    }
                    it = e.previous.iterator(); //already reversed
                    if (!it.hasNext()) {
                        n.previous.remove(i);
                        i--;
                     //   assert(n.previous.size() == i);
                        continue;
                    }
                    e2 = it.next();
                    it.remove();
                    if (e2 == n) {
                        n.previous.remove(i);
                        i--;
                        continue;
                    }
                    n.previous.set(i, e2);
                    break;
                } else {
                    i--;
                }
            }
        }
        if (e.getRoot() != null) e.getRoot().fullDelete(true);
    }

    public void deleteChronobag() {
        if (younger == null) {
            Chronobag c = mem.getChronobagSequenceEnds().get(level - 1);
            while (c != null) {
                c.nextLevel = older;
                c = c.older;
            }
        }

        if (older != null) older.younger = younger;
        if (younger != null) younger.older = older;
    }

    /**
     * Getter method for <b>maxNumberOfNodes</b>
     * Maximum number of nodes the chronobag could contain.
     * For daily chronobags it is calculated before first forgetting process
     * is triggered. For abstract chronobags it is sum of daily values from
     * daily chronobags covering the same time period
     *
     * @return  Returns value of <b>maxNumberOfNodes</b> variable.
     */
    public int getMaxNumberOfNodes() {
        return maxNumberOfNodes;
    }

    /**
     * Increases the <code>maxNumberOfNodes</code> variable by the number of nodes
     * in a chronobag. <code>maxNumberOfNodes</code> is initially zero and it increase
     * only via this method. For daily chronobags it should be called just
     * once just before forgetting process is triggered for the first time
     * as the chronobag is no longer the present chronobag. For abstract
     * chronobag it is called once for each daily chronobag whose age belongs
     * to the age period of abstract chronobag.
     *
     * @param   c   Daily chronobag whose actual number of nodes will be added
     * to the maximum number of nodes for this chronobag.
     */
    public void increaseMaxNumberOfNodes(Chronobag c) {
        assert c.level == 0;
        assert c.age.getMinAge() >= age.getMinAge();
        assert c.age.getMaxAge() <= age.getMaxAge();
        maxNumberOfNodes += c.numberOfEpisodeNodes + c.numberOfUsedObjects;
    }

    public Integer getLastEpisodeId() {
        if (lastEpisode == null) return null;
        return lastEpisode.getIdEpisode();
    }

    public Episode getEpisode(Integer id) {
        for (Episode e : getEpisodes()) {
            if (e.getIdEpisode() == id) return e;
        }
        return null;
    }
}
