package cz.cuni.amis.pogamut.episodic.decisions;

import cz.cuni.amis.pogamut.episodic.memory.AffordanceUsed;
import cz.cuni.amis.pogamut.episodic.memory.IdGenerator;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;

/**
 * Name of this class can be a little misleading because <code>DecisionTree</code>
 * is actually a forest constiting of several trees. Roots of these trees
 * are all possible agent's top-level goals. It is a copy of decision plans
 * from pogamut end part of the project.
 * <p>
 * These trees have to be constructed before any information can be added to
 * agent's memory structures.
 * <p>
 * Each action agent performs is representing by equivalent node in decision
 * tree. This node is located through this class by locating the correct
 * top-level goal first and by going through tree structure until the specified
 * node is located. Each time agent performs an action it is added to memory
 * by providing trace of such nodes that lead to a node representing atomic
 * action that was just executed.
 *
 * @author Michal Cermak
 * @see Node
 */
public class DecisionTree implements Serializable {
    /**
     * Determines if a de-serialized file is compatible with this class.
     *
     * Maintainers must change this value if and only if the new version
     * of this class is not compatible with old versions. See Sun docs
     * for <a href=http://java.sun.com/products/jdk/1.1/docs/guide
     * /serialization/spec/version.doc.html> details. </a>
     *
     * Not necessary to include in first version of the class, but
     * included here as a reminder of its importance.
     */
    private static final long serialVersionUID = 1L;

    /**
     * Map of all possible top-level goals agent can have. They are indexed
     * by their name (so there cannot be two top-level goals with identical
     * names). Through here it is possible to get to any node in the decision
     * trees.
     */
    public HashMap<String, Intention> topLevelGoals = new HashMap<String, Intention>();

    public int numberOfAtomicActions = 0;

    /**
     * This number is <strong>without</strong> atomic actions. It only includes
     * intentions and actions.
     */
    public int numberOfNodes = 0;

    public ArrayList<Integer> branchFactor = new ArrayList<Integer>();

    public ArrayList<Integer> getBranchFactor() {
        for (Intention i : topLevelGoals.values()) {
            branchFactor.add(i.getPossibleSubTrees());
        }
        
        return branchFactor;
    }

    /**
     * This method is used in schemabag when the trace specified by a list
     * of node names is not enough and other information on nodes are needed.
     * It transforms the trace of node names into the equivalent trace of
     * references to node themselves.
     * 
     * @param trace Ordered list of names of nodes beginning with a name
     * of one top-level goal a next item being always a name of one child node
     * of previous node.
     * @return  Returns the equivalent trace of references to the <code>Node</code>
     * structures that was specified by the names in parameter. 
     * <p>
     * If there is non-existent node name in the middle of the list, method
     * returns references to the nodes named by previous correct entries only.
     */
    public ArrayList<Node> getTrace(ArrayList<String> trace) {
        ArrayList<Node> list = new ArrayList<Node>();

        if (trace.isEmpty()) return list;
        Intention i = topLevelGoals.get(trace.get(0));
        Action a;
        int index = 1;
        while (index < trace.size()) {
            if (i != null) {
                list.add(i);
                a = (Action) i.getSubNode(trace.get(index));
                index++;
                if (a != null) {
                    list.add(a);
                    if (index < trace.size()) {
                        i = (Intention) a.getSubIntention(trace.get(index));
                    }
                    index++;
                } else break;
            } else break;
        }

        return list;
    }

    /**
     * In case actions performed were not included in xmls. This method creates
     * nodes in decision tree, representing these actions.
     *
     * @param trace Ordered list of names of nodes beginning with a name
     * of one top-level goal a next item being always a name of one child node
     * of previous node.
     * @return  Return true if a new node was added, false otherwise.
     */
    public boolean ensureNodes(String atomicAction, ArrayList<String> trace, ArrayList<AffordanceUsed> affordances, IdGenerator gen) {
        boolean updated = false;
        Intention i = topLevelGoals.get(trace.get(0));
        if (i == null) {
            i = new Intention(trace.get(0), 0);
            i.setId(gen.getNewId());
            topLevelGoals.put(trace.get(0), i);
        }
        Action a = null;
        int index = 1;
        while (index < trace.size()) {
            a = (Action) i.getSubNode(trace.get(index));
            if (a == null) {
                a = new Action(trace.get(index), 0);
                a.setId(gen.getNewId());
                i.addSubNode(a);
                updated = true;
            }
            index++;
            if (index == trace.size()) break;
            i = a.getSubIntention(trace.get(index));
            if (i == null) {
                i = new Intention(trace.get(index), 0);
                i.setId(gen.getNewId());
                a.addSubNode(i);
                a = null;
                updated = true;
            }
            index++;
        }

        if (a == null) {
            System.err.println("Incorrect number of nodes in the trace. Atomic action was not added.");
        } else {
            if (a.getAtomicAction(atomicAction) == null) {
                AtomicAction aa = new AtomicAction(atomicAction, 0);
                aa.setId(gen.getNewId());
                a.addAtomicAction(aa);
                updated = true;
            }
        }

        return updated;
    }
}
