/**
 * Copyright 1998-2010 Epic Games, Inc. All Rights Reserved.
 *
 * Uber post process effect
 *
 */
class UberPostProcessEffect extends DOFBloomMotionBlurEffect
	native;

/** */
var() vector SceneShadows;
/** */
var() vector SceneHighLights;
/** */
var() vector SceneMidTones;
/** */
var() float  SceneDesaturation;

/** Allows to specify the tone mapper function which maps HDR colors into the LDR color range. */
var(Tonemapper) enum ETonemapperType
{
	Tonemapper_Off, 
	Tonemapper_Filmic, 
	Tonemapper_Customizable, 
} TonemapperType;

/**
 * This tonemapper property allows to specify the HDR brightness value that is mapping to the maximum LDR value. Brighter values will be
 * mapped to white (good values are in the range 2 to 16). Only affects the "Customizable" tonemapper.
 */
var(Tonemapper) float TonemapperRange;

/**
 * This tonemapper property allows to adjust the mapping of the darker colors (tonemapper toe). 
 * As the adjustment is independent per color channel it can introduce slight shifts color and saturation changes.
 * Only affects the "Customizable" tonemapper.
 * 0=linear .. 1=crushed darks (more filmic) 
 */
var(Tonemapper) float TonemapperToeFactor;

/**
 * Scale the input for the tonemapper. Only used if a tonemapper is specified.
 * >=0, 0:black, 1(default), >1 brighter
 */
var(Tonemapper) float TonemapperScale;

/**
 * The radius of the soft edge for motion blur. A value bigger than 0 enables the soft edge motion blur. The method improves motion blur
 * by blurring the silhuette of moving objects. The method works in screen space. Therefore the performance of the method only depends
 * on screen size, not on the object/vertex/triangle count.
 */
var() float MotionBlurSoftEdgeKernelSize;

/** Whether the image grain (noise) is enabled, to fight 8 bit quantization artifacts and to simulate film grain (scaled by SceneImageGrainScale) */
var() bool bEnableImageGrain;

/** Image grain scale, only affects the darks, >=0, 0:none, 1(strong) should be less than 1 */
var() float SceneImageGrainScale;

var deprecated bool bEnableHDRTonemapper;
var deprecated float SceneHDRTonemapperScale;


cpptext
{
	// UPostProcessEffect interface

	/**
	 * Creates a proxy to represent the render info for a post process effect
	 * @param WorldSettings - The world's post process settings for the view.
	 * @return The proxy object.
	 */
	virtual class FPostProcessSceneProxy* CreateSceneProxy(const FPostProcessSettings* WorldSettings);

	// UObject interface

	/**
	* Called after this instance has been serialized.  UberPostProcessEffect should only
	* ever exists in the SDPG_PostProcess scene
	*/
	virtual void PostLoad();
	
	/**
	 * Called when properties change.  UberPostProcessEffect should only
	 * ever exists in the SDPG_PostProcess scene
	 */
	virtual void PostEditChangeProperty(FPropertyChangedEvent& PropertyChangedEvent);

	/**
	* Tells the SceneRenderer is this effect includes the uber post process.
	*/
	virtual UBOOL IncludesUberpostprocess() const
	{
		return TRUE;
	}
}

//
// The UberPostProcessingEffect performs DOF, Bloom, Material (Sharpen/Desaturate) and Tone Mapping
//
// For the DOF and Bloom parameters see DOFAndBloomEffect.uc.  The Material parameters are used as
// follows:
//
// Color0 = ((InputColor - SceneShadows) / SceneHighLights) ^ SceneMidTones
// Color1 = Luminance(Color0)
//
// OutputColor = Color0 * (1 - SceneDesaturation) + Color1 * SceneDesaturation
//

defaultproperties
{
    SceneShadows=(X=0.0,Y=0.0,Z=-0.003);
    SceneHighLights=(X=0.8,Y=0.8,Z=0.8);
    SceneMidTones=(X=1.3,Y=1.3,Z=1.3);
    SceneDesaturation=0.4; 
	bEnableHDRTonemapper=FALSE;
	bEnableImageGrain=FALSE;
	SceneHDRTonemapperScale=1.0;
	TonemapperScale=1.0;
	SceneImageGrainScale=0.02;
	TonemapperRange=8;
	TonemapperToeFactor=1;
}
