/**
 * Base class for all data providers which provide additional dynamic information about a specific static data provider instance.
 *
 * Copyright 1998-2010 Epic Games, Inc. All Rights Reserved.
 */
class UIResourceCombinationProvider extends UIDataProvider
	native(UIPrivate)
	PerObjectConfig
	Config(Game)
	implements(UIListElementProvider,UIListElementCellProvider)
	abstract;

/**
 * Each combo provider is linked to a single static resource data provider.  The name of the combo provider should match the name of the
 * static resource it's associated with, as the dynamic resource data store will match combo providers to the static provider with the same name.
 */
var	transient	UIResourceDataProvider					StaticDataProvider;

/**
 * The data provider which provides access to a player's profile data.
 */
var	transient	UIDataProvider_OnlineProfileSettings	ProfileProvider;

cpptext
{
	/* === UIDataProvider interface === */
	/**
	 * Gets the list of data fields exposed by this data provider.
	 *
	 * @param	out_Fields	will be filled in with the list of tags which can be used to access data in this data provider.
	 *						Will call GetScriptDataTags to allow script-only child classes to add to this list.
	 */
	virtual void GetSupportedDataFields( TArray<struct FUIDataProviderField>& out_Fields );

	/**
	 * Resolves the value of the data field specified and stores it in the output parameter.
	 *
	 * @param	FieldName		the data field to resolve the value for;  guaranteed to correspond to a property that this provider
	 *							can resolve the value for (i.e. not a tag corresponding to an internal provider, etc.)
	 * @param	out_FieldValue	receives the resolved value for the property specified.
	 *							@see GetDataStoreValue for additional notes
	 * @param	ArrayIndex		optional array index for use with data collections
	 */
	virtual UBOOL GetFieldValue( const FString& FieldName, struct FUIProviderFieldValue& out_FieldValue, INT ArrayIndex=INDEX_NONE );

	/**
	 * Parses the data store reference and resolves it into a value that can be used by the UI.
	 *
	 * @param	MarkupString	a markup string that can be resolved to a data field contained by this data provider, or one of its
	 *							internal data providers.
	 * @param	out_FieldValue	receives the value of the data field resolved from MarkupString.  If the specified property corresponds
	 *							to a value that can be rendered as a string, the field value should be assigned to the StringValue member;
	 *							if the specified property corresponds to a value that can only be rendered as an image, such as an object
	 *							or image reference, the field value should be assigned to the ImageValue member.
	 *							Data stores can optionally manually create a UIStringNode_Text or UIStringNode_Image containing the appropriate
	 *							value, in order to have greater control over how the string node is initialized.  Generally, this is not necessary.
	 *
	 * @return	TRUE if this data store (or one of its internal data providers) successfully resolved the string specified into a data field
	 *			and assigned the value to the out_FieldValue parameter; false if this data store could not resolve the markup string specified.
	 */
	virtual UBOOL GetDataStoreValue( const FString& MarkupString, struct FUIProviderFieldValue& out_FieldValue )
	{
		return GetFieldValue(MarkupString,out_FieldValue);
	}

	/* === IUIListElementProvider interface === */
	/**
	 * Retrieves the list of all data tags contained by this element provider which correspond to list element data.
	 *
	 * @return	the list of tags supported by this element provider which correspond to list element data.
	 */
	virtual TArray<FName> GetElementProviderTags();

	/**
	 * Returns the number of list elements associated with the data tag specified.
	 *
	 * @param	FieldName	the name of the property to get the element count for.  guaranteed to be one of the values returned
	 *						from GetElementProviderTags.
	 *
	 * @return	the total number of elements that are required to fully represent the data specified.
	 */
	virtual INT GetElementCount( FName FieldName );

	/**
	 * Retrieves the list elements associated with the data tag specified.
	 *
	 * @param	FieldName		the name of the property to get the element count for.  guaranteed to be one of the values returned
	 *							from GetElementProviderTags.
	 * @param	out_Elements	will be filled with the elements associated with the data specified by DataTag.
	 *
	 * @return	TRUE if this data store contains a list element data provider matching the tag specified.
	 */
	virtual UBOOL GetListElements( FName FieldName, TArray<INT>& out_Elements );

	/**
	 * Determines whether a member of a collection should be considered "enabled" by subscribed lists.  Disabled elements will still be displayed in the list
	 * but will be drawn using the disabled state.
	 *
	 * @param	FieldName			the name of the collection data field that CollectionIndex indexes into.
	 * @param	CollectionIndex		the index into the data field collection indicated by FieldName to check
	 *
	 * @return	TRUE if FieldName doesn't correspond to a valid collection data field, CollectionIndex is an invalid index for that collection,
	 *			or the item is actually enabled; FALSE only if the item was successfully resolved into a data field value, but should be considered disabled.
	 */
	virtual UBOOL IsElementEnabled( FName FieldName, INT CollectionIndex );

	/**
	 * Retrieves a UIListElementCellProvider for the specified data tag that can provide the list with the available cells for this list element.
	 * Used by the UI editor to know which cells are available for binding to individual list cells.
	 *
	 * @param	FieldName		the tag of the list element data field that we want the schema for.
	 *
	 * @return	a pointer to some instance of the data provider for the tag specified.  only used for enumerating the available
	 *			cell bindings, so doesn't need to actually contain any data (i.e. can be the CDO for the data provider class, for example)
	 */
	virtual TScriptInterface<class IUIListElementCellProvider> GetElementCellSchemaProvider( FName FieldName );

	/**
	 * Retrieves a UIListElementCellProvider for the specified data tag that can provide the list with the values for the cells
	 * of the list element indicated by CellValueProvider.DataSourceIndex
	 *
	 * @param	FieldName		the tag of the list element data field that we want the values for
	 * @param	ListIndex		the list index for the element to get values for
	 *
	 * @return	a pointer to an instance of the data provider that contains the value for the data field and list index specified
	 */
	virtual TScriptInterface<class IUIListElementCellProvider> GetElementCellValueProvider( FName FieldName, INT ListIndex );

	/* === IUIListElementCellProvider interface === */
	/**
	 * Retrieves the list of tags that can be bound to individual cells in a single list element.
	 *
	 * @param	FieldName		the name of the field the desired cell tags are associated with.  Used for cases where a single data provider
	 *							instance provides element cells for multiple collection data fields.
	 * @param	out_CellTags	receives the list of tag/column headers that can be bound to element cells for the specified property.
	 */
	virtual void GetElementCellTags( FName FieldName, TMap<FName,FString>& out_CellTags );

	/**
	 * Retrieves the field type for the specified cell.
	 *
	 * @param	FieldName		the name of the field the desired cell tags are associated with.  Used for cases where a single data provider
	 *							instance provides element cells for multiple collection data fields.
	 * @param	CellTag				the tag for the element cell to get the field type for
	 * @param	out_CellFieldType	receives the field type for the specified cell; should be a EUIDataProviderFieldType value.
	 *
	 * @return	TRUE if this element cell provider contains a cell with the specified tag, and out_CellFieldType was changed.
	 */
	virtual UBOOL GetCellFieldType( FName FieldName, const FName& CellTag, BYTE& out_CellFieldType );

	/**
	 * Resolves the value of the cell specified by CellTag and stores it in the output parameter.
	 *
	 * @param	FieldName		the name of the field the desired cell tags are associated with.  Used for cases where a single data provider
	 *							instance provides element cells for multiple collection data fields.
	 * @param	CellTag			the tag for the element cell to resolve the value for
	 * @param	ListIndex		the UIList's item index for the element that contains this cell.  Useful for data providers which
	 *							do not provide unique UIListElement objects for each element.
	 * @param	out_FieldValue	receives the resolved value for the property specified.
	 *							@see GetDataStoreValue for additional notes
	 * @param	ArrayIndex		optional array index for use with cell tags that represent data collections.  Corresponds to the
	 *							ArrayIndex of the collection that this cell is bound to, or INDEX_NONE if CellTag does not correspond
	 *							to a data collection.
	 */
	virtual UBOOL GetCellFieldValue( FName FieldName, const FName& CellTag, INT ListIndex, struct FUIProviderFieldValue& out_FieldValue, INT ArrayIndex=INDEX_NONE );
}

/**
 * Provides the data provider with the chance to perform initialization, including preloading any content that will be needed by the provider.
 *
 * @param	bIsEditor					TRUE if the editor is running; FALSE if running in the game.
 * @param	InStaticResourceProvider	the data provider that provides the static resource data for this combo provider.
 * @param	InProfileProvider			the data provider that provides profile data for the player associated with the owning data store.
 */
event InitializeProvider( bool bIsEditor, UIResourceDataProvider InStaticResourceProvider, UIDataProvider_OnlineProfileSettings InProfileProvider )
{
	StaticDataProvider = InStaticResourceProvider;
	ProfileProvider = InProfileProvider;
}

`define		debug_resourcecombo_provider	1==0

/**
 * Retrieves the list of all data tags contained by this element provider which correspond to list element data.
 *
 * @return	the list of tags supported by this element provider which correspond to list element data.
 */
event array<name> GetElementProviderTags()
{
	local array<name> Tags;

	`log(`location,`debug_resourcecombo_provider);
	Tags.Length = 0;
	return Tags;
}

/**
 * Returns the number of list elements associated with the data tag specified.
 *
 * @param	FieldName	the name of the property to get the element count for.  guaranteed to be one of the values returned
 *						from GetElementProviderTags.
 *
 * @return	the total number of elements that are required to fully represent the data specified.
 */
event int GetElementCount( name FieldName )
{
	local int Result;

	`log(`location @ `showvar(FieldName),`debug_resourcecombo_provider);
	Result = 0;
	return Result;
}

/**
 * Retrieves the list elements associated with the data tag specified.
 *
 * @param	FieldName		the name of the property to get the element count for.  guaranteed to be one of the values returned
 *							from GetElementProviderTags.
 * @param	out_Elements	will be filled with the elements associated with the data specified by DataTag.
 *
 * @return	TRUE if this data store contains a list element data provider matching the tag specified.
 */
event bool GetListElements(name FieldName, out array<int> out_Elements)
{
	local bool bResult;

	bResult = false;

	`log(`location @ `showvar(FieldName),`debug_resourcecombo_provider);
	return bResult;
}

/**
 * Determines whether a member of a collection should be considered "enabled" by subscribed lists.  Disabled elements will still be displayed in the list
 * but will be drawn using the disabled state.
 *
 * @param	FieldName			the name of the collection data field that CollectionIndex indexes into.
 * @param	CollectionIndex		the index into the data field collection indicated by FieldName to check
 *
 * @return	TRUE if FieldName doesn't correspond to a valid collection data field, CollectionIndex is an invalid index for that collection,
 *			or the item is actually enabled; FALSE only if the item was successfully resolved into a data field value, but should be considered disabled.
 */
event bool IsElementEnabled( name FieldName, int CollectionIndex )
{
	local bool bResult;

	bResult = false;

	`log(`location @ `showvar(FieldName),`debug_resourcecombo_provider);
	return bResult;
}

/**
 * Retrieves a UIListElementCellProvider for the specified data tag that can provide the list with the available cells for this list element.
 * Used by the UI editor to know which cells are available for binding to individual list cells.
 *
 * @param	FieldName		the tag of the list element data field that we want the schema for.
 *
 * @return	a pointer to some instance of the data provider for the tag specified.  only used for enumerating the available
 *			cell bindings, so doesn't need to actually contain any data (i.e. can be the CDO for the data provider class, for example)
 */
event bool GetElementCellSchemaProvider( name FieldName, out UIListElementCellProvider out_SchemaProvider )
{
	local bool bResult;

	bResult = false;

	`log(`location @ `showvar(FieldName),`debug_resourcecombo_provider);
	return bResult;
}

/**
 * Retrieves a UIListElementCellProvider for the specified data tag that can provide the list with the values for the cells
 * of the list element indicated by CellValueProvider.DataSourceIndex
 *
 * @param	FieldName		the tag of the list element data field that we want the values for
 * @param	ListIndex		the list index for the element to get values for
 *
 * @return	a pointer to an instance of the data provider that contains the value for the data field and list index specified
 */
event bool GetElementCellValueProvider( name FieldName, int ListIndex, out UIListElementCellProvider out_ValueProvider )
{
	local bool bResult;

	bResult = false;

	`log(`location @ `showvar(FieldName),`debug_resourcecombo_provider);
	return bResult;
}

/* === IUIListElementCellProvider interface === */
/**
 * Retrieves the list of tags that can be bound to individual cells in a single list element.
 *
 * @param	FieldName		the name of the field the desired cell tags are associated with.  Used for cases where a single data provider
 *							instance provides element cells for multiple collection data fields.
 * @param	out_CellTags	receives the list of tag/column headers that can be bound to element cells for the specified property.
 */
event GetElementCellTags( name FieldName, out array<name> CellFieldTags, optional out array<string> ColumnHeaderDisplayText )
{
	`log(`location @ `showvar(FieldName),`debug_resourcecombo_provider);
}

/**
 * Retrieves the field type for the specified cell.
 *
 * @param	FieldName		the name of the field the desired cell tags are associated with.  Used for cases where a single data provider
 *							instance provides element cells for multiple collection data fields.
 * @param	CellTag				the tag for the element cell to get the field type for
 * @param	out_CellFieldType	receives the field type for the specified cell; should be a EUIDataProviderFieldType value.
 *
 * @return	TRUE if this element cell provider contains a cell with the specified tag, and out_CellFieldType was changed.
 */
event bool GetCellFieldType( name FieldName, name CellTag, out EUIDataProviderFieldType FieldType )
{
	local bool bResult;

	bResult = false;

	`log(`location @ `showvar(FieldName),`debug_resourcecombo_provider);
	return bResult;
}

/**
 * Resolves the value of the cell specified by CellTag and stores it in the output parameter.
 *
 * @param	FieldName		the name of the field the desired cell tags are associated with.  Used for cases where a single data provider
 *							instance provides element cells for multiple collection data fields.
 * @param	CellTag			the tag for the element cell to resolve the value for
 * @param	ListIndex		the UIList's item index for the element that contains this cell.  Useful for data providers which
 *							do not provide unique UIListElement objects for each element.
 * @param	out_FieldValue	receives the resolved value for the property specified.
 *							@see GetDataStoreValue for additional notes
 * @param	ArrayIndex		optional array index for use with cell tags that represent data collections.  Corresponds to the
 *							ArrayIndex of the collection that this cell is bound to, or INDEX_NONE if CellTag does not correspond
 *							to a data collection.
 */
event bool GetCellFieldValue( name FieldName, name CellTag, int ListIndex, out UIProviderFieldValue out_FieldValue, optional int ArrayIndex=INDEX_NONE )
{
	local bool bResult;

	bResult = false;

	`log(`location @ `showvar(FieldName),`debug_resourcecombo_provider);
	return bResult;
}


/**
 * Clears all references in this data provider.  Called when the owning data store is unregistered.
 */
function ClearProviderReferences()
{
	StaticDataProvider = None;
	ProfileProvider = None;
}

/**
 * Utility function for replacing the value of a data field's provider reference with a different value.
 *
 * @param	out_Fields				the list of fields containing the data field to replace the value for
 * @param	TargetFieldTag			the tag for the field whose value should be replaced.
 * @param	ReplacementProvider		the provider to set as the new value for the field being changed
 *
 * @return	TRUE if the field's value was successfully changed.
 */
function bool ReplaceProviderValue( out array<UIDataProviderField> out_Fields, name TargetFieldTag, UIDataProvider ReplacementProvider )
{
	local int FieldIndex;
	local bool bResult;

	for ( FieldIndex = 0; FieldIndex < out_Fields.Length; FieldIndex++ )
	{
		if ( out_Fields[FieldIndex].FieldTag == TargetFieldTag )
		{
			if ( out_Fields[FieldIndex].FieldType == DATATYPE_Provider )
			{
				out_Fields[FieldIndex].FieldProviders[0] = ReplacementProvider;
				bResult = true;
			}

			break;
		}
	}

	return bResult;
}


/**
 * Utility function for replacing the value of a data field's provider collection with a different set of providers.
 *
 * @param	out_Fields				the list of fields containing the data field to replace the value for
 * @param	TargetFieldTag			the tag for the field whose value should be replaced.
 * @param	ReplacementProviders	the collection of provider to set as the new value for the field being changed
 *
 * @return	TRUE if the field's value was successfully changed.
 */
function bool ReplaceProviderCollection( out array<UIDataProviderField> out_Fields, name TargetFieldTag, const out array<UIDataProvider> ReplacementProviders )
{
	local int FieldIndex;
	local bool bResult;

	for ( FieldIndex = 0; FieldIndex < out_Fields.Length; FieldIndex++ )
	{
		if ( out_Fields[FieldIndex].FieldTag == TargetFieldTag )
		{
			if ( out_Fields[FieldIndex].FieldType == DATATYPE_ProviderCollection )
			{
				out_Fields[FieldIndex].FieldProviders = ReplacementProviders;
				bResult = true;
			}

			break;
		}
	}

	return bResult;
}

DefaultProperties
{

}
