package cz.cuni.pogamut.posh.palette;

import cz.cuni.amis.pogamut.sposh.PoshTreeEvent;
import cz.cuni.amis.pogamut.sposh.elements.Competence;
import cz.cuni.amis.pogamut.sposh.elements.CompetenceElement;
import cz.cuni.amis.pogamut.sposh.elements.CompetencePriorityElement;
import cz.cuni.amis.pogamut.sposh.elements.DriveCollection;
import cz.cuni.amis.pogamut.sposh.elements.DriveElement;
import cz.cuni.amis.pogamut.sposh.elements.DrivePriorityElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.amis.pogamut.sposh.elements.PoshElementListener;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.amis.pogamut.sposh.elements.Sense;
import cz.cuni.pogamut.posh.palette.external.BehaviourInterfaceBuilder;
import java.beans.PropertyChangeEvent;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import org.openide.nodes.Children;
import org.openide.util.Lookup;

/**
 * Whenever new Sense is created, this class is called and new node is added
 * using put(Sense, Node). Similar for deletion.
 * 
 * Beware of eqal nodes (different instances with same content), we want only 
 * one node for every set of equal nodes.
 * 
 * @author Honza
 */
public class SensesChildren extends Children.SortedMap<Sense> implements PoshElementListener {

    private List<String> undefinedSenses = new ArrayList<String>();
    private List<Sense> externalSenses = new ArrayList<Sense>();
    private List<String> oldSenses = new ArrayList<String>();

    private Lookup lookup;
    
    public SensesChildren(Lookup lookup) {
        this.lookup = lookup;
    }

    @Override
    public void nodeChanged(PoshTreeEvent event, PoshElement child) {
        refresh(child.getRootNode());
    }

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        refresh(((PoshElement) evt.getSource()).getRootNode());
    }

    public synchronized void refresh(PoshPlan plan) {
        // Get all senses
        List<Sense> senses = getAllSenses(plan);
        // Get their unique names
        List<String> senseNames = new ArrayList<String>();

        for (Sense sense : senses) {
            String senseName = sense.getSenseName();
            if (!senseNames.contains(senseName)) {
                senseNames.add(senseName);
            }
        }
        // Remove external senses
        removeDuplicateSenses(senseNames, externalSenses);


        oldSenses.addAll(undefinedSenses);
        oldSenses.removeAll(senseNames);
        removeDuplicateSenses(oldSenses, externalSenses);

        // Sort them.
        Collections.sort(senseNames);

        // Remove old nodes
        for (Sense sense : this.nodes.keySet().toArray(new Sense[0])) {
            remove(sense);
        }
        undefinedSenses.clear();

        // Add new ones
        for (String senseName : senseNames) {
            Sense sense = new Sense(senseName);
            this.put(sense, new SenseNode(sense));
            undefinedSenses.add(senseName);
        }
        // add old senses
        for (String oldSenseName : oldSenses) {
            Sense os = new Sense(oldSenseName);
            this.put(os, new OldSenseNode(os));
        }

        // Add external senses
        for (Sense sense  : externalSenses) {
            this.put(sense, new ExternalSenseNode(sense));
        }

        // refresh
        this.refresh();
        new Thread(new BehaviourInterfaceBuilder(lookup)).start();
    }


    private List<Sense> getAllSenses(PoshPlan plan) {
        List<Sense> senses = new LinkedList<Sense>();

        // Add senses from goal of dc
        DriveCollection dc = plan.getDriveCollection();
        if (dc.getGoal() != null) {
            senses.addAll(dc.getGoal().getSenses());
        }
        // Add senses from triggers of dc
        for (DrivePriorityElement dpe : dc.getPriorityElements()) {
            for (DriveElement de : dpe.getDriveElements()) {
                senses.addAll(de.getTriggers().getSenses());
            }
        }


        List<Competence> competences = plan.getCompetences();

        // Add all senses from each competence
        for (Competence competence : competences) {
            // First add all senses from goal.
            if (competence.getGoal() != null) {
                senses.addAll(competence.getGoal().getSenses());
            }
            // And then add all senses from its priority elements
            for (CompetencePriorityElement cpe : competence.getPriorityElements()) {
                for (CompetenceElement ce : cpe.getElements()) {
                    senses.addAll(ce.getTriggerSenses());
                }
            }
        }
        return senses;
    }

    public synchronized void setExternalSenses(List<Method> externalMethods) {
        // Take old external actions and remove them from category
        removeSenses(externalSenses);
        externalSenses.clear();

        // Add new external actions
        for (Method m : externalMethods) {
            Sense sense = new Sense(m.getName());
            externalSenses.add(sense);
            this.put(sense, new ExternalSenseNode(sense));
        }
    }

    private void removeSenses(List<Sense> list) {
        for (Sense sense : list) {
            remove(sense);
        }
    }

    /**
     * remove all external senses from action names
     * @param senseNames
     * @param tobeRemoved
     */
    private void removeDuplicateSenses(List<String> senseNames, List<Sense> tobeRemoved) {
        for (String senseName : senseNames.toArray(new String[0])) {
            for (Sense sense : tobeRemoved) {
                if (sense.getSenseName().equals(senseName)) {
                    senseNames.remove(sense.getSenseName());
                }
            }
        }
    }

    /**
     * Get list of actions that are used in the posh tree, but
     * are not defined in some external behaviour library.
     * @return unmodifiable list of such actions
     */
    public synchronized List<String> getUndefinedSenses() {
        return Collections.unmodifiableList(undefinedSenses);
    }

    /**
     * Remove old sense from list of old senses and remove associated node
     * @param oldSense key sense we are going to remove, has to be in oldSenses
     */
    synchronized void removeOldSense(Sense oldSense) {
        oldSenses.remove(oldSense.getSenseName());
        remove(oldSense);
    }

}
