package cz.cuni.amis.dash;

import cz.cuni.amis.pogamut.sposh.dbg.engine.IDebugEngineListener;
import cz.cuni.amis.pogamut.sposh.dbg.lap.LapBreakpoint;
import cz.cuni.amis.pogamut.sposh.elements.LapPath;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.pogamut.posh.explorer.*;
import cz.cuni.pogamut.shed.presenter.IPresenterFactory;
import cz.cuni.pogamut.shed.widget.LapSceneFactory;
import cz.cuni.pogamut.shed.widget.ShedScene;
import java.awt.BorderLayout;
import java.awt.Container;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import javax.swing.*;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ui.OpenProjects;
import org.openide.windows.IOProvider;
import org.openide.windows.OutputWriter;
import org.openide.windows.TopComponent;

/**
 *
 * @author Honza
 */
public class DashWindow extends TopComponent implements IDebugEngineListener {

    private EngineData engineData;
    private final YaposhEngine engine;
    private IPresenterFactory presenter;
    private ShedScene dashScene;
    /**
     * Timestamp when was window last notified about finished evaluation.
     */
    private long lastEvaluationFinished = 0;
    private static int REDRAW_INTERVAL_MS = 50;

    public DashWindow(YaposhEngine engine, String name) {
        this.engine = engine;

        this.setDisplayName(name);
        this.setLayout(new BorderLayout());
        this.setContent(new JScrollPane(new InfoPanel("Waiting to be connected to the engine")));
    }

    @Override
    public void connected() {
        this.setContent(new JScrollPane(new InfoPanel("Connected to the engine, waiting for the plan.")));
    }

    @Override
    public void planRecieved(String name, PoshPlan plan) {
        setDisplayName(name);
        ShedScene scene = new ShedScene(plan);
        engineData = new EngineData(plan);
        presenter = new DashPresenterFactory(engine, engineData, scene, scene.getPresenter());
        dashScene = LapSceneFactory.createDashScene(plan, scene, presenter);
        setContent(new JScrollPane(dashScene.createView()));

        Project[] openProjects = OpenProjects.getDefault().getOpenProjects();
        Set<Project> openProjectsSet = new HashSet<Project>(Arrays.asList(openProjects));
        {
            ClassCrawler actionCrawler = ClassCrawlerFactory.createActionCrawler(openProjectsSet);
            CrawlerListener<PrimitiveData> actionNameCrawler = new NameMapCrawler(scene.getPresenter());
            actionCrawler.addListener(actionNameCrawler);
            actionCrawler.crawl();
        }
        {
            ClassCrawler senseCrawler = ClassCrawlerFactory.createSenseCrawler(openProjectsSet);
            CrawlerListener<PrimitiveData> senseNameCrawler = new NameMapCrawler(scene.getPresenter());
            senseCrawler.addListener(senseNameCrawler);
            senseCrawler.crawl();
        }
    }

    @Override
    public void evaluationReached() {
        // Do nothing, handled by individual presenters
        long currentTime = System.currentTimeMillis();
        long elapsedTime = currentTime - lastEvaluationFinished;
        if (dashScene != null && elapsedTime > REDRAW_INTERVAL_MS) {
            IOProvider.getDefault().getIO("Dash", false).getOut().println("Repaint elapsed " + elapsedTime);
            dashScene.validate();
            dashScene.repaint();
            dashScene.getView().repaint();
        }
    }

    @Override
    public void pathReached(LapPath path) {
        // We don't redraw, it would be too much.
    }

    @Override
    public void evaluationFinished() {
        lastEvaluationFinished = System.currentTimeMillis();
    }

    @Override
    public void breakpointAdded(LapBreakpoint breakpoint) {
        // Do nothing, handled by individual presenters
    }

    @Override
    public void breakpointRemoved(LapBreakpoint breakpoint) {
        // Do nothing, handled by individual presenters
    }

    @Override
    public void disconnected(String message, boolean error) {
        if (error) {
            this.add(new JScrollPane(new InfoPanel("Error: " + message)), BorderLayout.NORTH);
        } else {
            this.add(new JScrollPane(new InfoPanel(message)), BorderLayout.NORTH);
        }
    }

    /**
     * Make sure to clean up after window will be closed.
     */
    @Override
    protected void componentClosed() {
        engine.disconnect("The debugging window is being closed.", false);
    }

    @Override
    public int getPersistenceType() {
        return PERSISTENCE_NEVER;
    }

    private void setContent(Container container) {
        this.removeAll();
        this.add(container, BorderLayout.CENTER);
        this.revalidate();
    }
}

class InfoPanel extends JPanel {

    private final JLabel label;

    public InfoPanel(String message) {
        label = new JLabel(message, SwingConstants.CENTER);
        setLayout(new BorderLayout());
        add(label, BorderLayout.CENTER);
    }
}