/*
 * Copyright (C) 2013 Martin Cerny
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package cz.cuni.amis.experiments.impl.metrics;

import cz.cuni.amis.experiments.ILoggingHeaders;
import cz.cuni.amis.experiments.IMetric;
import cz.cuni.amis.experiments.impl.LoggingHeaders;
import cz.cuni.amis.utils.NullCheck;
import java.util.ArrayList;
import java.util.List;

/**
 * This class is not thread-safe.
 * @author Martin Cerny
 */
public class MetricCollection {
    protected List<IMetric> metrics;
    
    private boolean headersReturned = false;
    private ILoggingHeaders headers;
    
    public MetricCollection(){
        metrics = new ArrayList<IMetric>();
    }
    
    /**
     * Add a metric. No metric can be added after headers are first read!
     * @param metric 
     */
    public void addMetric(IMetric metric){
        if(headersReturned){
            throw new IllegalStateException("Cannot add metrics after headers have been returned");
        }
        NullCheck.check(metric, "metric");
        metrics.add(metric);
    }
    
    public boolean isEmpty(){
        return metrics.isEmpty();
    }
    
    public ILoggingHeaders getHeaders(){
        if(!headersReturned){
            headersReturned = true;
            List<String> columnNames = new ArrayList<String>(metrics.size());
            for(IMetric metric : metrics){
                columnNames.add(metric.getName());
            }
            headers = new LoggingHeaders(columnNames);
        }
        return headers;
    }
    
    public List<Object> getValues(){
        List<Object> values = new ArrayList<Object>(metrics.size());
        for(IMetric metric : metrics){
            values.add(metric.getValue());
        }        
        return values;
    }
    
    public void reset(){
        for(IMetric metric: metrics){
            metric.reset();
        }
    }
    
    public void stopMeasurement(){
        for(IMetric metric: metrics){
            metric.stopMeasurement();
        }        
    }
}
