/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.experiments.impl;

import cz.cuni.amis.experiments.*;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.log4j.Logger;

/**
 * Helper class for all experiment suite runners that need only one thread
 * @author Martin Cerny
 */
public class SingleThreadExperimentSuiteRunner<EXPERIMENT_TYPE extends IExperiment> extends AbstractExperimentSuiteRunner<EXPERIMENT_TYPE > {
      
    private final Logger logger = Logger.getLogger(SingleThreadExperimentSuiteRunner.class);

    private IExperimentRunner<EXPERIMENT_TYPE> runner;

    
    public SingleThreadExperimentSuiteRunner(IExperimentRunner<EXPERIMENT_TYPE> runner, ILogCentral logCentral) {
        super(logCentral);
        this.runner = runner;
    }
    
    @Override
    protected List<IExperimentRunner<EXPERIMENT_TYPE>> getAllRunners() {
        return Collections.singletonList(runner);
    }
    
    
    
    @Override
    protected void runExperimentSuiteInternal(IExperimentSuite<EXPERIMENT_TYPE > suite, int startExperiment, int stopAfter, IExperimentFilter<EXPERIMENT_TYPE> filter) {
        logger.info("Starting suite " + suite.getName() + " start experiment: " + startExperiment + " stopAfter:" + stopAfter + " filter:"  + filter) ;
        logger.info("Running beforeAllExperiments()");
        suite.beforeAllExperiments();
        
        int numExperimentsRun = 0;

        
        ILoggingHeaders additionalLoggingHeaders = new LoggingHeaders("SuiteName", "ID");
        
        List<EXPERIMENT_TYPE> experiments = suite.getExperiments();
        for(int i = startExperiment; i < experiments.size(); i++){
            if(stopAfter > 0 && numExperimentsRun >= stopAfter){
                break;
            }
            EXPERIMENT_TYPE experiment = experiments.get(i);
            if(filter == null || filter.accept(experiment)){
                suite.beforeExperiment(experiment);
                logger.info("Running experiment no.: " + i + " : " + experiment.getDescription());
                
                List<Object> additionalLoggingValues = Arrays.asList(new Object[] {suite.getName(), i});                
                ILogCentral decoratedCentral = new DecoratedLogCentral(logCentral, additionalLoggingHeaders, additionalLoggingValues);
                
                runSingleExperiment(runner, experiment, decoratedCentral);                
                suite.afterExperiment(experiment);
                System.gc();
                numExperimentsRun++;
            } else if(logger.isDebugEnabled()){
                logger.debug("Experiment no." + i + " was filtered out.");
            }
        }
        
        logger.info("All exepriments done, running afterAllExperiments().");
        suite.afterAllExperiments();
        logger.info("Suite " + suite.getName() + " finished");
    }
    
}
