/*
 * Copyright (C) 2011 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.experiments.impl;

import cz.cuni.amis.experiments.ILoggingHeaders;
import cz.cuni.amis.experiments.IValueConverter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

/**
 * A simple helper class for creating result writers that write to CSV files (to be read by R or something)
 * @author Martin Cerny
 */
public class CSVLoggingOutput extends AbstractLoggingOutput {

    private File targetFile;

    private FileWriter writer;
    private boolean append;
    
    protected String fieldDelimiter = ",";
    protected String lineDelimiter = "\r\n";
    protected String quote = "\"";
            
    private IValueConverter valueConverter;
    
    private boolean writtenSomethingOnCurrentLine = false;

    public CSVLoggingOutput(File targetFile) {
        this(targetFile, new DefaultValueConverter());
    }

    public CSVLoggingOutput(File targetFile, boolean append) {
        this(targetFile, new DefaultValueConverter(), false);
    }

    public CSVLoggingOutput(File targetFile, IValueConverter valueConverter) {
        this(targetFile, valueConverter, false);
    }
    
    public CSVLoggingOutput(File targetFile, IValueConverter valueConverter, boolean append) {
        this.targetFile = targetFile;
        this.valueConverter = valueConverter;
        this.append = append;
    }
    
        
    @Override
    protected void closeInternal() throws IOException {
        writer.close();
    }

    protected void writeField(String value) throws IOException{
        if(writtenSomethingOnCurrentLine){
            writer.write(fieldDelimiter);
        } else {
            writtenSomethingOnCurrentLine = true;
        }
        //quote is escaped by doubling it
        writer.append(quote).append(value.replace(quote, quote + quote)).append(quote);
    }
    
    protected void endLine() throws IOException {
        writer.write(lineDelimiter);
        writer.flush();
        writtenSomethingOnCurrentLine = false;
    }
    
    @Override
    public void init(ILoggingHeaders headers) throws IOException {
        super.init(headers);        
        writer = new FileWriter(targetFile, append);
        if(!append){
            for(String header: headers.getColumnNames()){
                writeField(header);
            }
        } 
        endLine();
    }

    @Override
    protected void logDataInternal(List<Object> data) throws IOException {
        for(Object field : data){
            writeField(valueConverter.valueToString(field));
        }
        endLine();
    }

    @Override
    protected void flushInternal() throws IOException {
        writer.flush();
    }

    
}
