/*
 * Copyright (C) 2013 Martin Cerny
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package cz.cuni.amis.experiments.impl;

import cz.cuni.amis.experiments.*;
import cz.cuni.amis.utils.flag.Flag;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import org.apache.log4j.Logger;

/**
 *
 * @author Martin Cerny
 */
public abstract class AbstractExperimentRunner<EXPERIMENT_TYPE extends IExperiment> implements IExperimentRunner<EXPERIMENT_TYPE> {

    private final Logger logger = Logger.getLogger(AbstractExperimentRunner.class);
    
    private LoggingHeaders executionStatusHeaders = new LoggingHeaders("startTime","duration", "status");
    
    @Override
    public void shutdown() {        
    }
    
    protected void prepareExperiment(EXPERIMENT_TYPE experiment){
        
    }

    /**
     * Does all the logging related setup and calls {@link #runExperimentInternal(cz.cuni.amis.experiments.IExperiment) }
     * @param experiment
     * @return 
     */
    @Override
    public final EExperimentRunResult runExperiment(EXPERIMENT_TYPE experiment, ILogCentral logCentral) {

        
        ILogCentral parametersDecoratedCentral = new DecoratedLogCentral(logCentral, 
                experiment.getExperimentParametersHeaders(), 
                experiment.getExperimentParameters());

        prepareExperiment(experiment);
        
        /**
         * Setup runtime logging
         */                
        for(ILogDataProvider provider : getAllLogProviders(experiment)){
            //the runtime logging is requested by the provider, so that there are no conflicts
            ILoggingHeaders headers = provider.getRuntimeLoggingHeaders();
            if(headers != null && headers.getColumnCount() > 0) {
                IBareLoggingOutput runtimeLoggingOutput = parametersDecoratedCentral.requestLoggingOutput(provider.getIdentifier(), ELogType.RUNTIME, provider, headers);
                provider.setRuntimeLoggingOutput(runtimeLoggingOutput);                        
            }
        }
        
        ExperimentProcess experimentProcess = new ExperimentProcess(experiment);
        long startTime = System.currentTimeMillis();
        
        new Thread(experimentProcess).start();
        if(experiment.getTimeout() > 0){
            experimentProcess.getFinished().waitFor(experiment.getTimeout(), true);
        } else {
            experimentProcess.getFinished().waitFor(true);
        }

        long endTime = System.currentTimeMillis();
        
        EExperimentRunResult result;
        long duration = endTime - startTime;
        if(experimentProcess.getFinished().getFlag()){
            result = experimentProcess.getResult();
        } else {
            result = EExperimentRunResult.TIMEOUT;
        }
        
        List<Object> executionStatusData = Arrays.asList(new Object[] {startTime, duration, result} );
        
        /**
         * Release asynchronuous logs and log per-experiment data
         */
        for(ILogDataProvider provider : getAllLogProviders(experiment)){
            
            ILoggingHeaders headers = provider.getRuntimeLoggingHeaders();
            if(headers != null && headers.getColumnCount() > 0) {            
                parametersDecoratedCentral.releaseLoggingOutput(provider.getIdentifier(),ELogType.RUNTIME, provider);

            }
            //per experiment output is requested by this, as it makes more sense
            IBareLoggingOutput perExperimentOutput = parametersDecoratedCentral.requestLoggingOutput(provider.getIdentifier(), ELogType.PER_EXPERIMENT, this,
                    executionStatusHeaders, getAdditionalLoggingHeaders(experiment, provider),  provider.getPerExperimentLoggingHeaders());
            perExperimentOutput.logData(executionStatusData, getAdditionalLoggingDataValues(experiment, provider), provider.getPerExperimentLoggingData());
            parametersDecoratedCentral.releaseLoggingOutput(provider.getIdentifier(), ELogType.PER_EXPERIMENT, this);
        }
        
        return result;
    }
    
    protected abstract List<ILogDataProvider> getAllLogProviders(EXPERIMENT_TYPE experiment);
    

    
    /**
     * Allows subclass to further modify logging for managed providers. This method is called AFTER experiment
     * is performed.
     * @param provider
     * @return 
     */
    protected ILoggingHeaders getAdditionalLoggingHeaders(EXPERIMENT_TYPE experiment, ILogDataProvider provider){
        return LoggingHeaders.EMPTY_LOGGING_HEADERS;
    }
    
    /**
     * Allows subclass to further modify logging for managed providers. This method is called AFTER experiment
     * is performed.
     * @param provider
     * @return 
     */
    protected List<Object> getAdditionalLoggingDataValues(EXPERIMENT_TYPE experiment, ILogDataProvider provider){
        return Collections.EMPTY_LIST;
    }
    
    /**
     * Run the experiment internally. Running time, exceptions thrown and timeout do not need to be
     * handled in this method.
     * @param experiment
     * @return the run result
     */
    protected abstract EExperimentRunResult runExperimentInternal(EXPERIMENT_TYPE experiment);
    

    @Override
    public void init() {
    }

    private class ExperimentProcess implements Runnable {
        private EXPERIMENT_TYPE experiment;
        private Flag<Boolean> finished;
        
        private EExperimentRunResult result;

        public ExperimentProcess(EXPERIMENT_TYPE experiment) {
            this.experiment = experiment;
            finished = new Flag<Boolean>(false);
        }

        public Flag<Boolean> getFinished() {
            return finished;
        }

        public EExperimentRunResult getResult() {
            return result;
        }




        
        
        @Override
        public void run() {
            try {
                result = runExperimentInternal(experiment);
            } catch(Exception ex){
                result = EExperimentRunResult.EXCEPTION;
                logger.error("Exception during experiment.", ex);                
            }
            finished.setFlag(true);
        }
        
        
        
        
        
    }
}
