package sk.stuba.fiit.pogamut.jungigation.worldInfo;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.base3d.worldview.IVisionWorldView;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.communication.messages.ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Item;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;
import cz.cuni.amis.pogamut.ut2004.communication.translator.shared.events.MapPointListObtained;
import cz.cuni.amis.utils.maps.HashMapMap;

/**
 * <p>
 * Memory module specialized on items on the map. Includes also informations about players.
 * </p>
 * 
 * @author Juraj 'Loque' Simlovic
 * @author Jimmy
 * @author LuVar
 */
public class SharedItems {
	/**
	 * Retreives list of all items, which includes all known pickups and all visible thrown items.
	 * 
	 * <p>
	 * Note: The returned Map is unmodifiable and self updating throughout time. Once you obtain a specific Map of items
	 * from this module, the Map will get updated based on what happens within the map.
	 * 
	 * @return List of all items. Note: Spawned items are included only.
	 */
	public Map<UnrealId, Item> getAllItems() {
		return Collections.unmodifiableMap(this.items.all);
	}
	
	public Map<UnrealId, Player> getAllPlayers() {
		return Collections.unmodifiableMap(this.players.allPlayers);
	}
	
	public Map<UnrealId, Self> getAllSelfs() {
		return Collections.unmodifiableMap(this.players.allSelfs);
	}
	
	/**
	 * Retrieves list of all items <b>of specific type</b>, which includes all known pickups and all visible thrown
	 * items.
	 * 
	 * <p>
	 * Note: The returned Map is unmodifiable and self updating throughout time. Once you obtain a specific Map of items
	 * from this module, the Map will get updated based on what happens within the map.
	 * 
	 * @return List of all items of specific type. Note: Spawned items are included only.
	 */
	public Map<UnrealId, Item> getAllItems(ItemType type) {
		return Collections.unmodifiableMap(this.items.allCategories.get(type));
	}
	
	/**
	 * Retrieves a specific item from the all items in the map.
	 * <p>
	 * <p>
	 * Once obtained it is self-updating based on what happens in the game.
	 * 
	 * @param id
	 * @return A specific Item be it Spawned or Dropped (Dropped item must be visible though!).
	 */
	public Item getItem(UnrealId id) {
		Item item = this.items.all.get(id);
		if (item == null) {
			item = this.items.visible.get(id);
		}
		return item;
	}
	
	/**
	 * Retrieves a specific item from the all items in the map.
	 * <p>
	 * <p>
	 * Once obtained it is self-updating based on what happens in the game.
	 * 
	 * @param stringUnrealId
	 * @return A specific Item be it Spawned or Dropped (Dropped item must be visible though!).
	 */
	public Item getItem(String stringUnrealId) {
		return getItem(UnrealId.get(stringUnrealId));
	}
	
	/* ======================================================================== */

	/**
	 * Retreives list of all visible items, which includes all visible known pickups and all visible thrown items.
	 * 
	 * <p>
	 * Note: The returned Map is unmodifiable and self updating throughout time. Once you obtain a specific Map of items
	 * from this module, the Map will get updated based on what happens within the map.
	 * 
	 * @return List of all visible items. Note: Spawned items are included only.
	 */
	public Map<UnrealId, Item> getVisibleItems() {
		return Collections.unmodifiableMap(this.items.visible);
	}
	
	/**
	 * Retreives list of all visible items <b> of specific type</b>, which includes all visible known pickups and all
	 * visible thrown items.
	 * 
	 * <p>
	 * Note: The returned Map is unmodifiable and self updating throughout time. Once you obtain a specific Map of items
	 * from this module, the Map will get updated based on what happens within the map.
	 * 
	 * @return List of all visible items of specific type. Note: Spawned items are included only.
	 */
	public Map<UnrealId, Item> getVisibleItems(ItemType type) {
		return Collections.unmodifiableMap(this.items.visibleCategories.get(type));
	}
	
	/**
	 * Retrieves a specific item from the visible items in the map. If item of specified id is not visible returns null.
	 * <p>
	 * <p>
	 * Once obtained it is self-updating based on what happens in the game.
	 * 
	 * @param id
	 * @return A specific Item be it Spawned or Dropped.
	 */
	public Item getVisibleItem(UnrealId id) {
		Item item = this.items.visible.get(id);
		return item;
	}
	
	/**
	 * Retrieves a specific item from the visible items in the map. If item of specified id is not visible returns null.
	 * <p>
	 * <p>
	 * Once obtained it is self-updating based on what happens in the game.
	 * 
	 * @param stringUnrealId
	 * @return A specific Item be it Spawned or Dropped.
	 */
	public Item getVisibleItem(String stringUnrealId) {
		return getVisibleItem(UnrealId.get(stringUnrealId));
	}
	
	/* ======================================================================== */

	/**
	 * Retreives list of all reachable items, which includes all reachable known pickups and all reachable and visible
	 * thrown items.
	 * 
	 * <p>
	 * Note: The returned Map is unmodifiable and self updating throughout time. Once you obtain a specific Map of items
	 * from this module, the Map will get updated based on what happens within the map.
	 * 
	 * @return List of all reachable items. Note: Spawned items are included only.
	 */
	public Map<UnrealId, Item> getReachableItems() {
		return Collections.unmodifiableMap(this.items.reachable);
	}
	
	/**
	 * Retreives list of all reachable items <b>of specific type</b>, which includes all reachable known pickups and all
	 * reachable and visible thrown items.
	 * 
	 * <p>
	 * Note: The returned Map is unmodifiable and self updating throughout time. Once you obtain a specific Map of items
	 * from this module, the Map will get updated based on what happens within the map.
	 * 
	 * @return List of all reachable items of specific type. Note: Spawned items are included only.
	 */
	public Map<UnrealId, Item> getReachableItems(ItemType type) {
		return Collections.unmodifiableMap(this.items.reachableCategories.get(type));
	}
	
	/**
	 * Retrieves a specific item from the all items in the map that is currently reachable. If item of specified item is
	 * not reachable returns null.
	 * <p>
	 * <p>
	 * Once obtained it is self-updating based on what happens in the game.
	 * 
	 * @param id
	 * @return A specific Item be it Spawned or Dropped (Dropped item must be visible though!).
	 */
	public Item getReachableItem(UnrealId id) {
		Item item = this.items.reachable.get(id);
		if (item == null) {
			item = this.items.visible.get(id);
			if (!item.isReachable()) {
				return null;
			}
		}
		return item;
	}
	
	/**
	 * Retrieves a specific item from the all items in the map that is currently reachable. If item of specified item is
	 * not reachable returns null.
	 * <p>
	 * <p>
	 * Once obtained it is self-updating based on what happens in the game.
	 * 
	 * @param stringUnrealId
	 * @return A specific Item be it Spawned or Dropped (Dropped item must be visible though!).
	 */
	public Item getReachableItem(String stringUnrealId) {
		return getReachableItem(UnrealId.get(stringUnrealId));
	}
	
	/* ======================================================================== */

	/**
	 * Retreives list of all known item pickup points.
	 * 
	 * <p>
	 * Note: The returned Map is unmodifiable and self updating throughout time. Once you obtain a specific Map of items
	 * from this module, the Map will get updated based on what happens within the map.
	 * 
	 * @return List of all items. Note: Empty pickups are included as well.
	 * 
	 * @see isPickupSpawned(Item)
	 */
	public Map<UnrealId, Item> getKnownPickups() {
		return Collections.unmodifiableMap(this.items.known);
	}
	
	/**
	 * Retreives list of all known item pickup points <b>of specific type</b>.
	 * 
	 * <p>
	 * Note: The returned Map is unmodifiable and self updating throughout time. Once you obtain a specific Map of items
	 * from this module, the Map will get updated based on what happens within the map.
	 * 
	 * @return List of all items of specific type. Note: Empty pickups are included as well.
	 * 
	 * @see isPickupSpawned(Item)
	 */
	public Map<UnrealId, Item> getKnownPickups(ItemType type) {
		return Collections.unmodifiableMap(this.items.knownCategories.get(type));
	}
	
	/**
	 * Retrieves a specific pickup point.
	 * <p>
	 * <p>
	 * Once obtained it is self-updating based on what happens in the game.
	 * 
	 * @param id
	 * @return A specific Item be it Spawned or Dropped (Dropped item must be visible though!).
	 */
	public Item getKnownPickup(UnrealId id) {
		return this.items.known.get(id);
	}
	
	/**
	 * Retrieves a specific pickup point.
	 * <p>
	 * <p>
	 * Once obtained it is self-updating based on what happens in the game.
	 * 
	 * @param stringUnrealId
	 * @return A specific Item be it Spawned or Dropped (Dropped item must be visible though!).
	 */
	public Item getKnownPickup(String stringUnrealId) {
		return getKnownPickup(UnrealId.get(stringUnrealId));
	}
	
	/* ======================================================================== */

	/**
	 * Tells, whether the given pickup point contains a spawned item.
	 * 
	 * @param item Item, for which its pickup point is to be examined.
	 * @return True, if the item is spawned; false if the pickup is empty.
	 * 
	 * @see getKnownPickups(boolean,boolean)
	 */
	public boolean isPickupSpawned(Item item) {
		// FIXME[js]: implement when available
		throw new UnsupportedOperationException("Not supported yet");
	}
	
	/* ======================================================================== */

	/**
	 * Maps of items of specific type.
	 */
	private class ItemMaps {
		/** Map of all items (known and thrown). */
		private HashMap<UnrealId, Item> all = new HashMap<UnrealId, Item>();
		/** Map of visible items of the specific type. */
		private HashMap<UnrealId, Item> visible = new HashMap<UnrealId, Item>();
		/** Map of visible items of the specific type. */
		private HashMap<UnrealId, Item> reachable = new HashMap<UnrealId, Item>();
		/** Map of all known items of the specific type. */
		private HashMap<UnrealId, Item> known = new HashMap<UnrealId, Item>();
		/** Map of all items (known and thrown) of specific categories. */
		private HashMapMap<ItemType, UnrealId, Item> allCategories = new HashMapMap<ItemType, UnrealId, Item>();
		/** Map of visible items of the specific type. */
		private HashMapMap<ItemType, UnrealId, Item> visibleCategories = new HashMapMap<ItemType, UnrealId, Item>();
		/** Map of visible items of the specific type. */
		private HashMapMap<ItemType, UnrealId, Item> reachableCategories = new HashMapMap<ItemType, UnrealId, Item>();
		/** Map of all known items of the specific type. */
		private HashMapMap<ItemType, UnrealId, Item> knownCategories = new HashMapMap<ItemType, UnrealId, Item>();
		
		/** Map of all items and the time when they were last seen. */
		
		/**
		 * Processes events.
		 * 
		 * @param item Item to process.
		 */
		private void notify(Item item) {
			UnrealId uid = item.getId();
			
			// be sure to be within all
			if (!this.all.containsKey(uid)) {
				this.all.put(uid, item);
				this.allCategories.put(item.getType(), item.getId(), item);
			}
			
			// previous visibility
			boolean wasVisible = this.visible.containsKey(uid);
			boolean isVisible = item.isVisible();
			
			// refresh visible
			if (isVisible && !wasVisible) {
				// add to visibles
				this.visible.put(uid, item);
				this.visibleCategories.put(item.getType(), item.getId(), item);
			} else if (!isVisible && wasVisible) {
				// remove from visibles
				this.visible.remove(uid);
				this.visibleCategories.remove(item.getType(), item.getId());
			}
			
			// remove non-visible thrown items
			if (!isVisible && item.isDropped()) {
				this.all.remove(uid);
				this.allCategories.remove(item.getType(), item.getId());
			}
			
			// previous reachability
			boolean wasReachable = this.reachable.containsKey(uid);
			boolean isReachable = item.isReachable();
			
			// refresh reachable
			if (isReachable && !wasReachable) {
				// add to reachables
				this.reachable.put(uid, item);
				this.reachableCategories.put(item.getType(), item.getId(), item);
			} else if (!isReachable && wasReachable) {
				// remove from reachables
				this.reachable.remove(uid);
				this.reachableCategories.remove(item.getType(), item.getId());
			}
		}
		
		/**
		 * Processes events.
		 * 
		 * @param items Map of known items to process.
		 */
		private void notify(Map<UnrealId, Item> items) {
			// register all known items
			this.known.putAll(items);
			for (Item item : items.values()) {
				this.knownCategories.put(item.getType(), item.getId(), item);
			}
		}
	}// end of class ItemMaps
	
	/**
	 * Maps of players.
	 */
	private class PlayerMaps {
		private HashMap<UnrealId, Player> allPlayers = new HashMap<UnrealId, Player>();
		private HashMap<UnrealId, Self> allSelfs = new HashMap<UnrealId, Self>();
		
		private void notify(Player player) {
			UnrealId uid = player.getId();
			if (!this.allPlayers.containsKey(uid)) {
				this.allPlayers.put(uid, player);
			}
		}
		
		private void notify(Self player) {
			UnrealId uid = player.getId();
			if (!this.allSelfs.containsKey(uid)) {
				this.allSelfs.put(uid, player);
			}
		}
	}// end of class PlayerMaps
	
	/** Maps of all items. */
	private ItemMaps items = new ItemMaps();
	
	private PlayerMaps players = new PlayerMaps();
	
	/* ======================================================================== */

	protected class PlayersListener implements IWorldObjectListener<Player> {
		
		public PlayersListener(IWorldView worldView) {
			worldView.addObjectListener(Player.class, WorldObjectUpdatedEvent.class, this);
		}
		
		@Override
		public void notify(IWorldObjectEvent<Player> event) {
			SharedItems.this.players.notify(event.getObject());
		}
	}
	
	protected class SelfListener implements IWorldObjectListener<Self> {
		
		public SelfListener(IWorldView worldView) {
			worldView.addObjectListener(Self.class, WorldObjectUpdatedEvent.class, this);
		}
		
		@Override
		public void notify(IWorldObjectEvent<Self> event) {
			SharedItems.this.players.notify(event.getObject());
		}
	}
	
	protected class ItemsListener implements IWorldObjectListener<Item> {
		
		public ItemsListener(IWorldView worldView) {
			worldView.addObjectListener(Item.class, WorldObjectUpdatedEvent.class, this);
		}
		
		public void notify(IWorldObjectEvent<Item> event) {
			SharedItems.this.items.notify(event.getObject());
		}
		
	}
	
	protected List<ItemsListener> itemsListener = new ArrayList<ItemsListener>();
	protected List<PlayersListener> playersListener = new ArrayList<PlayersListener>();
	protected List<SelfListener> selfListener = new ArrayList<SelfListener>();
	
	/* ======================================================================== */

	/**
	 * MapPointsListObtained listener.
	 */
	protected class MapPointsListener implements IWorldEventListener<MapPointListObtained> {
		private final IWorldView worldView;
		
		/**
		 * Constructor. Registers itself on the given WorldView object.
		 * 
		 * @param worldView WorldView object to listen to.
		 */
		public MapPointsListener(IWorldView worldView) {
			this.worldView = worldView;
			this.worldView.addEventListener(MapPointListObtained.class, this);
		}
		
		@Override
		public void notify(MapPointListObtained event) {
			SharedItems.this.items.notify(event.getItems());
			this.worldView.removeEventListener(MapPointListObtained.class, this);
		}
		
	}
	
	/** MapPointsListObtained listener */
	protected List<MapPointsListener> mapPointsListener = new ArrayList<MapPointsListener>();
	
	/* ======================================================================== */

	/**
	 * Constructor. Setups the memory module based on bot's world view.
	 * 
	 * @param worldView worldView of agent to register to for event receiving about world
	 */
	public SharedItems(IVisionWorldView worldView) {
		this(worldView, null);
	}
	
	/**
	 * Constructor. Setups the memory module based on bot's world view.
	 * 
	 * @param worldView worldView of agent to register to for event receiving about world
	 * @param log Logger to be used for logging runtime/debug info, if null is provided the module creates its own
	 *            logger
	 */
	public SharedItems(IWorldView worldView, Logger log) {
		this.worldView.add(worldView);
		// create listeners
		this.itemsListener.add(new ItemsListener(worldView));
		this.mapPointsListener.add(new MapPointsListener(worldView));
		this.playersListener.add(new PlayersListener(worldView));
		this.selfListener.add(new SelfListener(worldView));
	}
	
	protected final Set<IWorldView> worldView = Collections.synchronizedSet(new HashSet<IWorldView>());
	
	/**
	 * <p>
	 * Gets all world views associated with this {@link SharedItems} instance.
	 * </p>
	 * 
	 * @return
	 */
	public Set<IWorldView> getWorldView() {
		return this.worldView;
	}
	
	/**
	 * <p>
	 * If you want to use Items class with more than one bot. This will allow shared instance along more than one bot.
	 * </p>
	 * <p>
	 * NOTE: This method checks for duplicates of added worldView.
	 * </p>
	 * 
	 * @param worldView worldView of agent to register to for event receiving about world
	 */
	public void addMoreWorldView(IWorldView worldView) {
		if (this.worldView.contains(worldView)) {
			return;
		}
		this.worldView.add(worldView);
		// create listeners
		this.itemsListener.add(new ItemsListener(worldView));
		this.mapPointsListener.add(new MapPointsListener(worldView));
		this.playersListener.add(new PlayersListener(worldView));
		this.selfListener.add(new SelfListener(worldView));
	}
	
	/**
	 * <p>
	 * When player left team, or so...
	 * </p>
	 * 
	 * @param worldView
	 */
	public void removeWorldView(IWorldView worldView) {
		throw new UnsupportedOperationException("Not supported yet");
		/*
		 * if (this.worldView.contains(worldView)) { return; } this.worldView.add(worldView); // create listeners
		 * itemsListener.add(new ItemsListener(worldView)); mapPointsListener.add(new MapPointsListener(worldView));
		 * playersListener.add(new PlayersListener(worldView)); selfListener.add(new SelfListener(worldView));
		 */
	}
}
