package sk.stuba.fiit.pogamut.jungigation.objects;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import sk.stuba.fiit.pogamut.jungigation.transformers.GraphMetadataTransformer;

import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import edu.uci.ics.jung.graph.DirectedSparseGraph;

/**
 * <p>
 * I hope fully thread safe {@link DirectedSparseGraph} with little effort.
 * </p>
 * <p>
 * This class also provides some helpful methods for "translating" {@link NavPoint} into {@link MyVertice}. Have look at
 * {@link #getVerticeFromLocation(Location)}.
 * </p>
 * 
 * @author LuVar
 * 
 * @see GraphMetadataTransformer
 */
@SuppressWarnings("serial")
public class NavigationGraphSynchronized extends DirectedSparseGraph<MyVertice, MyEdge> {
    private double learningTime;
    private long stucksCount;
    private long respawnCount;
    private Map<String, String> additionalProperties = null;

    public static final String PROPERTY_LearningTime = "LearningTime";
    public static final String PROPERTY_StucksCount = "StucksCount";
    public static final String PROPERTY_RespawnCount = "RespawnCount";

    /**
     * <p>
     * Do not use this constructor, please.
     * </p>
     */
    public NavigationGraphSynchronized() {
	super();
	this.vertices = Collections.synchronizedMap(this.vertices);
	this.edges = Collections.synchronizedMap(this.edges);
    }

    /**
     * <p>
     * Standard constructor for navigation graph. This class use synchronized maps to be thread safe.
     * </p>
     * 
     * @param learningTime
     * @param stucksCount
     * @param respawnCount
     */
    public NavigationGraphSynchronized(double learningTime, long stucksCount, long respawnCount) {
	this();
	this.learningTime = learningTime;
	this.stucksCount = stucksCount;
	this.respawnCount = respawnCount;
    }
    
    public NavigationGraphSynchronized(double learningTime, long stucksCount, long respawnCount, Map<String, String> additionalProperties) {
	this(learningTime, stucksCount, respawnCount);
	this.additionalProperties = additionalProperties;
    }

    /**
     * <p>
     * Method will iterate through all vertices and will find vertice which has same location as given location.
     * Comparison is made by calling {@link Location#equals(Location)} method. If no vertex location
     * is same as given one, runtime exception is thrown.
     * </p>
     * 
     * @param navPoint
     * @return
     */
    public MyVertice getVerticeFromLocation(Location navPoint) {
	for (MyVertice vert : this.getVertices()) {
	    if (vert.getLocation().equals(navPoint)) {
		return vert;
	    }
	}
	throw new RuntimeException("Vertice not found. Given location:" + navPoint);
    }

    /**
     * <p>
     * Returns rounded time to two digits.
     * </p>
     * <p>
     * Gets learning time. Learning time is set in constructor {@link #NavigationGraphSynchronized(double, long, long)} and than
     * can be increased by calling {@link #addLearningTime(double)} method. It is used when bot is learning map.
     * </p>
     * 
     * @return	current stucks count
     */
    public double getLearningTime() {
	double tmp = Math.round(this.learningTime * 100.0) / 100.0;
	return tmp;
    }

    /**
     * <p>
     * Gets stucks count. Stucks count is set in constructor {@link #NavigationGraphSynchronized(double, long, long)} and than
     * can be increased by calling {@link #increaseStucksCount()} method.
     * </p>
     * 
     * @return	current stucks count
     */
    public long getStucksCount() {
	return this.stucksCount;
    }

    /**
     * <p>
     * Gets respawns count. Respawns count is set in constructor {@link #NavigationGraphSynchronized(double, long, long)} and than
     * can be increased by calling {@link #increaseRespawnCount()} method.
     * </p>
     * 
     * @return	current respawns count
     */
    public long getRespawnCount() {
	return this.respawnCount;
    }

    public void increaseStucksCount() {
	this.stucksCount++;
    }

    public void increaseRespawnCount() {
	this.respawnCount++;
    }

    public void addLearningTime(double time) {
	this.learningTime += time;
    }

    public MyEdge getEdgeBetweenVertices(MyVertice from, MyVertice to) {
	MyEdge navrat = this.findEdge(from, to);
	if (navrat != null) {
	    return navrat;
	}
	throw new RuntimeException("No edge found between vertice " + from + " and vertice " + to + "!");
    }
    
    /**
     * <p>
     * Return internal map instance which holds additional properties for navigation graph. RespawnCount, LearningTime and StucksCount are not
     * stored in this map. Do not add there any of {@value #PROPERTY_LearningTime}, {@value #PROPERTY_RespawnCount} or {@value #PROPERTY_StucksCount}
     * mapping. Also do not remove what is not your.
     * </p>
     * <p>
     * All properties stored in this map will be stored to graphML xml file when serializing navigation graph into graphML file.
     * </p>
     * 
     * @return	internal map holding additional properties
     */
    public Map<String, String> getAdditionalProperties() {
	if(this.additionalProperties == null) {
	    this.additionalProperties = new HashMap<String, String>();
	}
	return this.additionalProperties;
    }
    
    /**
     * <p>
     * Puts given value for given key to {@link #additionalProperties} map. This map can be also obtained directly from
     * {@link #getAdditionalProperties()} method.
     * </p>
     * 
     * @param key	desired key to be associated with value
     * @param value	desired value to be associated with key
     */
    public void addAdditionalParameter(String key, String value) {
	this.getAdditionalProperties().put(key, value);
    }
    
    /**
     * <p>
     * Retrieves value for given key from {@link #additionalProperties} map. This map can be also obtained directly from
     * {@link #getAdditionalProperties()} method.
     * </p>
     * 
     * @param key	desired key
     * @return		value for given key
     */
    public String getAdditionalParameter(String key) {
	return this.getAdditionalProperties().get(key);
    }
}
