package jung.myalghoritm;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.apache.commons.collections15.Transformer;
import org.apache.log4j.Logger;

import edu.uci.ics.jung.graph.Graph;

/**
 * <p>
 * This pseudo algorithm is able to find shortest path in graphs with negative edges. Even graphs
 * with negative cycles can be used. Algorithm is dedicated only to graphs with few negative edges.
 * </p>
 * <p>
 * To find shortest path, algorithm in first step search for negative edges. In second step it
 * search for shortest path from start to end through some midpoints. Midpoints are chosen from
 * set of negative edges. Simple parts of routes are planned with Dijkstra algorithm. After
 * computation of path, negative cost of edge is added to whole cost of path.
 * </p>
 * 
 * @author LuVar
 *
 * @param <V>	vertex class
 * @param <E>	edge class
 */
public class FewNegativeEdgesDijkstraShortestPath<V, E> implements MyShortestPath<V, E> {
	private static final Logger log = Logger.getLogger(FewNegativeEdgesDijkstraShortestPath.class);

	private final Graph<V, E> graph;
	private final boolean cacheNegativeEdgesList;
	protected final Transformer<E, ? extends Number> transformer;
	private final int waypointsMaxCount;
	private final boolean dropSlowercomplexPathsEarly;

	/**
	 * <p>
	 * Instance of {@link MyDijkstraShortestPath} class, with modified transformer. Original
	 * transformer is wrapped in {@link PositiveTransformer} transformer.
	 * </p>
	 */
	private final MyDijkstraShortestPath<V, E> dijkstra;
	private List<E> negativeEdges = null;

	/**
	 * <p>
	 * This constructor set {@link #cacheNegativeEdgesList} to true. It means that (nearly)
	 * no change in graph, nor in transformer, will be taken into account after first call
	 * of method {@link #getPath(Object, Object)}.
	 * </p>
	 * 
	 * @param graph
	 * @param transformer
	 */
	public FewNegativeEdgesDijkstraShortestPath(Graph<V, E> graph, Transformer<E, ? extends Number> transformer, boolean dropSlowercomplexPathsEarly) {
		this.graph = graph;
		this.transformer = transformer;
		this.cacheNegativeEdgesList = true;
		this.dijkstra = new MyDijkstraShortestPath<V, E>(graph, new PositiveTransformer<E>(transformer), this.cacheNegativeEdgesList);
		this.waypointsMaxCount = 1;
		this.dropSlowercomplexPathsEarly = dropSlowercomplexPathsEarly;
	}

	/**
	 * 
	 * @param graph
	 * @param transformer
	 * @param cacheNegativeEdgesList	if set to false, any call of {@link #getPath(Object, Object)}
	 * 						method will include finding of negative edges in graph (relatively
	 * 						a lot of computation)
	 */
	public FewNegativeEdgesDijkstraShortestPath(Graph<V, E> graph, Transformer<E, ? extends Number> transformer, boolean cacheNegativeEdgesList, boolean dropSlowercomplexPathsEarly) {
		this.graph = graph;
		this.transformer = transformer;
		this.cacheNegativeEdgesList = cacheNegativeEdgesList;
		this.dijkstra = new MyDijkstraShortestPath<V, E>(graph, new PositiveTransformer<E>(transformer), this.cacheNegativeEdgesList);
		this.waypointsMaxCount = 1;
		this.dropSlowercomplexPathsEarly = dropSlowercomplexPathsEarly;
	}

	/**
	 * 
	 * @param graph
	 * @param transformer
	 * @param cacheNegativeEdgesList	if set to false, any call of {@link #getPath(Object, Object)}
	 * 						method will include finding of negative edges in graph (relatively
	 * 						a lot of computation)
	 * @param waypointsMaxCount		maximum count of "way-points" when planning complex path. Default is 1.
	 */
	public FewNegativeEdgesDijkstraShortestPath(Graph<V, E> graph, Transformer<E, ? extends Number> transformer, boolean cacheNegativeEdgesList, int waypointsMaxCount, boolean dropSlowercomplexPathsEarly) {
		this.graph = graph;
		this.transformer = transformer;
		this.cacheNegativeEdgesList = cacheNegativeEdgesList;
		this.dijkstra = new MyDijkstraShortestPath<V, E>(graph, new PositiveTransformer<E>(transformer), this.cacheNegativeEdgesList);
		this.waypointsMaxCount = waypointsMaxCount;
		this.dropSlowercomplexPathsEarly = dropSlowercomplexPathsEarly;
	}

	@Override
	public List<E> getPath(V source, V target) {
		if(!this.cacheNegativeEdgesList) {
			this.refreshNegativeEdgesList();
		}
		List<E> negativeEdges = this.findNegativeEdges();

		List<E> bestPath = null;
		double bestPathCost = Double.MAX_VALUE;
		//bestPath = this.dijkstra.getPath(source, target);
		bestPath = this.getComplexPath(source, null, target);
		bestPathCost = this.getPathCost(bestPath);

		//TODO try all possibilities also with more via
		if(this.waypointsMaxCount >= 1) {
			for (E e : negativeEdges) {
				List<E> tmpPath;
				if(this.dropSlowercomplexPathsEarly) {
					tmpPath = this.getComplexPath(source, (E[])new Object[]{e}, target, bestPathCost);
					if(tmpPath == null) {
						continue;
					}
				} else {
					tmpPath = this.getComplexPath(source, (E[])new Object[]{e}, target);
				}
				double tmpPathCost = this.getPathCost(tmpPath);
				if(tmpPathCost < bestPathCost) {
					log.debug("Better path found through " + e + " edge.");
					bestPath = tmpPath;
					bestPathCost = tmpPathCost;
				}
				if(this.waypointsMaxCount >= 2) {
					for (E f : negativeEdges) {
						if(e.equals(f)) {
							continue;
						}
						if(this.dropSlowercomplexPathsEarly) {
							tmpPath = this.getComplexPath(source, (E[])new Object[]{e, f}, target, bestPathCost);
							if(tmpPath == null) {
								continue;
							}
						} else {
							tmpPath = this.getComplexPath(source, (E[])new Object[]{e, f}, target);
						}
						tmpPathCost = this.getPathCost(tmpPath);
						if(tmpPathCost < bestPathCost) {
							log.debug("Better path found through " + e + " edge.");
							bestPath = tmpPath;
							bestPathCost = tmpPathCost;
						}
						if(this.waypointsMaxCount >= 3) {
							for (E g : negativeEdges) {
								if(e.equals(g) || f.equals(g)) {
									continue;
								}
								if(this.dropSlowercomplexPathsEarly) {
									tmpPath = this.getComplexPath(source, (E[])new Object[]{e, f, g}, target, bestPathCost);
									if(tmpPath == null) {
										continue;
									}
								} else {
									tmpPath = this.getComplexPath(source, (E[])new Object[]{e, f, g}, target);
								}
								tmpPathCost = this.getPathCost(tmpPath);
								if(tmpPathCost < bestPathCost) {
									log.debug("Better path found through " + e + " edge.");
									bestPath = tmpPath;
									bestPathCost = tmpPathCost;
								}
							}//end of foreach g
						}
					}
				}
			}// end of foreach e in negativeEdges
		}
		return bestPath;
	}

	protected void getPermutations() {

	}

	/**
	 * <p>
	 * Will plan a bunch of routes using {@link #dijkstra} path planner and merge that paths
	 * together.
	 * </p>
	 * 
	 * @param source	source vertex
	 * @param via	array of "way points" to plan path through
	 * @param target	final target
	 * @return
	 */
	public List<E> getComplexPath(V source, E[] via, V target) {
		if(via == null || via.length == 0) {
			return this.dijkstra.getPath(source, target);
		}

		V tmpSource = source;
		V tmpTarget = this.graph.getSource(via[0]);
		List<E> path = this.dijkstra.getPath(tmpSource, tmpTarget);
		for(int i = 1; i < via.length-1; i++) {
			tmpSource = tmpTarget;
			tmpTarget = this.graph.getSource(via[i]);
			path.addAll(this.dijkstra.getPath(tmpSource, tmpTarget));
		}
		tmpSource = tmpTarget;
		tmpTarget = target;
		path.addAll(this.dijkstra.getPath(tmpSource, tmpTarget));
		return path;
	}

	/**
	 * <p>
	 * Like {@link #getComplexPath(Object, Object[], Object)} methods, this method returns
	 * complex path, but if cost of path cross given limit in max parameter, method will
	 * return null.
	 * </p>
	 * <p>
	 * Dont mind that cost of path can exceed given maximum cost, which will cause ending
	 * of path finding. This can possibly drop cheaper path, which can get cheaper after
	 * planning a next subpath.
	 * </p>
	 * 
	 * @param source
	 * @param via
	 * @param target
	 * @param max
	 * @return
	 */
	public List<E> getComplexPath(V source, E[] via, V target, double max) {
		double navrat = 0;
		if(via == null || via.length == 0) {
			return this.dijkstra.getPath(source, target);
		}

		V tmpSource = source;
		V tmpTarget = this.graph.getSource(via[0]);
		List<E> path = this.dijkstra.getPath(tmpSource, tmpTarget);

		//Check if current path doesnt cost more than max allowed.
		for (E e : path) {
			navrat += this.transformer.transform(e).doubleValue();
		}// end of foreach MyEdge in path
		if(navrat > max) {
			return null;
		}

		for(int i = 1; i < via.length-1; i++) {
			tmpSource = tmpTarget;
			tmpTarget = this.graph.getSource(via[i]);
			path = this.dijkstra.getPath(tmpSource, tmpTarget);
			path.addAll(path);

			//Check if current path doesnt cost more than max allowed.
			for (E e : path) {
				navrat += this.transformer.transform(e).doubleValue();
			}// end of foreach MyEdge in path
			if(navrat > max) {
				return null;
			}
		}
		tmpSource = tmpTarget;
		tmpTarget = target;
		path.addAll(this.dijkstra.getPath(tmpSource, tmpTarget));
		return path;
	}

	private void refreshNegativeEdgesList() {
		this.negativeEdges = null;
		this.findNegativeEdges();
	}

	private List<E> findNegativeEdges() {
		if(this.negativeEdges != null) {
			return this.negativeEdges;
		}
		log.debug("Going to find negative edges.");
		Collection<E> edges = this.graph.getEdges();
		this.negativeEdges = new ArrayList<E>();
		for (E e : edges) {
			Number n = this.transformer.transform(e);
			if(n.doubleValue() < 0) {
				this.negativeEdges.add(e);
			}
		}// end of foreach e in edges
		log.debug("Found " + this.negativeEdges.size() + " negative edges.");
		return this.negativeEdges;
	}

	public double getPathCost(List<E> path) {
		double navrat = 0;
		for (E e : path) {
			navrat += this.transformer.transform(e).doubleValue();
		}// end of foreach myEdge in path
		return navrat;
	}

	@Override
	public String toString() {
		//return super.toString();
		return super.toString() + "-" + this.dropSlowercomplexPathsEarly + "-" + this.waypointsMaxCount;
		//return "FewNegativeEdgesDijkstraShortestPath(transformer=" + this.transformer +
		//",cacheNegativeEdgesList=" + this.cacheNegativeEdgesList +
		//",dropSlowercomplexPathsEarly=" + this.dropSlowercomplexPathsEarly + ")";
	}
}

/**
 * <p>
 * Transformer that can wrap any transformer and filter negative numbers on output.
 * Any negative number gets transformed to zero value.
 * </p>
 * 
 * @author LuVar
 *
 * @param <E>	edge class
 */
class PositiveTransformer<E> implements Transformer<E, Number> {
	protected final Transformer<E, ? extends Number> transformer;

	public PositiveTransformer(Transformer<E, ? extends Number> transformer) {
		this.transformer = transformer;
	}

	@Override
	public Number transform(E input) {
		Number n =this.transformer.transform(input);
		if(n.doubleValue() < 0) {
			return 0;
		}
		return n;
	}
}

/*class Permutations {
	// print N! permutation of the elements of array a (not in order)
	public static List<Object[]> perm2(Object s[]) {
		int n = s.length;
		return perm2(s, n);
	}

	private static List<Object[]> perm2(Object[] a, int n) {
		List<Object[]> navrat = new ArrayList<Object[]>();
		if (n == 1) {
			navrat.add(a);
			return navrat;
		}
		for (int i = 0; i < n; i++) {
			Object c;
			c = a[i]; a[i] = a[n-1]; a[n-1] = c;
			navrat.addAll(perm2(a, n-1));
			c = a[i]; a[i] = a[n-1]; a[n-1] = c;
		}
		return navrat;
	}
}
 */