/**
 * 
 */
package jung.myalghoritm;

import java.security.InvalidParameterException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections15.Transformer;
import org.apache.log4j.Logger;

import edu.uci.ics.jung.graph.Graph;
import jung.myalghoritm.FloydWarshallShortestPath.*;

/**
 * <p>
 * Refactored from FloydWarshallMap from project Pogamut.
 * </p>
 * <p>
 * NOTE, that negative cycles will cause stack overflow...
 * </p>
 * 
 * @author LuVar
 * @author Martin Krulis
 * @author Jakub Gemrot
 * 
 */
public class FloydWarshallShortestPath<V, E> implements MyShortestPath<V, E> {
    private static final Logger log = Logger.getLogger(FloydWarshallShortestPath.class);

    public class PathMatrixNode<Vv, Ee> {
	private float distance = Float.POSITIVE_INFINITY;
	private Integer viaNode = null;
	private List<Ee> path = null;

	/**
	 * Doesn't leading anywhere
	 */
	public PathMatrixNode() {
	}

	public PathMatrixNode(float distance) {
	    this.distance = distance;
	}

	public float getDistance() {
	    return this.distance;
	}

	public void setDistance(float distance) {
	    this.distance = distance;
	}

	/**
	 * Returns indice.
	 * 
	 * @return
	 */
	public Integer getViaNode() {
	    return this.viaNode;
	}

	public void setViaNode(Integer indice) {
	    this.viaNode = indice;
	}

	public List<Ee> getPath() {
	    return this.path;
	}

	public void setPath(List<Vv> path, Graph<Vv, Ee> g) {
	    // TODO look if thic converting is necessary
	    List<Ee> convertedPath = new ArrayList<Ee>(path.size() - 1);
	    Vv fromV = path.get(0);
	    for (int i = 1; i < path.size(); i++) {
		Vv toV = path.get(i);
		Ee e = g.findEdge(fromV, toV);
		convertedPath.add(e);
		fromV = toV;
	    }
	    this.path = convertedPath;
	}

	@Override
	public String toString() {
	    String navrat = "Node:" + FloydWarshallShortestPath.this.integerToVertice.get(this.viaNode);
	    return navrat;
	}
    }

    // Warshall's matrix of distances.
    protected PathMatrixNode<V, E>[][] pathMatrix;

    protected final Transformer<E, ? extends Number> transformer;
    protected final Graph<V, E> g;

    private final Map<V, Integer> verticeToInteger = new HashMap<V, Integer>();
    private final Map<Integer, V> integerToVertice = new HashMap<Integer, V>();

    /**
     * <p>
     * Constructor. It will log informations about speed of some stages in creating structure needed for shortest path
     * computation.
     * </p>
     * 
     * @param g graph
     * @param transformer transformer for cost function for edges
     * @param precomputePaths it true, in constructor will be constructed all paths
     */
    public FloydWarshallShortestPath(Graph<V, E> g, Transformer<E, ? extends Number> transformer, boolean precomputePaths) {
	this.transformer = transformer;
	this.g = g;
	performFloydWarshall(precomputePaths);
    }

    protected void performFloydWarshall(boolean precomputePaths) {
	List<V> vertices = new ArrayList<V>(this.g.getVertices());
	int size = vertices.size();
	log.debug("Floyd-Warshall algorithm starting on " + size + " vertices.");
	long start = System.currentTimeMillis();

	// prepares data structures
	this.pathMatrix = new PathMatrixNode[size][size];

	// Initialize mapping.
	for (int i = 0; i < vertices.size(); ++i) {
	    this.verticeToInteger.put(vertices.get(i), i);
	    this.integerToVertice.put(i, vertices.get(i));
	}

	// Initialize distance matrix.
	for (int i = 0; i < size; i++) {
	    for (int j = 0; j < size; j++) {
		this.pathMatrix[i][j] = new PathMatrixNode((i == j) ? 0.0f : Float.POSITIVE_INFINITY);
	    }
	}

	long initEnd = System.currentTimeMillis();
	log.debug("Floyd-Warshall init done in " + (initEnd - start) + " ms.");

	// Set edge lengths into distance matrix.
	for (int i = 0; i < size; i++) {
	    V v = vertices.get(i);
	    for (E edge : this.g.getOutEdges(v)) {
		int index2 = this.verticeToInteger.get(this.g.getOpposite(v, edge));
		this.pathMatrix[i][index2].setDistance(this.transformer.transform(edge).floatValue());
	    }
	}
	long settingLengthsEnd = System.currentTimeMillis();
	log.debug("Floyd-Warshall setting lengths done in " + (settingLengthsEnd - initEnd) + " ms.");

	// Perform Floyd-Warshall.
	for (int k = 0; k < size; k++) {
	    for (int i = 0; i < size; i++) {
		for (int j = 0; j < size; j++) {
		    float newLen = this.pathMatrix[i][k].getDistance() + this.pathMatrix[k][j].getDistance();
		    if (this.pathMatrix[i][j].getDistance() > newLen) {
			this.pathMatrix[i][j].setDistance(newLen);
			this.pathMatrix[i][j].setViaNode(k);
		    }
		}
	    }
	}

	long floydWarshallEnd = System.currentTimeMillis();
	log.debug("Floyd-Warshall floyd warshall core done in " + (floydWarshallEnd - settingLengthsEnd) + " ms.");

	if (precomputePaths) {
	    // Construct paths + log unreachable paths.
	    int count = 0;
	    for (int i = 0; i < size; i++) {
		for (int j = 0; j < size; j++) {
		    if (this.pathMatrix[i][j].getDistance() == Float.POSITIVE_INFINITY) {
			log.trace("Unreachable path from " + vertices.get(i) + " -> " + vertices.get(j));
			count++;
		    } else {
			// path exists ... retrieve it
			// TODO possibly do it in more threads
			this.pathMatrix[i][j].setPath(retrievePath(i, j), this.g);
		    }
		}
	    }
	    log.info(this + ": There are " + count + " unreachable nav point pairs.");
	    long constructsPathsEnd = System.currentTimeMillis();
	    log.debug("Floyd-Warshall constructing paths done in " + (constructsPathsEnd - floydWarshallEnd) + " ms.");
	} else {
	    log.info("Not precomputing paths. Paths will be constructed on the fly.");
	}

	long endTime = System.currentTimeMillis();
	log.info(this + ": computation for " + size + " navpoints took " + (endTime - start) + " millis");
    }

    /**
     * Sub-routine of retrievePath - do not use! ... Well you may, it returns path without 'from', 'to' or null if such
     * path dosn't exist.
     * <p>
     * <p>
     * DO NOT USE OUTSIDE CONSTRUCTOR (relies on indicesNavPoints).
     * 
     * @param from
     * @param to
     * @return
     */
    private List<V> retrievePathInner(Integer from, Integer to) {
	PathMatrixNode node = this.pathMatrix[from][to];
	if (node.getDistance() == Float.POSITIVE_INFINITY) {
	    throw new RuntimeException("Unreachanable target path planning. (" + this.integerToVertice.get(from) + "," + this.integerToVertice.get(to) + ") " + node);
	}
	if (node.getViaNode() == null) {
	    return new ArrayList<V>(0);
	}

	List<V> path = new ArrayList<V>();
	path.addAll(retrievePathInner(from, node.getViaNode()));
	path.add(this.integerToVertice.get(node.getViaNode()));
	path.addAll(retrievePathInner(node.getViaNode(), to));

	return path;
    }

    /**
     * Returns path between from-to or null if path doesn't exist. Path begins with 'from' and ends with 'to'.
     * <p>
     * <p>
     * DO NOT USE OUTSIDE CONSTRUCTOR (relies on indicesNavPoints).
     * 
     * @param from
     * @param to
     * @return
     */
    private List<V> retrievePath(Integer from, Integer to) {
	List<V> path = new ArrayList<V>();
	path.add(this.integerToVertice.get(from));
	path.addAll(retrievePathInner(from, to));
	path.add(this.integerToVertice.get(to));
	return path;
    }

    protected PathMatrixNode getPathMatrixNode(V np1, V np2) {
	int i = this.verticeToInteger.get(np1);
	int j = this.verticeToInteger.get(np2);
	if (this.pathMatrix[i][j].path == null) {
	    this.pathMatrix[i][j].setPath(retrievePath(i, j), this.g);
	}
	return this.pathMatrix[i][j];
    }

    /**
     * Whether navpoint 'to' is reachable from the navpoint 'from'.
     * 
     * @param from
     * @param to
     * @return
     */
    public boolean reachable(V from, V to) {
	if ((from == null) || (to == null)) {
	    return false;
	}
	return getPathMatrixNode(from, to).getDistance() != Float.POSITIVE_INFINITY;
    }

    /**
     * Calculate's distance between two nav points (using pathfinding).
     * 
     * @return Distance or POSITIVE_INFINITY if there's no path.
     */
    public float getDistance(V from, V to) {
	if ((from == null) || (to == null)) {
	    return Float.POSITIVE_INFINITY;
	}
	return getPathMatrixNode(from, to).getDistance();
    }

    @Override
    public String toString() {
	return "FloydWarshallMap";
    }

    /**
     * Returns path between vertices 'from' -> 'to'. The path begins with 'from' and ends with 'to'. If such path
     * doesn't exist,
     * 
     * @param from
     * @param to
     * @return
     */
    public List<E> getPath(V source, V target) {
	if ((source == null) || (target == null)) {
	    throw new InvalidParameterException("Source, nor target can be null!");
	}
	return getPathMatrixNode(source, target).getPath();
    }

}
