package jung.myio;

import java.io.IOException;
import java.io.Writer;
import java.util.Collection;
import java.util.Map;

import org.apache.commons.collections15.Transformer;

import edu.uci.ics.jung.graph.Graph;
import edu.uci.ics.jung.graph.Hypergraph;
import edu.uci.ics.jung.graph.UndirectedGraph;
import edu.uci.ics.jung.graph.util.EdgeType;
import edu.uci.ics.jung.graph.util.Pair;
import edu.uci.ics.jung.io.GraphIOException;
import edu.uci.ics.jung.io.GraphMLWriter;
import edu.uci.ics.jung.io.graphml.EdgeMetadata;
import edu.uci.ics.jung.io.graphml.GraphMLReader2;
import edu.uci.ics.jung.io.graphml.GraphMetadata;
import edu.uci.ics.jung.io.graphml.HyperEdgeMetadata;
import edu.uci.ics.jung.io.graphml.NodeMetadata;

/**
 * <p>
 * Class for writing GraphML files with only one {@link Transformer} to one item
 * type. You have provide one transformer for edges, one for vertices and one
 * for graph. Simplest possible solution as I hope.
 * </p>
 * <p>
 * This class was born in upset evening from {@link GraphMLWriter} class and
 * {@link GraphMLReader2} class. From first class was stolen implementation and
 * from second idea of simple one transformer serializing of data.
 * </p>
 * 
 * @author LuVar - luvar@plaintext.sk
 * 
 * @param <G>
 *            The graph type to be read from the GraphML file
 * @param <V>
 *            The vertex type used by the graph
 * @param <E>
 *            The edge type used by the graph
 * @see "http://graphml.graphdrawing.org/specification.html"
 */
public class GraphMLWriter2<G extends Hypergraph<V, E>, V, E> implements GraphWriter<G, V, E> {
	private final Writer fileWriter;
	private final Transformer<G, GraphMetadata> graphTransformer;
	private final Transformer<V, NodeMetadata> vertexTransformer;
	private final Transformer<E, EdgeMetadata> edgeTransformer;
	private final Transformer<E, HyperEdgeMetadata> hyperEdgeTransformer;
	private boolean directed = false;

	/**
	 * Constructs a GraphML writer around the given writer. Last parameter
	 * hyperEdgeTransformer can be left null, because it is not used and
	 * it is here for future implementation needs. 
	 * 
	 * @param fileWriter
	 *            the writer for the output GraphML document. Filename should
	 *            have "graphML" suffix.
	 * @param graphTransformer
	 *            Transformation function to convert from graph objects to
	 *            GraphML GraphMetadata. This must be non-null.
	 * @param vertexTransformer
	 *            Transformation function to convert from vertex objects to
	 *            GraphML NodeMetadata. This must be non-null.
	 * @param edgeTransformer
	 *            Transformation function to convert from edge objects to
	 *            GraphML EdgeMetadata. This must be non-null.
	 * @param hyperEdgeTransformer
	 *            Transformation function to convert from edge objects to
	 *            GraphML HyperEdgeMetadata. This must be non-null.
	 * @throws IllegalArgumentException
	 *             thrown if any of the arguments are null.
	 */
	public GraphMLWriter2(Writer fileWriter, Transformer<G, GraphMetadata> graphTransformer, Transformer<V, NodeMetadata> vertexTransformer,
			Transformer<E, EdgeMetadata> edgeTransformer, Transformer<E, HyperEdgeMetadata> hyperEdgeTransformer) {

		if (fileWriter == null) {
			throw new IllegalArgumentException("Argument fileWriter must be non-null");
		}

		if (graphTransformer == null) {
			throw new IllegalArgumentException("Argument graphTransformer must be non-null");
		}

		if (vertexTransformer == null) {
			throw new IllegalArgumentException("Argument vertexTransformer must be non-null");
		}

		if (edgeTransformer == null) {
			throw new IllegalArgumentException("Argument edgeTransformer must be non-null");
		}

		//TODO include this check after implementing it!
		//if (hyperEdgeTransformer == null) {
		//	throw new IllegalArgumentException("Argument hyperEdgeTransformer must be non-null");
		//}

		this.fileWriter = fileWriter;
		this.graphTransformer = graphTransformer;
		this.vertexTransformer = vertexTransformer;
		this.edgeTransformer = edgeTransformer;
		this.hyperEdgeTransformer = hyperEdgeTransformer;
	}

	/**
	 * Gets the current transformer that is being used for graph objects.
	 * 
	 * @return the current transformer.
	 */
	public Transformer<G, GraphMetadata> getGraphTransformer() {
		return graphTransformer;
	}

	/**
	 * Gets the current transformer that is being used for vertex objects.
	 * 
	 * @return the current transformer.
	 */
	public Transformer<V, NodeMetadata> getVertexTransformer() {
		return vertexTransformer;
	}

	/**
	 * Gets the current transformer that is being used for edge objects.
	 * 
	 * @return the current transformer.
	 */
	public Transformer<E, EdgeMetadata> getEdgeTransformer() {
		return edgeTransformer;
	}

	/**
	 * Gets the current transformer that is being used for hyperedge objects.
	 * 
	 * @return the current transformer.
	 */
	public Transformer<E, HyperEdgeMetadata> getHyperEdgeTransformer() {
		return hyperEdgeTransformer;
	}

	/**
	 * <p>
	 * Closes the GraphML writer and disposes of any resources. This method is
	 * called after successful writing of graph in
	 * {@link #writeGraph(Hypergraph)} method.
	 * </p>
	 * 
	 * @throws edu.uci.ics.jung.io.GraphIOException
	 *             thrown if an error occurs.
	 */
	public void close() throws GraphIOException {
		try {

			if (this.fileWriter != null) {
				this.fileWriter.close();
			}

		} catch (IOException ex) {
			throw new GraphIOException("Error while closing filewriter in close method! Error:" + ex.getMessage(), ex);
		} finally {
			/*
			 * this.fileWriter = null; this.graphTransformer = null;
			 * this.vertexTransformer = null; this.edgeTransformer = null;
			 * this.hyperEdgeTransformer = null;
			 */
		}
	}

	@SuppressWarnings("unchecked")
	@Override
	public void writeGraph(G graph) throws GraphIOException {
		try {
			Writer wr = this.fileWriter;
			// write out boilerplate header
			wr.write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
			wr.write("<graphml xmlns=\"http://graphml.graphdrawing.org/xmlns/graphml\"\n" + "xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"  \n");
			wr.write("xsi:schemaLocation=\"http://graphml.graphdrawing.org/xmlns/graphml\">\n");

			// write out graph-level information
			// set edge default direction
			this.fileWriter.write("<graph edgedefault=\"");
			directed = !(graph instanceof UndirectedGraph);
			if (directed) {
				wr.write("directed\">\n");
			} else {
				wr.write("undirected\">\n");
			}

			GraphMetadata gm = this.getGraphTransformer().transform(graph);
			// write graph description, if any
			String desc = gm.getDescription();
			if (desc != null) {
				wr.write("<desc>" + desc + "</desc>\n");
			}

			// write graph data out if any
			for (Map.Entry<String, String> keyValue : gm.getProperties().entrySet()) {
				if (keyValue.getValue() != null)
					wr.write(format("data", "key", keyValue.getKey(), keyValue.getValue()) + "\n");
			}

			// write vertex information
			writeVertexData(graph, wr);

			// write edge information
			writeEdgeData(graph, wr);

			// close graph
			wr.write("</graph>\n");
			wr.write("</graphml>\n");
			wr.flush();

			this.close();
		} catch (Exception ex) {
			throw new GraphIOException("Unable to read Graph from document - the document could be empty");
		}
	}// end of method writeGraph

	/**
	 * Writes part of vertex data to xml graphML format.
	 * 
	 * @param graph
	 * @param w
	 * @throws IOException
	 */
	protected void writeVertexData(Hypergraph<V, E> graph, Writer w) throws IOException {
		for (V v : graph.getVertices()) {
			NodeMetadata nm = this.vertexTransformer.transform(v);
			String v_string = String.format("<node id=\"%s\"", nm.getId());
			boolean closed = false;
			// write description out if any
			String desc = nm.getDescription();
			if (desc != null) {
				w.write(v_string + ">\n");
				closed = true;
				w.write("<desc>" + desc + "</desc>\n");
			}
			// write data out if any
			for (Map.Entry<String, String> keyValue : nm.getProperties().entrySet()) {
				if (keyValue.getValue() != null) {
					if (!closed) {
						w.write(v_string + ">\n");
						closed = true;
					}
					w.write(format("data", "key", keyValue.getKey(), keyValue.getValue()) + "\n");
				}
			}
			if (!closed)
				w.write(v_string + "/>\n"); // no contents; close the node with
			// "/>"
			else
				w.write("</node>\n");
		}
	}

	protected void writeEdgeData(Hypergraph<V, E> g, Writer w) throws IOException {
		for (E e : g.getEdges()) {
			EdgeMetadata em = this.edgeTransformer.transform(e);
			Collection<V> vertices = g.getIncidentVertices(e);
			String id = em.getId();
			String e_string;
			boolean is_hyperedge = !(g instanceof Graph);
			if (is_hyperedge) {
				e_string = "<hyperedge ";
				// add ID if present
				if (id != null)
					e_string += "id=\"" + id + "\" ";
			} else {
				Pair<V> endpoints = new Pair<V>(vertices);
				V v1 = endpoints.getFirst();
				V v2 = endpoints.getSecond();
				e_string = "<edge ";
				// add ID if present
				if (id != null)
					e_string += "id=\"" + id + "\" ";
				// add edge type if doesn't match default
				EdgeType edge_type = g.getEdgeType(e);
				if (directed && edge_type == EdgeType.UNDIRECTED)
					e_string += "directed=\"false\" ";
				if (!directed && edge_type == EdgeType.DIRECTED)
					e_string += "directed=\"true\" ";
				//TODO probably cache transformed metadata to some map for speed
				e_string += "source=\"" + this.vertexTransformer.transform(v1).getId() + "\" target=\"" + this.vertexTransformer.transform(v2).getId() + "\"";
			}

			boolean closed = false;
			// write description out if any
			String desc = em.getDescription();
			if (desc != null) {
				w.write(e_string + ">\n");
				closed = true;
				w.write("<desc>" + desc + "</desc>\n");
			}
			// write data out if any
			for (Map.Entry<String, String> keyValue : em.getProperties().entrySet()) {
				if (keyValue.getValue() != null) {
					if (!closed) {
						w.write(e_string + ">\n");
						closed = true;
					}
					w.write(format("data", "key", keyValue.getKey(), keyValue.getValue()) + "\n");
				}
			}
			// if this is a hyperedge, write endpoints out if any
			if (is_hyperedge) {
				for (V v : vertices) {
					if (!closed) {
						w.write(e_string + ">\n");
						closed = true;
					}
					//TODO I dont have used hyperedge transformer. wtf? It should be probably used here...
					if(1==1) throw new RuntimeException("Hyperedge saving is not implemented. Please consult source code :-/");
					w.write("<endpoint node=\"" + this.vertexTransformer.transform(v).getId() + "\"/>\n");
				}
			}

			if (!closed)
				w.write(e_string + "/>\n"); // no contents; close the edge with
											// "/>"
			else if (is_hyperedge)
				w.write("</hyperedge>\n");
			else
				w.write("</edge>\n");
		}
	}

	/**
	 * From {@link GraphMLWriter} class originally.
	 * 
	 * @param type
	 * @param attr
	 * @param value
	 * @param contents
	 * @return
	 */
	protected String format(String type, String attr, String value, String contents) {
		return String.format("<%s %s=\"%s\">%s</%s>", type, attr, value, contents, type);
	}
}
