package sk.stuba.fiit.pogamut.jungigation.worldInfo;

import java.util.ArrayList;
import java.util.Collection;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Item;

/**
 * Module specialized on items on the map. This module extends the {@link SharedItems}
 * module by adding filters for usability and/or takeability. These filters may
 * help the agent to decide, whether it is useful for him, to forage specific
 * items from the map.
 * <p><p>
 * You have to provide the implementation of {@link IUsefulItemFilter} interface 
 * where you have to specify <i>what usefulness is</i>. 
 *
 * @author Juraj 'Loque' Simlovic
 * @author Jimmy
 * @author LuVar
 */
public class AdvancedSharedItems extends SharedItems
{

	/**
	 * Retreives list of all items (both known and thrown).
	 *
	 * @param usefulness degree of usefulness, 0 - return also useless, 1 - return only <b>really truly</b> useful items which are MUST HAVE!
	 * @return List of all items. Note: Spawned items are included only.
	 */
	public Collection<Item> getAllItems(double usefulness)
	{
		return filterUsefulItems (
			super.getAllItems().values(), usefulness
		);
	}

	/*========================================================================*/

	/**
	 * Retreives list of all visible items.
	 *
	 * @param usefulness degree of usefulness, 0 - return also useless, 1 - return only <b>really truly</b> useful items which are MUST HAVE!
	 * @return List of all visible items. Note: Spawned items are included only.
	 */
	public Collection<Item> getVisibleItems(double usefulness)
	{
		return filterUsefulItems (
			super.getVisibleItems().values(), usefulness
		);
	}

	/*========================================================================*/

	/**
	 * Retreives list of all reachable items.
	 *
	 * @param usefulness degree of usefulness, 0 - return also useless, 1 - return only <b>really truly</b> useful items which are MUST HAVE!
	 * @return List of all reachable items. Note: Spawned items are included only.
	 */
	public Collection<Item> getReachableItems(double usefulness)
	{
		return filterUsefulItems (
			super.getReachableItems().values(), usefulness
		);
	}

	/*========================================================================*/

	/**
	 * Retreives list of all item pickup points.
	 *
	 * @param usefulness degree of usefulness, 0 - return also useless, 1 - return only <b>really truly</b> useful items which are MUST HAVE!
	 * @return List of all items. Note: Empty pickups are included as well.
	 *
	 * @see isPickupSpawned(Item)
	 */
	public Collection<Item> getKnownPickups(double usefulness)
	{
		return filterUsefulItems (
			super.getKnownPickups().values(), usefulness
		);
	}

	/*========================================================================*/

	/**
	 * Determines, whether an item can be useful for the agent.
	 *
	 * @param item Given item to be checked.
	 * @param usefulness degree of usefulness, 0 - return also useless, 1 - return only <b>really truly</b> useful items which are MUST HAVE!
	 * @return True, if the item can be useful.
	 */
	public boolean isItemUseful(Item item, double usefulness) {
		if (usefulness < 0) usefulness = 0;
		else if (usefulness > 1) usefulness = 1;
		return filter.isItemUseful(this, item, usefulness);
	}

	/**
	 * Determines, whether an item can be useful for the agent.
	 *
	 * @param items Items to be filtered on.
	 * @param usefulness degree of usefulness, 0 - return also useless, 1 - return only <b>really truly</b> useful items which are MUST HAVE!
	 * @return True, if the item can be useful.
	 */
	public Collection<Item> filterUsefulItems(Collection<Item> items, double usefulness)
	{
		// new empty list
		ArrayList<Item> list = new ArrayList<Item> (items.size());

		// run through items and filter them out
		for (Item i: items)
			if (isItemUseful(i, usefulness))
				// add only useful to the results
				list.add(i);

		return list;
	}

	/*========================================================================*/

	/**
	 * Filter of useful items.
	 */
	private IUsefulItemFilter filter;

	/**
	 * <p>
	 * With this method you can replace original filter passed in constructor.
	 * </p>
	 * <p>
	 * NOTE: Not thread safe!
	 * </p>
	 *
	 * @param filter	new filter for filtering items
	 */
	public void setFilter(IUsefulItemFilter filter) {
		this.filter = filter;
	}
	
	/**
	 * Constructor. Setups the memory module based on bot's world view.
	 * @param bot owner of the module that is using it
	 * @param agentInfo AgentInfo memory module
	 */
	public AdvancedSharedItems(IWorldView worldView, IUsefulItemFilter filter)
	{
		this(worldView, null, filter);
	}

	
	/**
	 * Constructor. Setups the memory module based on bot's world view.
	 * @param bot owner of the module that is using it
	 * @param agentInfo AgentInfo memory module
	 * @param log Logger to be used for logging runtime/debug info. If <i>null</i>, module creates its own logger.
	 */
	public AdvancedSharedItems(IWorldView worldView, Logger log, IUsefulItemFilter filter)
	{
		super(worldView, log);
		this.filter = filter;
	}

}
