package nl.tudelft.pogamut.ut3.agent.module.shooting.weapon;

import nl.tudelft.pogamut.unreal.agent.module.sensor.Projectiles;
import nl.tudelft.pogamut.unreal.agent.module.shooting.AbstractWeaponShooting;
import nl.tudelft.pogamut.unreal.agent.module.shooting.util.FacingUtil;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weaponry;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPref;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPrefs;
import cz.cuni.amis.pogamut.ut2004.bot.command.AdvancedShooting;
import cz.cuni.amis.pogamut.ut2004.bot.command.ImprovedShooting;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut3.communication.messages.UT3ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.IncomingProjectile;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;

/**
 * <p>
 * Module to work efficiently with the shock rifle. Will use shock combos when
 * possible.
 * </p>
 * 
 * <p>
 * To create a shock combo
 * {@link ShockRifleShooting#shoot(WeaponPref, ILocated)} must be called with a
 * weaponPref that allows the secondary fire mode to be used. When used through
 * {@link AdvancedShooting} the appropriate weapon preference must be set on
 * {@link WeaponPrefs}.
 * </p>
 * 
 * <p>
 * To avoid self damage the module won't use the secondary firemode when the
 * projectile may blow up in it's face and won't be able to survive the damage.
 * Generally {@link WeaponPrefs} should handle this but it may not be within
 * {@link ImprovedShooting#getChangeWeaponCooldown()}. When unable to use the
 * secondary, the primary will be used.
 * <p>
 * <p>
 * TODO: No more shield gun in UT3
 * To avoid projectiles being deflected back by a shield gun, the module won't
 * directly shoot at players who have an active shield gun facing them. It will
 * however attempt to detonate a combo above the target if they are in range.
 * </p>
 * 
 * @author mpkorstanje
 * 
 */
public class ShockRifleShooting extends AbstractWeaponShooting {

	/**
	 * Primary mode preference.
	 */
	protected static final WeaponPref SHOCK_RIFLE_PRIMARY = new WeaponPref(UT3ItemType.SHOCK_RIFLE, true);

	/**
	 * Secondary mode preference.
	 */
	protected static final WeaponPref SHOCK_RIFLE_SECONDARY = new WeaponPref(UT3ItemType.SHOCK_RIFLE, false);

	/**
	 * Radius of the shock combo blast. Experimentally determined.
	 * 
	 */
	protected static final double SHOCK_RIFLE_COMBO_RADIUS = 400;

	/**
	 * Radius of the shock orb splash damage. Experimentally determined.
	 */
	protected static final double SHOCK_RIFLE_PROJECTILE_SPLASH_RADIUS = 150;

	/**
	 * Damage to self when hit by projectile. Experimentally determined.
	 */
	protected static final int SHOCK_RIFLE_PROJECTILE_DAMAGE = 35;

	/**
	 * Minimum distance in UT units the orb will travel before reload.
	 * Experimentally determined.
	 */
	protected static final int SHOCK_RIFLE_MIN_DISTANCE_COMBO = 900;

	protected Projectiles projectiles;

	/**
	 * Last known location of our target. Used to detonate shock combos on
	 * targets that have hidden. They might still be close.
	 */
	protected Location lastLocation;

	public ShockRifleShooting(UT2004Bot<?, ?, ?> bot, AgentInfo info, ImprovedShooting shoot, Weaponry weaponry, Projectiles projectiles) {
		super(bot, info, shoot, weaponry);

		this.weaponry = weaponry;
		this.projectiles = projectiles;

	}

	/**
	 * The actual shooting.
	 */
	@Override
	protected void shoot() {

		// We should have the shock rifle ready.
		if (!isWeaponReady()) {
			return;
		}

		// See if we have an orb to shoot down.
		// Will also try to catch hidden players in blast.
		if (!hasTarget() && shootCombo(lastLocation)) {
			return;
		}

		// Reset last location, we've either shot
		// a combo or there is no possibility of a combo.
		lastLocation = null;

		// No target, no shoot.
		if (!hasTarget()) {
			shoot.stopShooting();
			return;
		}

		// Reset focus
		this.focus.setFocus(target);

		boolean safeToSHoot = isSafeToShoot(target, weaponPref);
		boolean facing = FacingUtil.isFacing2D(info, target, FACING_ANGLE);

		// Don't blow orbs into walls.
		if (!(target instanceof Player)) {
			shootTarget(safeToSHoot, facing);
			return;
		}

		Player player = (Player) target;
		
		// See if we have an orb to shoot down.
		// Will also try to catch hidden players in blast.
		if (shootCombo(player)) {
			return;
		}
		
		// Target not visible, hold fire.
		if (!player.isVisible()) {
			shoot.stopShooting();
			return;
		}

		// Store last known location. We'll use this to flush out hidden
		// players.
		lastLocation = player.getLocation();

		// Shoot if we are facing the right way.
		// Don't want to blow an orb on a wall.		

                shootTarget(safeToSHoot, facing);
	}

	/**
	 * Shoot without getting splash damage and while looking in the right
	 * direction.
	 * 
	 * @param safeToShoot
	 * @param facing
	 */
	protected void shootTarget(boolean safeToShoot, boolean facing) {
		log.fine(String.format("Shooting target. safeToSHoot=%s facing=%s ", safeToShoot, facing));
		if (safeToShoot && facing) {
			shoot.shoot(weaponPref, target);
		} else if (facing) {
			shoot.shoot(SHOCK_RIFLE_PRIMARY, target);
		} else {
			shoot.stopShooting();
		}
	}

	/**
	 * Don't shoot unless we are far away enough to avoid damaging ourselves, or
	 * healthy enough to survive the damage, or not using the secondary mode.
	 * 
	 * @param target
	 *            to shoot
	 * @return true iff we won't kill ourselves.
	 */
	protected boolean isSafeToShoot(ILocated target, WeaponPref weaponPref) {
		boolean primary = weaponPref.isPrimary();

		double distance = info.getLocation().getDistance(target.getLocation());
		boolean safeDistance = distance > SHOCK_RIFLE_PROJECTILE_SPLASH_RADIUS;
		boolean healty = info.getHealth() > SHOCK_RIFLE_PROJECTILE_DAMAGE;

		return (primary || safeDistance || healty);
	}

	protected boolean shootCombo(ILocated target) {
		if (target == null || target.getLocation() == null) {
			log.fine("No valid target");
			return false;
		}

		IncomingProjectile projectile = projectiles.getNearestProjectile(target, UT3ItemType.SHOCK_RIFLE_PROJECTILE);
                
		if (projectile == null) {
			log.fine("No valid projectile to shoot");
			return false;
		}

		double distanceOrbTarget = projectile.getLocation().getDistance(target.getLocation());
		double distanceSelfTarget = info.getLocation().getDistance(target.getLocation());
		double distanceSelfOrb = info.getLocation().getDistance(projectile.getLocation());

		boolean orbPassedTarget = distanceSelfOrb > distanceSelfTarget + SHOCK_RIFLE_COMBO_RADIUS;
		boolean targetInComboRadius = distanceOrbTarget < SHOCK_RIFLE_COMBO_RADIUS;
                
		log.fine(String.format("Selected projectile: %s", projectile));
		log.fine(String.format("Projectile has passed target? %s", orbPassedTarget));
		log.fine(String.format("Target in combo radius? %s", targetInComboRadius));

		// Only shoot when we have a chance of blowing up the orb on our
		// target. We'll wait for it to move into place.
		// TODO: We might be waiting for an orb that is moving towards us.

		if (!orbPassedTarget && !targetInComboRadius) {
			log.fine("Waiting for projectile to close in on target");
			shoot.stopShooting();
			focus.setFocus((ILocated) projectile);
			return true;
		}
		// Shoot orb
		else if (!orbPassedTarget) {
			log.fine("Shoot projectile");
			shoot.shoot(SHOCK_RIFLE_PRIMARY, projectile.getId());
			return true;
		}

		return false;
	}

	@Override
	protected WeaponPref getDefaultWeaponPref() {
		return SHOCK_RIFLE_SECONDARY;
	}

}
