package nl.tudelft.pogamut.ut3.agent.module.shooting.weapon;

import nl.tudelft.pogamut.unreal.agent.module.shooting.AbstractWeaponShooting;
import nl.tudelft.pogamut.unreal.agent.module.shooting.util.FacingUtil;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensomotoric.Weaponry;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.WeaponPref;
import cz.cuni.amis.pogamut.ut2004.bot.command.ImprovedShooting;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut3.communication.messages.UT3ItemType;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.agent.module.sensor.AgentInfo;

/**
 * <p>
 * Module to work efficiently with the rocket launcher.
 * </p>
 * <p>
 * The rocket launcher is fairly hard to use as it effectiveness lies in
 * predicting where the player will be. Something a little too complex. This
 * module only provides support for shooting players that are currently visible
 * or have recently hidden.
 * </p>
 * <p>
 * To avoid self damage, this module won't fire when the target is too close and
 * the bot survive the blast.
 * </p>
 * <p>
 * When a target hides the bot will shoot a little below its last known
 * location, trying to flush the target out.
 * </p>
 *
 * @author mpkorstanje
 *
 */
public class RocketLauncherShooting extends AbstractWeaponShooting {

    protected static final WeaponPref ROCKET_LAUNCHER_PRIMARY = new WeaponPref(UT3ItemType.ROCKET_LAUNCHER, true);
    protected static final double ROCKET_LAUNCHER_PROJECTILE_SPLASH_RADIUS = 250;
    protected static final int ROCKET_LAUNCHER_PROJECTILE_DAMAGE = 80;
    protected static final int ROCKET_LAUNCHER_THREE_PROJECTILE_DAMAGE = 3 * ROCKET_LAUNCHER_PROJECTILE_DAMAGE;
    protected static final double ROCKET_LAUNCHER_CHARGE_TIME_SECONDS = 1.0;
    protected Location lastLocation = null;

    public RocketLauncherShooting(UT2004Bot<?, ?, ?> agent, AgentInfo info, ImprovedShooting shoot, Weaponry weaponry) {
        super(agent, info, shoot, weaponry);
    }

    @Override
    protected void shoot() {

        // Wrong weapon, wait up.
        if (!isWeaponReady()) {
            return;
        }

        // No target, no shoot.
        if (!hasTarget()) {
            shoot.stopShooting();
            return;
        }

        boolean facing = FacingUtil.isFacing2D(info, target, FACING_ANGLE);
        boolean safeToShoot = isSafeToShoot(target);

        if (!(target instanceof Player)) {
            shootLocation(facing, safeToShoot);
            return;
        }

        Player player = (Player) target;

        // Target hidden, flush target out.
        if (!player.isVisible() && info.isSecondaryShooting()) {
            shoot.shootPrimary(lastLocation.sub(BELOW_PLAYER_OFFSET));
            // Break of charge.
            shoot.stopShooting();
            return;
        } // Target not visible, hold fire.
        else if (!player.isVisible()) {
            shoot.stopShooting();
            return;
        }

        // Store last location so we can flush player out.
        lastLocation = player.getLocation();

        // Shoot if we are facing the right way.
        // Don't want to blow a rocket on a wall.
        if (facing && safeToShoot) {
            if (weaponPref.isPrimary()) {
                shoot.shoot(weaponPref, target);
            } else {
                shoot.shootSecondaryCharged(target, ROCKET_LAUNCHER_CHARGE_TIME_SECONDS);
            }
        } else {
            shoot.stopShooting();
        }
    }

    private void shootLocation(boolean facing, boolean safeToShoot) {
        if (facing && safeToShoot) {
            if (weaponPref.isPrimary()) {
                shoot.shoot(weaponPref, target);
            } else {
                shoot.shootSecondaryCharged(target, ROCKET_LAUNCHER_CHARGE_TIME_SECONDS);
            }
        } else {
            shoot.stopShooting();
        }
    }

    /**
     * Don't shoot unless we are far away enough to avoid damaging ourselves, or
     * healthy enough to survive the damage.
     *
     * @param target to shoot
     * @return true iff we won't kill ourselves.
     */
    protected boolean isSafeToShoot(ILocated target) {

        double distance = info.getLocation().getDistance(target.getLocation());
        boolean safeDistance = distance > ROCKET_LAUNCHER_PROJECTILE_SPLASH_RADIUS;
        double damage = weaponPref.isPrimary() ? ROCKET_LAUNCHER_PROJECTILE_DAMAGE
                : ROCKET_LAUNCHER_THREE_PROJECTILE_DAMAGE;
        boolean healty = info.getHealth() > damage;

        return (safeDistance || healty);
    }

    @Override
    protected WeaponPref getDefaultWeaponPref() {
        return ROCKET_LAUNCHER_PRIMARY;
    }
}
