/**
 * BaseUnrealEnvironment, an implementation of the environment interface standard that 
 * facilitates the connection between GOAL and the UT2004 engine. 
 * 
 * Copyright (C) 2012 BaseUnrealEnvironment authors.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 * the terms of the GNU General Public License as published by the Free Software
 * Foundation, either version 3 of the License, or (at your option) any later
 * version.
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program. If not, see <http://www.gnu.org/licenses/>.
 */
package nl.tudelft.goal.unreal.messages;

import java.util.logging.Level;

/**
 * List of valid parameter keys that can be used to initialize the environment.
 * 
 * 
 * @author M.P. Korstanje
 * 
 */
public enum Key {

	/**
	 * List of names for bots.
	 * 
	 * The number of names provided matches the number of bots launched by the
	 * environment.
	 */
	BOTNAMES("botNames"),
	/**
	 * Address of the control server.
	 * 
	 * Should be of the form protocol//host:port
	 */
	CONTROLSERVER("controlServer"),
	/**
	 * Weather or not the bot aims ahead of the target.
	 * 
	 * Either "true" of "false".
	 */
	LEADTARGET("leadTarget"),
	/**
	 * Log level used. Controls how many messages are displayed on the console.
	 * 
	 * Valid log levels are any from {@link Level}.
	 */
	LOGLEVEL("logLevel"),
	/**
	 * Address of the unreal server.
	 * 
	 * Should be of the form protocol//host:port
	 */
	BOTSERVER("botServer"),
	/**
	 * Name used by the server. Any string.
	 */
	CONTROLSERVERNAME("controlServerName"),
	/**
	 * Skill of the bot between 0 (poor) and 7 (good).
	 * 
	 * Controls how well the bot aims.
	 */
	SKILL("skill"),
	/**
	 * Skin used by the bot.
	 * 
	 * Any one of {@link Skin} will do.
	 */
	SKIN("skin"),
	/**
	 * Team of the bot.
	 * 
	 * Either 0 (red) or 1 (blue).
	 * 
	 */
	TEAM("team"),
	/**
	 * Address for the visualizer service.
	 * 
	 */
	VISUALIZERSERVER("visualizer"),

	/**
	 * Start location for the bot.
	 */
	STARTLOCATION("startLocation"), 
	
	/**
	 * Start rotation for the bot.
	 */
	STARTROTATION("startRotation");

	// Human readable (camelCase) form of the enum.
	private String key;

	private Key(String name) {
		this.key = name;
	}

	/**
	 * 
	 * @return a human readable name.
	 */
	@Override
	public String toString() {
		return key;
	}

	/**
	 * 
	 * @return a list of valid values as a string.
	 */
	private static String listValid() {
		String ret = "";

		Key[] keys = Key.values();
		for (int i = 0; i < keys.length; i++) {
			ret += keys[i].toString();

			if (i < keys.length - 1)
				ret += ", ";
		}
		return ret;
	}

	/**
	 * Returns the enum with the value of the string. matches.
	 * 
	 * @param value
	 * @return an ParameterKey.
	 * @throws IllegalArgumentException
	 *             if the provided value was not a valid parameter key.
	 */
	public static Key parseKey(String value) throws IllegalArgumentException {
		assert value != null;

		for (Key key : Key.values()) {
			if (key.key.equalsIgnoreCase(value)) {
				return key;
			}
		}

		String message = "%s is not a valid parameter key. Valid keys are: %s.";
		message = String.format(message, value, listValid());
		throw new IllegalArgumentException(message);
	}

	public Object getKey() {
		return key;
	}
}