package nl.tudelft.goal.EIS2Java.handlers;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import nl.tudelft.goal.EIS2Java.exception.TranslationException;
import nl.tudelft.goal.EIS2Java.translation.Translator;
import nl.tudelft.goal.EIS2Java.util.EIS2JavaUtil;

import eis.exceptions.ActException;
import eis.exceptions.EntityException;
import eis.iilang.Action;
import eis.iilang.Parameter;
import eis.iilang.Percept;

public class SynchronizedActionHandler extends ActionHandler {

	protected final Map<String, Method> actionMethods;
	protected final AsynchronousEntity entity;

	public SynchronizedActionHandler(AsynchronousEntity entity) throws EntityException {
		this.entity = entity;
		this.actionMethods = EIS2JavaUtil.processActionAnnotations(entity.getClass());
	}

	@Override
	public final boolean isSupportedByEntity(Action action) {
		String actionName = EIS2JavaUtil.getNameOfAction(action);
		return actionMethods.get(actionName) != null;
	}

	@Override
	public final Percept performAction(Action action) throws ActException {
		String actionName = EIS2JavaUtil.getNameOfAction(action);
		Method actionMethod = actionMethods.get(actionName);
		if (actionMethod == null) {
			throw new ActException(ActException.FAILURE, "Entity does not support action: " + action);
		}

		// workaround/fix for an EIS issue. #1986.
		try {
			return performAction(entity, actionMethod, action.getName(), action.getParameters());
		} catch (Exception e) {
			if (e instanceof ActException) {
				throw (ActException) e;
			}
			throw new ActException(ActException.FAILURE, "execution of action " + action + "failed", e);
		}
	}

	/**
	 * Performs the action method on the given method using the parameters. The
	 * returned {@link Percept} will have the same name as the action.
	 * 
	 * @param entity
	 *            The entity to perform the method on.
	 * @param method
	 *            The method to invoke on the entity.
	 * @param actionName
	 *            The name of the action that is being performed.
	 * @param parameters
	 *            The parameters to the method (before translation).
	 * @return The percept generated by the action.
	 * @throws ActException
	 *             if the action can not be performed for any reason.
	 */
	private Percept performAction(AsynchronousEntity entity, Method method, String actionName,
			LinkedList<Parameter> parameters) throws ActException {
		Translator translator = Translator.getInstance();

		Class<?>[] parameterTypes = method.getParameterTypes();

		Object[] arguments = new Object[parameters.size()];
		// Doing it the hard way because LinkedList.get(index) sucks!
		int i = 0;
		for (Parameter parameter : parameters) {
			try {
				arguments[i] = translator.translate2Java(parameter, parameterTypes[i]);
			} catch (TranslationException e) {
				throw new ActException(ActException.FAILURE, "Action " + actionName + " with parameters " + parameters
						+ " failed to be translated", e);
			}
			i++;
		}

		Object returnValue;
		try {
			entity.acquire();
			try {
				returnValue = method.invoke(entity, arguments);
			} finally {
				entity.release();
			}

		} catch (IllegalArgumentException e) {
			throw new ActException(ActException.FAILURE, "Action " + actionName + " with parameters " + parameters
					+ " failed to execute", e);
		} catch (IllegalAccessException e) {
			throw new ActException(ActException.FAILURE, "Action " + actionName + " with parameters " + parameters
					+ " failed to execute", e);
		} catch (InvocationTargetException e) {
			throw new ActException(ActException.FAILURE, "Action " + actionName + " with parameters " + parameters
					+ " failed to execute", e);
		} catch (InterruptedException e) {
			throw new ActException(ActException.FAILURE, "Action " + actionName + " with parameters " + parameters
					+ " failed to execute. Interupted while aquiring the entity.", e);
		}
		if (returnValue == null) {
			return null;
		} else {
			try {
				return new Percept(actionName, translator.translate2Parameter(returnValue));
			} catch (TranslationException e) {
				throw new ActException(ActException.FAILURE, "Action " + actionName + " with parameters " + parameters
						+ " failed to return a proper failue", e);
			}
		}
	}

}
