package cz.cuni.pogamut.shed.widget;

import cz.cuni.amis.pogamut.sposh.elements.PoshElement;
import cz.cuni.pogamut.shed.presenter.IPresenter;
import java.awt.*;
import javax.swing.Action;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import org.netbeans.api.visual.action.ActionFactory;
import org.netbeans.api.visual.action.PopupMenuProvider;
import org.netbeans.api.visual.widget.Widget;

/**
 * Ancestor of all widgets in the {@link ShedScene}. It provides common
 * infrastructure and helper methods for all widgets, e.g. {@link ShedWidget#createAcceptProviders() drag and drop},
 * display name ect.
 * <p/>
 * The widget is painted as colored rectangle with text drawn over it. * If size
 * of widget is too small, the text will be truncated (three dots will be
 * appended to the truncated text) so it fits.
 * <p/>
 * The origin of the widget ([0, 0]) is at the top left corner.
 *
 * @author HonzaH
 * @param <T> Type of element this widget is attached to.
 */
public class ShedWidget extends Widget implements PopupMenuProvider, IPresentedWidget {

    /**
     * Display name of the widget.
     */
    private String displayName;
    /**
     * Distance between left border of the widget and first letter of the text
     */
    protected final static int textOfs = 4;
    /**
     * Color of the widget
     */
    protected Color color;
    /**
     * Provider of actions for this widget, menu, accept and others.
     */
    protected IPresenter actionProvider;
    /**
     * Default width of {@link ShedWidget}.
     */
    protected static final int width = 240;
    /**
     * Default height of {@link ShedWidget}.
     */
    protected static final int height = 30;
    /**
     * Size of font used for rendering title text
     */
    final static int TITLE_FONT_SIZE = 16;
    /**
     * Font used to display text on the widget
     */
    protected final static Font font = new Font("Helvetica", Font.BOLD, TITLE_FONT_SIZE);

    /**
     * Create new widget for the Shed.
     *
     * @param scene scene the widget belongs to.
     * @param displayName display name of the widget.
     */
    ShedWidget(ShedScene scene, String displayName, Color color) {
        super(scene);

        assert displayName != null;
        this.displayName = displayName;
        this.setToolTipText(this.displayName);
        assert color != null;
        this.color = color;

        setPreferredBounds(new Rectangle(width, height));
        getActions().addAction(ActionFactory.createPopupMenuAction(this));
    }

    /**
     * Get display name of the widget
     *
     * @return display name
     */
    public final String getDisplayName() {
        return displayName;
    }

    public final void setDisplayName(String newDisplayName) {
        this.displayName = newDisplayName;
        this.setToolTipText(this.displayName);
        this.revalidate(true);
    }
    
    
    /**
     * Method wrapping {@link Widget#getScene() } to avoid casting every time.
     *
     * @return the scene this widget belongs to
     */
    public ShedScene getShedScene() {
        return (ShedScene) getScene();
    }

    /**
     * Return menu associated with the widget (will be used as context menu).
     *
     * @param widget widget on which the popup menu is requested (always this)
     * @param localLocation Where was popup menu invoked, if null, invoked by
     * keyboard
     * @return Created menu or null, if no actions from {@link #getMenuActions()
     * }.
     */
    @Override
    public final JPopupMenu getPopupMenu(Widget widget, Point localLocation) {
        assert widget == this;

        Action[] actions = getMenuActions();
        if (actions == null) {
            return null;
        }
        JPopupMenu menu = new JPopupMenu();
        for (Action action : actions) {
            // Action is automatically fired since the item is created from it, no need for addActionListener()
            JMenuItem item = new JMenuItem(action);
            menu.add(item);
        }

        return menu;
    }

    /**
     * Get(create) list of actions for the context menu of the widget.
     *
     * @return Array of actions (can be null).
     */
    protected final Action[] getMenuActions() {
        return actionProvider.getMenuActions();
    }

    @Override
    public final IPresenter getPresenter() {
        return actionProvider;
    }

    /**
     * Set new presenter of the widget. Upon creation of the widget, it should
     * be null.
     *
     * @param newPresenter New presenter.
     */
    public final void setPresenter(IPresenter newPresenter) {
        this.actionProvider = newPresenter;
    }

    @Override
    protected void paintWidget() {
        Rectangle bounds = getBounds();
        Graphics2D g = getGraphics();
        g.setColor(color);
        g.fillRect(bounds.x, bounds.y, bounds.width, bounds.height);
        drawString(g, font, getForeground(), getDisplayName());
    }

    /**
     * Draw vertically centered string at {@link TextWidget#textOfs} that will
     * fit into the widget. If text is too long, truncate it and add ellipsis
     * (three dots) at the end.
     *
     * @param g context used to draw
     * @param font font which should be used to compute
     * @param color what color should text have
     * @param text text to draw.
     */
    protected final void drawString(Graphics2D g, Font font, Color color, String text) {
        FontMetrics metric = g.getFontMetrics(font);
        // we are supposed to have only one line, so no leading space between two lines
        int ascent = metric.getAscent();

        String fitString = getFittingString(text, metric, getBounds().width - 2 * textOfs);
        g.setFont(font);
        g.setColor(color);
        g.drawString(fitString, textOfs, getBounds().height / 2 + ascent / 4);
    }

    /**
     * Get string that will be shorter than maxWidth. If Passed text is too long
     * return part of text with "..." at the end.
     *
     * @param text Text we want to fit into maxWidth
     * @param metrics font metrics to measure length of strings
     * @param maxWidth maximal length the returned string can fit into.
     * @return text if it fits into maxWidth, otherwise maximal
     * text.substring(0,X).concat("...") that will fit into maxWidth.
     */
    protected final String getFittingString(String text, FontMetrics metrics, int maxWidth) {
        if (metrics.stringWidth(text) < maxWidth) {
            return text;
        }

        for (int index = text.length() - 1; index > 0; index--) {
            String shorter = text.substring(0, index).concat("...");
            if (metrics.stringWidth(shorter) < maxWidth) {
                return shorter;
            }
        }
        return "...";
    }

    @Override
    public String toString() {
        return "ShedWidget: " + getDisplayName();
    }
    
    
}
