package cz.cuni.pogamut.posh.palette;

import com.sun.source.tree.AnnotationTree;
import com.sun.source.tree.ClassTree;
import com.sun.source.tree.CompilationUnitTree;
import com.sun.source.tree.ExpressionTree;
import com.sun.source.tree.IdentifierTree;
import com.sun.source.tree.MethodTree;
import com.sun.source.tree.NewClassTree;
import com.sun.source.tree.ThrowTree;
import com.sun.source.tree.Tree;
import com.sun.source.util.TreePathScanner;
import cz.cuni.amis.pogamut.sposh.SPOSHAction;
import cz.cuni.amis.pogamut.sposh.SPOSHSense;
import cz.cuni.pogamut.posh.PoshDataObject;
import cz.cuni.pogamut.posh.palette.external.ExternalBehaviour2PaletteTask;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.awt.event.ActionEvent;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.lang.model.element.Element;
import javax.lang.model.element.Modifier;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.TypeKind;
import javax.lang.model.util.ElementFilter;
import javax.swing.AbstractAction;
import javax.swing.Action;
import org.netbeans.api.java.project.JavaProjectConstants;
import org.netbeans.api.java.source.CancellableTask;
import org.netbeans.api.java.source.CompilationInfo;
import org.netbeans.api.java.source.JavaSource;
import org.netbeans.api.java.source.JavaSource.Phase;
import org.netbeans.api.java.source.TreeMaker;
import org.netbeans.api.java.source.WorkingCopy;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.api.project.Sources;
import org.netbeans.spi.palette.DragAndDropHandler;
import org.netbeans.spi.palette.PaletteActions;
import org.netbeans.spi.palette.PaletteController;
import org.netbeans.spi.palette.PaletteFactory;
import org.openide.DialogDisplayer;
import org.openide.NotifyDescriptor;
import org.openide.awt.StatusDisplayer;
import org.openide.filesystems.FileObject;
import org.openide.nodes.Node;
import org.openide.util.Lookup;
import org.openide.util.datatransfer.ExTransferable;

/**
 * Palette factory that will create the palette that is used by POSH editor.
 * @author Honza
 */
public class PoshPaletteFactory {

    /**
     * Create a palette with root PaletteRoot
     * @return new palette for POSH editor
     */
    public static PaletteController createPalette(Lookup lookup) {
        PaletteRoot paletteRoot = new PaletteRoot(lookup);
        paletteRoot.setName("Palette Root");

        return PaletteFactory.createPalette(paletteRoot, new MyActions(lookup), null, new MyDnDHandler());
    }

    /**
     * Custom actions that are in palette
     */
    private static class MyActions extends PaletteActions {

        final private Lookup lookup;

        MyActions(Lookup lookup) {
            this.lookup = lookup;
        }

        @Override
        public Action[] getImportActions() {
            return null;
        }

        private static class Search4Behavior extends AbstractAction {

            final private Lookup lookup;

            Search4Behavior(String name, Lookup lookup) {
                super(name);
                this.lookup = lookup;
            }

            @Override
            public void actionPerformed(ActionEvent event) {
                // Hopw do I get behaviour file?
                // Let's assume that it is in same package as the plan.

                // Get the plan dataObject
                PoshDataObject dataObj = lookup.lookup(PoshDataObject.class);
                if (dataObj == null) {
                    StatusDisplayer.getDefault().setStatusText("Data object for this plan not found in the lookup.");
                    return;
                }
                // Get the project from plan
                Project project = FileOwnerQuery.getOwner(dataObj.getPrimaryFile());
                if (project == null) {
                    StatusDisplayer.getDefault().setStatusText("File " + dataObj.getPrimaryFile().getNameExt() + " is not part of project.");
                    return;
                }
                // Search through the project to find the class that implements the Behaviour
                Sources sources = ProjectUtils.getSources(project);
                SourceGroup[] sgs = sources.getSourceGroups(JavaProjectConstants.SOURCES_TYPE_JAVA);
                List<FileObject> behaviorFiles = new LinkedList<FileObject>();
                for (SourceGroup sg : sgs) {
                    behaviorFiles.addAll(search4Behaviour(sg.getRootFolder()));
                }

                if (behaviorFiles.size() == 0) {
                    NotifyDescriptor.Message nd = new NotifyDescriptor.Message(
                            "Behavior class not found. "
                            + "The name of file with behavior class has to containt \"behavior\"", NotifyDescriptor.ERROR_MESSAGE);
                    DialogDisplayer.getDefault().notify(nd);
                    return;
                }

                if (behaviorFiles.size() > 1) {
                    StringBuilder sb = new StringBuilder(
                            "Multiple files that may contain Behavior were found.\n" +
                            "In the future, selection dialog will be here. For now, just rename the ones you are not interested in.");
                    for (FileObject fo : behaviorFiles) {
                        sb.append('\n');
                        sb.append(fo.getNameExt());
                    }
                    NotifyDescriptor.Message nd = new NotifyDescriptor.Message(sb.toString(), NotifyDescriptor.ERROR_MESSAGE);
                    DialogDisplayer.getDefault().notify(nd);
                    return;
                }


                // for each class
                FileObject potentialBehavior = behaviorFiles.get(0);
                JavaSource js = JavaSource.forFileObject(potentialBehavior);

                String[] actions = getActions();
                String[] senses = getSenses();
/*
                System.err.print("Palette Actions:");
                for (String action : actions) {
                    System.err.print(" " + action);
                }
                System.err.println();

                System.err.print("Palette Senses:");
                for (String sebse : senses) {
                    System.err.print(" " + sebse);
                }
                System.err.println();
/*/

                try {
                    PrimitiveStubAdder methodAdder = new PrimitiveStubAdder(potentialBehavior.getName(), senses, actions);
                    js.runModificationTask(methodAdder).commit();
                } catch (IOException e) {
                    StatusDisplayer.getDefault().setStatusText(e.getMessage());
                    Logger.getLogger("global").log(Level.SEVERE, e.getMessage(), e);
                }

            }

            private String[] getSenses() {
                PaletteController pc = lookup.lookup(PaletteController.class);
                PaletteRoot palRoot = pc.getRoot().lookup(PaletteRoot.class);

                List<String> senses = new LinkedList<String>();
                Node[] nodes = palRoot.getSensesChildren().getNodes();
                for (Node node : nodes) {
                    senses.add(node.getDisplayName().replace("!", ""));
                }
                return senses.toArray(new String[0]);
            }

            private String[] getActions() {
                PaletteController pc = lookup.lookup(PaletteController.class);
                PaletteRoot palRoot = pc.getRoot().lookup(PaletteRoot.class);

                List<String> actions = new LinkedList<String>();
                Node[] nodes = palRoot.getActionsChildren().getNodes();
                for (Node node : nodes) {
                    actions.add(node.getDisplayName().replace("!", ""));
                }
                return actions.toArray(new String[0]);
            }

            /**
             * Add stubs of senses and actions to the behaviour file.
             */
            private static class PrimitiveStubAdder implements CancellableTask<WorkingCopy> {

                private String behaviorFileName;

                private static class AnnotationFlags {

                    public static AnnotationFlags createSense() {
                        AnnotationFlags af = new AnnotationFlags();
                        af.sense = true;
                        return af;
                    }

                    public static AnnotationFlags createAction() {
                        AnnotationFlags af = new AnnotationFlags();
                        af.action = true;
                        return af;
                    }
                    public boolean sense;
                    public boolean action;

                    @Override
                    public String toString() {
                        return "S:" + sense + " A:" + action;
                    }
                }
                private Map<String, AnnotationFlags> primitives = new HashMap<String, AnnotationFlags>();

                /**
                 * Create an adder that checks for name senses and actions
                 * @param behaviorFileName
                 * @param senses
                 * @param actions
                 */
                PrimitiveStubAdder(String behaviorFileName, String[] senses, String[] actions) {
                    this.behaviorFileName = behaviorFileName;

                    for (String sense : senses) {
                        if (!primitives.containsKey(sense)) {
                            primitives.put(sense, AnnotationFlags.createSense());
                        }
                    }

                    for (String action : actions) {
                        if (primitives.containsKey(action)) {
                            primitives.get(action).action = true;
                        } else {
                            primitives.put(action, AnnotationFlags.createAction());
                        }
                    }
                }

                /**
                 * Get tree for behaviour class (multiple can be defined, we choose
                 * the one with same name as the name of file).
                 * @param cu tree of whole file
                 * @return tree of behavior class or null
                 */
                private ClassTree getBehaviorTree(CompilationUnitTree cu) {
                    ClassTree behaviorClass = null;
                    for (Tree tree : cu.getTypeDecls()) {
                        if (tree.getKind() == Tree.Kind.CLASS) {
                            ClassTree cand = (ClassTree) tree;
                            // Is this main class of the file?
                            if (behaviorFileName.equals(cand.getSimpleName().toString())) {
                                behaviorClass = cand;
                            }
                        }
                    }
                    return behaviorClass;
                }

                private boolean isAnnotatedBy(AnnotationTree at, String annotationType) {

                    if (at.getAnnotationType().getKind() == Tree.Kind.IDENTIFIER) {
                        IdentifierTree identifier = (IdentifierTree) at.getAnnotationType();
                        System.err.println("Name of annotation" + identifier.getName());
                    } else {
                        Logger.getLogger("global").log(Level.SEVERE, "Unknown annotation kind " + at.getAnnotationType().getKind());
                    }
                    return false;
                }

                @Override
                public void run(WorkingCopy workingCopy) throws Exception {
                    workingCopy.toPhase(Phase.ELEMENTS_RESOLVED);

                    TreeMaker make = workingCopy.getTreeMaker();

                    CompilationUnitTree cuTree = workingCopy.getCompilationUnit();
                    ClassTree behaviorClass = getBehaviorTree(cuTree);
                    if (behaviorClass == null) {
                        StatusDisplayer.getDefault().setStatusText("Behavior class not found in file " + behaviorFileName);
                        return;
                    }

                    // Check if some methods are already implemented
                    for (Tree member : behaviorClass.getMembers()) {
                        // if it is method
                        if (member.getKind() == Tree.Kind.METHOD) {
                            MethodTree mt = (MethodTree) member;

                            String methodName = mt.getName().toString();
                            if (primitives.containsKey(methodName)) {
                                primitives.remove(methodName);
                                /*
                                AnnotationFlags flags = primitives.get(methodName);
                                List<? extends AnnotationTree> methodAnnotations = mt.getModifiers().getAnnotations();

                                System.out.println("Method " + mt.getName() + 
                                " is in list of primitivesm, checking for annotations " +
                                flags + " vs " + methodAnnotations);


                                // remove annotations the method has already specified
                                for (AnnotationTree annotations : methodAnnotations) {
                                if (isAnnotatedBy(annotations, SPOSHAction.class.getSimpleName())
                                || isAnnotatedBy(annotations, SPOSHAction.class.getName()))   {
                                flags.action = false;
                                }
                                if (isAnnotatedBy(annotations, SPOSHSense.class.getSimpleName())
                                || isAnnotatedBy(annotations, SPOSHSense.class.getName()) )  {
                                flags.sense = false;
                                }
                                }
                                // add the ones it hasn't specified
                                if (flags.action) {
                                methodAnnotations.add(make.Annotation(make.Identifier(SPOSHAction.class.getName()), Collections.EMPTY_LIST));
                                }

                                if (flags.sense) {
                                AnnotationTree at = make.Annotation(make.Identifier(SPOSHSense.class.getName()), Collections.EMPTY_LIST);
                                methodAnnotations.add(at);
                                }
                                 *
                                 */
                            }
                        }
                    }

                    // Now to remove the primitives that are still unimplemented
                    ClassTree newClassTree = behaviorClass;

                    for (Entry<String, AnnotationFlags> af : primitives.entrySet()) {
                        if (af.getValue().action || af.getValue().sense) {
                            MethodTree newMethod = createMethodTree(make, af.getKey(), TypeKind.INT, af.getValue().action, af.getValue().sense);
                            newClassTree = make.addClassMember(newClassTree, newMethod);
                        }
                    }
                    //copy = make.addCompUnitImport(copy,make.Import(make.Identifier("cz.cuni.amis.Test"), false));
//                    MethodTree newMethod2 = createMethodTree(make, "newMethod2", TypeKind.INT, true, true);

                    workingCopy.rewrite(behaviorClass, newClassTree);
                }

                /**
                 * Create tree for public parameterless method.
                 * @param make
                 * @param methodName
                 * @param returnType
                 * @param action
                 * @param sense
                 * @return
                 */
                private MethodTree createMethodTree(TreeMaker make, String methodName, TypeKind returnType, boolean action, boolean sense) {
                    // create modifiers according to passed flags
                    List<AnnotationTree> annotations = new ArrayList<AnnotationTree>(2);
                    if (action) {
                        annotations.add(make.Annotation(make.Identifier(SPOSHAction.class.getName()), Collections.EMPTY_LIST));
                    }
                    if (sense) {
                        annotations.add(make.Annotation(make.Identifier(SPOSHSense.class.getName()), Collections.EMPTY_LIST));
                    }


                    NewClassTree newClassTree = make.NewClass(null,
                            Collections.EMPTY_LIST,//List<? extends ExpressionTree> typeArguments,
                            make.Identifier("UnsupportedOperationException"),
                            Collections.singletonList(make.Literal("Not supported yet.")),//List<? extends ExpressionTree> arguments,
                            null);

                    ThrowTree defaultThrow = make.Throw(newClassTree);

                    return make.Method(
                            make.Modifiers(Collections.singleton(Modifier.PUBLIC), annotations), // access and annotations
                            methodName, // name
                            make.PrimitiveType(returnType), // return type
                            Collections.EMPTY_LIST, // type parameters for parameters
                            Collections.EMPTY_LIST, // parameters
                            Collections.EMPTY_LIST, // throws, nothing, else use make.Identitifer
                            //                            make.Block(Collections.EMPTY_LIST, false), // empty statement block
                            make.Block(Collections.singletonList(defaultThrow), false),
                            null // default value - not applicable here, used by annotations
                            );
                }

                @Override
                public void cancel() {
                }
            }

            /**
             * Look for files with Behaviour or Behavior in its name
             * @return
             */
            List<FileObject> search4Behaviour(FileObject root) {
                List<FileObject> files = new LinkedList<FileObject>();
                Enumeration<? extends FileObject> en = root.getChildren(true);

                while (en.hasMoreElements()) {
                    FileObject fo = en.nextElement();

                    //check that it is a file
                    if (!fo.isData()) {
                        continue;
                    }

                    // check if it is what I am looking for
                    if (!"java".equalsIgnoreCase(fo.getExt())) {
                        continue;
                    }
                    //check that name of file contains "behaviour" or "behavior"
                    String lowerName = fo.getName().toLowerCase();
                    if (!lowerName.contains("behavior") && !lowerName.contains("behaviour")) {
                        continue;
                    }

                    // yup, it passed all checks
                    files.add(fo);
                }
                return files;
            }
        }

        @Override
        public Action[] getCustomPaletteActions() {
            return new Action[]{
                        new AbstractAction("Import external behaviour") {

                            @Override
                            public void actionPerformed(ActionEvent e) {
                                new Thread(new ExternalBehaviour2PaletteTask(lookup)).start();
                            }
                        },
                        new Search4Behavior("Add stubs to behavior", lookup)};
        }

        @Override
        public Action[] getCustomCategoryActions(Lookup lookup) {
            return null;
        }

        @Override
        public Action[] getCustomItemActions(Lookup lookup) {
            AbstractPoshPaletteNode res = lookup.lookup(AbstractPoshPaletteNode.class);
            if (res == null) {
                return null;
            }
            return res.getActions(false);
        }

        @Override
        public Action getPreferredAction(Lookup lookup) {
            return null;
        }
    }

    /**
     * DnD handler that customizes the transferable when they are strarted dragged
     * across their journey.
     *
     * Take a currently pushed PoshPaletteNode and add node.createTransferable to the
     * transferable.
     */
    private static class MyDnDHandler extends DragAndDropHandler {

        @Override
        public void customize(ExTransferable exTransferable, Lookup lookup) {
            PoshPaletteNode node = lookup.lookup(PoshPaletteNode.class);

            if (node != null) {
                final String name = node.getName();

                exTransferable.put(node.createTransferable());
                //exTransferable.put(new DataNodeExTransferable(node.getPoshDataNode()));

                exTransferable.put(new ExTransferable.Single(DataFlavor.stringFlavor) {

                    @Override
                    protected Object getData() throws IOException, UnsupportedFlavorException {
                        return "DnD_" + name;
                    }
                });
            }

        }
    }
}

