package cz.cuni.pogamut.posh.palette.external;

import cz.cuni.pogamut.posh.PoshDataObject;
import cz.cuni.amis.pogamut.sposh.SPOSHAction;
import cz.cuni.amis.pogamut.sposh.SPOSHSense;
import cz.cuni.pogamut.posh.palette.PaletteRoot;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.List;
import java.util.Properties;
import java.util.logging.Logger;
import org.netbeans.api.java.project.JavaProjectConstants;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.api.project.Sources;
import org.netbeans.spi.palette.PaletteController;
import org.openide.filesystems.FileAlreadyLockedException;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.util.Exceptions;
import org.openide.util.Lookup;

/**
 * This class is responsible for generating sposh behaviour interface
 * according to info from palette.
 * @author Honza
 */
public class BehaviourInterfaceBuilder implements Runnable {

    final public String behaviourInterfaceKey = "behaviour.interface.class";

    private Lookup lookup;

    public BehaviourInterfaceBuilder(Lookup lookup) {
        this.lookup = lookup;
    }

    @Override
    public void run() {
        PoshDataObject dataObj = lookup.lookup(PoshDataObject.class);
        if (dataObj == null) {
            return;
        }

        Project project = FileOwnerQuery.getOwner(dataObj.getPrimaryFile());
        if (project == null)
            return;
        try {
            buildInterface(project);
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(BehaviourInterfaceBuilder.class.getName()).finest(ex.getMessage());
        } catch (IOException ex) {
            Logger.getLogger(BehaviourInterfaceBuilder.class.getName()).finest(ex.getMessage());
        }
    }

    /**
     * Return value of property from project.properties.
     *
     * @param project project from which we want property
     * @param key key of property we want
     * @return value of property or null if no such property exists
     */
    protected String getProjectProperty(Project project, String key)
            throws FileNotFoundException, IOException {
        FileObject propFile = project.getProjectDirectory().getFileObject("nbproject/project.properties");
        if (propFile == null) {
            throw new FileNotFoundException("project.properties of project were not found.");
        }

        Properties projectProperties = new Properties();
        InputStream is = null;
        try {
            is = propFile.getInputStream();
            projectProperties.load(is);
        } finally {
            if (is != null) {
                is.close();
            }
        }
        String prop = projectProperties.getProperty(key);
        if (prop == null || prop.equals("")) {
            throw new IllegalArgumentException("Property specifing the behaviour interface (" + behaviourInterfaceKey + ") for project \"" + project + "\" is empty.");
        }
        return prop;
    }

    protected void buildInterface(Project project) throws FileNotFoundException, IOException {
        // Get key from project.properties
        String behaviourInterface = getProjectProperty(project, behaviourInterfaceKey);
        if (behaviourInterface == null) {
            throw new IllegalArgumentException("Property specifing the behaviour interface (" + behaviourInterfaceKey + ") for project \"" + project + "\" not found");
        }

        // Find the dir where are stored java files, if multiple, return first one
        FileObject sourceDir = getSourceFolder(project);
        if (sourceDir == null) {
            throw new FileNotFoundException("Unable to find folder with sources for project \"" + project + "\"");
        }

        FileObject interfaceFile = createInterfaceFile(sourceDir, behaviourInterface);

        writeInterface(interfaceFile, behaviourInterface);
    }

    /**
     * Heuristic test if passed {@code SourceGroup} contains source of module
     * or some other kind of sources (like a test sourcees)
     * 
     * @param sg sourge group that is being tested
     * @return true if it contains sources of application
     */
    private boolean isSourceFolder(SourceGroup sg) {
        if (sg.getDisplayName().contains("source") || sg.getDisplayName().contains("Source")) {
            return true;
        }
        return false;
    }

    /**
     * Return source folder that contains java classes for project.
     *
     * @param project project we want source folder from
     * @return folder with sources of application or null is no such sources exists. If more
     *                than one exists, return first one.
     */
    private FileObject getSourceFolder(Project project) {
        Sources sources = ProjectUtils.getSources(project);
        SourceGroup[] sourceGroups = sources.getSourceGroups(JavaProjectConstants.SOURCES_TYPE_JAVA);

        for (SourceGroup sg : sourceGroups) {
            if (isSourceFolder(sg)) {
                return sg.getRootFolder();
            }
        }
        return null;
    }

    /**
     * Take a source dir and go through folder hiearchy to correct package and 
     * create a new java file with to represent behaviourInterface. Return
     * the fileobject representing the file.
     * <p>
     * If some folder is missing, create it. In case of error (like file has name of directory),
     * throw an IOException.
     *
     * @param sourceDir root folder of sources from which we traverse to the behaviour interface package
     * @param behaviourInterface canonical name of behaviour interface (you know, dot separated)
     * @return FileObject that is representation of behaviourInterface in the correct package dir in sourceDir
     */
    private FileObject createInterfaceFile(FileObject sourceDir, String behaviourInterface) throws IOException {
        System.out.println("Source " + sourceDir.getPath());

        String[] canonicalPath = behaviourInterface.split("\\.");

        FileObject packageDir = sourceDir;
        for (int i = 0; i < canonicalPath.length - 1; i++) {
            FileObject nextPack = packageDir.getFileObject(canonicalPath[i]);
            if (nextPack == null) {
                // if directory is missing, create it
                System.out.println("Create " + canonicalPath[i]);
                packageDir = packageDir.createFolder(canonicalPath[i]);
            } else if (nextPack.isValid() && nextPack.isFolder()) {
                // if directory exists, use it
                System.out.println("Traverse " + canonicalPath[i]);
                packageDir = nextPack;
            } else {
                // well, something is wrong, maybe it is a file
                throw new IOException("Part of path to interface behaviour is file(or some other trouble): " + nextPack.getPath());
            }
        }
        String filename = canonicalPath[canonicalPath.length - 1] + ".java";
        FileObject interfaceFile = packageDir.getFileObject(filename);
        if (interfaceFile != null) {
            return interfaceFile;
        }
        return packageDir.createData(filename);
    }

    /**
     * Write interface that contains all actions and behaviours from palette
     * to the file.
     * @param interfaceFile
     * @param behaviourInterface canonical name of created interface file
     */
    private void writeInterface(FileObject interfaceFile, String behaviourInterface) throws FileAlreadyLockedException, IOException {
        String[] splitInterfaceName = behaviourInterface.split("\\.");
        PrintWriter writer = null;
        try {
            writer = new PrintWriter(interfaceFile.getOutputStream());
            writer.write("/* DO NOT EDIT THIS FILE, IT IS AUTO-REGENERATED EVERY TIME THE PALETTE IS CHANGED */\n");

            // if interface is not in default package, write package declaration
            if (splitInterfaceName.length > 1) {
                writer.write("package ");
                for (int i = 0; i < splitInterfaceName.length - 2; i++) {
                    writer.write(splitInterfaceName[i] + ".");
                }
                writer.write(splitInterfaceName[splitInterfaceName.length - 2] + ";\n\n");
            }

            // import annotations
            writer.write("import " + SPOSHAction.class.getCanonicalName() + ";\n");
            writer.write("import " + SPOSHSense.class.getCanonicalName() + ";\n\n");


            // write interface header
            writer.write("public interface " + splitInterfaceName[splitInterfaceName.length - 1] + "{\n" );

            // write declarations of actions
            for (String action : getActions()) {
                writer.write("\n    @SPOSHAction\n    public boolean " + action + "();");
            }
            // write declarations of senses
            for (String sense : getSenses()) {
                writer.write("\n    @SPOSHSense\n    public Object " + sense + "(String param);");
            }

            writer.write("\n}\n" );
        } finally {
            if (writer != null) {
                writer.close();
            }
        }
    }

    private List<String> getActions() {
        PaletteController paletteController = lookup.lookup(PaletteController.class);
        if (paletteController == null)
            throw new RuntimeException("Palette controller not present in the lookup.");
        
        PaletteRoot pal = paletteController.getRoot().lookup(PaletteRoot.class);
        if (pal == null)
            throw new RuntimeException("Palette root not present in the palette controller.");

        return pal.getActionsChildren().getUndefinedActions();
    }

    private List<String> getSenses() {
        PaletteController paletteController = lookup.lookup(PaletteController.class);
        PaletteRoot pal = paletteController.getRoot().lookup(PaletteRoot.class);

        return pal.getSensesChildren().getUndefinedSenses();
    }
}
