package cz.cuni.amis.pogamut.sposh.ut2004;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.util.logging.Level;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.utils.guice.AgentScoped;
import cz.cuni.amis.pogamut.sposh.elements.ParseException;
import cz.cuni.amis.pogamut.sposh.elements.PoshParser;
import cz.cuni.amis.pogamut.sposh.elements.PoshPlan;
import cz.cuni.amis.pogamut.sposh.engine.FireResult;
import cz.cuni.amis.pogamut.sposh.engine.PoshEngine;
import cz.cuni.amis.pogamut.sposh.engine.PoshEngine.EvaluationResultInfo;
import cz.cuni.amis.pogamut.sposh.engine.VariableContext;
import cz.cuni.amis.pogamut.sposh.engine.timer.ITimer;
import cz.cuni.amis.pogamut.sposh.engine.timer.SystemClockTimer;
import cz.cuni.amis.pogamut.sposh.executor.ActionResult;
import cz.cuni.amis.pogamut.sposh.executor.ILogicWorkExecutor;
import cz.cuni.amis.pogamut.sposh.executor.IWorkExecutor;
import cz.cuni.amis.pogamut.sposh.executor.PrimitiveInfo;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004BotLogicController;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;

/**
 * Logic controller that utilizes sposh engine for decision making of bot in UT2004
 * environment.
 * <p/>
 * Sposh requires two things: plan and primitives. The plan is provided by 
 * {@link SposhLogicController#getPlan() } method and are supplied by {@link IWorkExecutor}.
 * {@link IWorkExecutor} is instantiated during first call of logic, so if the varioud
 * modules are already initialized.
 * <p/>
 * If needed, override {@link SposhLogicController#createTimer() }, but it
 * shouldn't be needed.
 * @author Honza H.
 */
@AgentScoped
public abstract class SposhLogicController<BOT extends UT2004Bot, WORK_EXECUTOR extends IWorkExecutor> extends UT2004BotLogicController<BOT> {

    public static final String SPOSH_LOG_CATEGORY = "SPOSH";
    /**
     * Posh engine that is evaluating the plan.
     */
    private PoshEngine engine;
    /**
     * Primitive executor that is executing the primitves when engine requests it,
     * passes the variables and returns the value of executed primitive.
     */
    private WORK_EXECUTOR workExecutor;
    /**
     * Timer for posh engine, for things like timeouts and so on.
     */
    private ITimer timer;

    @Override
    public void initializeController(BOT bot) {
        super.initializeController(bot);
        PoshPlan plan;

        // parse the plan.
        try {
            plan = parsePlan(getPlan());
        } catch (Exception ex) {
            Logger.getLogger(SposhLogicController.class.getName()).log(Level.SEVERE, null, ex);
            throw new RuntimeException(ex);
        }
        // create engine from parsed plan
        engine = createEngine(plan);
    }

    /**
     * Create {@link IWorkExecutor} that will execute primitives contained in the plan.
     * This method will be called only once.
     * @return executor to execute primitives.
     */
    protected abstract WORK_EXECUTOR createWorkExecutor();

    /**
     * Get work executor. If work executor is not yet created, create one using
     * {@link SposhLogicController#createWorkExecutor() }.
     * @return work executor for this bot controller.
     */
    protected final WORK_EXECUTOR getWorkExecutor() {
        if (workExecutor == null) {
            workExecutor = createWorkExecutor();
        }
        return workExecutor;
    }

    /**
     * Logic method evaluates the posh plan every time it is called.
     */
    @Override
    public final void logic() {
        logicBeforePlan();
        if (getEngine().getLog() != null) {
            getEngine().getLog().info("Invoking SPOSH engine.");
        }
        LoggableWorkExecutor loggableWorkExecutor = new LoggableWorkExecutor(getWorkExecutor());
        while (true) {
            EvaluationResultInfo result = getEngine().evaluatePlan(loggableWorkExecutor);
            String lastPrimitive = loggableWorkExecutor.getLastExecutedPrimitive();
            if (result.type != null && (result.type == FireResult.Type.CONTINUE || result.type == FireResult.Type.FOLLOW || result.type == FireResult.Type.FULFILLED
            		                    || result.type == FireResult.Type.SURFACE_CONTINUE || result.type == FireResult.Type.FAILED)) {
                if (getEngine().getLog() != null) {
                    getEngine().getLog().info("Plan evaluation continues...");
                }
                continue;
            }
            break;
        }
        if (getEngine().getLog() != null) {
            getEngine().getLog().info("Plan evaluation end.");
        }
        logicAfterPlan();
    }

    /**
     * Wrapper of some {@link IWorkExecutor}. 
     */
    private static class LoggableWorkExecutor implements IWorkExecutor {
        public static final int DEFAULT_HISTORY_SIZE = 100;
        private final IWorkExecutor workExecutor;
        /**
         * History of all executed primitives
         */
        private final List<String> history;
        /**
         * Unmodifiable wrapper of {@link #history}.
         */
        private final List<String> historyUm;
        /**
         * History of all executed primitives
         */
        private final int historySize;
        /**
         * Create new loggabel action executor with default {@link #DEFAULT_HISTORY_SIZE}.
         */
        public LoggableWorkExecutor(IWorkExecutor workExecutor) {
            this(workExecutor, DEFAULT_HISTORY_SIZE);
        }

        /**
         * Create new loggable action executor wrapper with specified history size.
         * @param workExecutor
         * @param historySize must be greater than 0
         */
        public LoggableWorkExecutor(IWorkExecutor workExecutor, int historySize) {
            this.workExecutor = workExecutor;
            this.history = new LinkedList<String>();
            this.historyUm = Collections.unmodifiableList(this.history);
            this.historySize = historySize;
        }
                
        /**
         * Get history of executed primitives
         * @return Unmodifiable list
         */
        public List<String> getHistory() {
            return this.historyUm;
        }

        /**
         * get last executed primitive. Throws an {@link IndexOutOfBoundsException}
         * if no primitive yet executed.
         * @return Last executed primitive of the executor
         */
        public String getLastExecutedPrimitive() {
            return historyUm.get(0);
        }

		@Override
		public ActionResult executeAction(String actionName, VariableContext ctx) {
			history.add(0, actionName);
            while (history.size() > historySize) {
                history.remove(historySize);
            }

            return workExecutor.executeAction(actionName, ctx);
		}

		@Override
		public Object executeSense(String senseName, VariableContext ctx) {
			history.add(0, senseName);
            while (history.size() > historySize) {
                history.remove(historySize);
            }

            return workExecutor.executeSense(senseName, ctx);
		}
        
    }
        
    /**
     * Method that is triggered every time the plan for executor is evaluated.
     * It is triggered right before the plan evaluation.
     * Currently, it checks if {@link SposhLogicController#workExecutor} is a
     * {@link ILogicWorkExecutor} and if it is, it executes {@link ILogicWorkExecutor#logicBeforePlan() }.
     */
    protected void logicBeforePlan() {
        if (workExecutor instanceof ILogicWorkExecutor) {
            ILogicWorkExecutor logicExecutor = (ILogicWorkExecutor) workExecutor;
            logicExecutor.logicBeforePlan();
        }
    }

    /**
     * Method that is triggered every time the plan for executor is evaluated.
     * It is triggered right after the plan evaluation.
     * Currently, it checks if {@link SposhLogicController#workExecutor} is a
     * {@link ILogicWorkExecutor} and if it is, it executes {@link ILogicWorkExecutor#logicBeforePlan() }.
     */
    protected void logicAfterPlan() {
        if (workExecutor instanceof ILogicWorkExecutor) {
            ILogicWorkExecutor logicExecutor = (ILogicWorkExecutor) workExecutor;
            logicExecutor.logicAfterPlan();
        }
    }

    /**
     * Create timer for posh engine. By default, use {@link SystemClockTimer}.
     * @see SposhLogicController#getTimer() 
     * @return
     */
    protected ITimer createTimer() {
        return new SystemClockTimer();
    }

    /**
     * Get timer that is used by posh engine to make sure timeouts and other stuff
     * that requires time are working properly.
     * @return
     */
    protected final ITimer getTimer() {
        if (timer == null) {
            timer = createTimer();
        }
        return timer;
    }

    /**
     * Parse the supplied plan.
     * @param planSource plan source to be parsed
     * @return parsed plan
     * @throws ParseException if there is an syntax error in the plan.
     */
    private PoshPlan parsePlan(String planSource) throws ParseException {
        StringReader planReader = new StringReader(planSource);
        PoshParser parser = new PoshParser(planReader);
        return parser.parsePlan();
    }

    /**
     * Create posh engine to be used in this class.
     */
    private PoshEngine createEngine(PoshPlan plan) {
        return new PoshEngine(plan, getTimer(), bot.getLogger().getCategory(SPOSH_LOG_CATEGORY));
    }

    /**
     * Get sposh engine for this logic
     * @return null if engine wasn't yet created(is created in {@link SposhLogicController#initializeController(UT2004Bot) } ) or the engine.
     */
    protected final PoshEngine getEngine() {
        return engine;
    }

    /**
     * Get the POSH plan for the bot.
     * Easiest way is to use
     * {@link SposhLogicController#getPlanFromResource(java.lang.String) getPlanFromResource},
     * {@link SposhLogicController#getPlanFromFile(java.lang.String)  getPlanFromFile} or
     * {@link SposhLogicController#getPlanFromStream(java.io.InputStream) getPlanFromStream}.
     *
     * @return Text of the whole plan
     */
    protected abstract String getPlan() throws IOException;

    /**
     * Read POSH plan from the stream and return it. Close the stream.
     * @param in Input stream from which the plan is going to be read
     * @return Text of the plan, basically content of the stream
     * @throws IOException If there is some error while reading the stream
     */
    protected final String getPlanFromStream(InputStream in) throws IOException {
        BufferedReader br = new BufferedReader(new InputStreamReader(in));

        StringBuilder plan = new StringBuilder();
        String line;
        try {
            while ((line = br.readLine()) != null) {
                plan.append(line);
            }
        } finally {
            br.close();
        }

        return plan.toString();
    }

    /**
     * Read POSH plan from the file and return it.
     * @param filename Path to the file that contains the POSH plan.
     * @return Text of the plan, basically content of the file
     * @throws IOException If there is some error while reading the stream
     */
    protected final String getPlanFromFile(String filename) throws IOException {
        FileInputStream f = new FileInputStream(filename);
        return getPlanFromStream(f);
    }

    /**
     * Get POSh plan from resource int the same jar as the class.
     * <p>
     * <pre>
     *  // Plan is stored in package cz.cuni.amis.pogamut.testbot under name poshPlan.lap
     *  // This can get the file from .jar or package structure
     *  getPlanFromResource("cz/cuni/amis/pogamut/testbot/poshPlan.lap");
     * </pre>
     * @param resourcePath Path to the plan in some package
     * @return Content of the plan.
     * @throws IOException if something goes wrong, like file is missing, hardisk has blown up ect.
     */
    protected final String getPlanFromResource(String resourcePath) throws IOException {
        ClassLoader cl = this.getClass().getClassLoader();
        return getPlanFromStream(cl.getResourceAsStream(resourcePath));
    }
}
