/*
 * Copyright (C) 2013 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.pogamut.ut2004.agent.navigation.navmesh;

import cz.cuni.amis.pogamut.base.agent.navigation.IPathFuture;
import cz.cuni.amis.pogamut.base.agent.navigation.IPathPlanner;
import cz.cuni.amis.pogamut.base.agent.navigation.impl.PrecomputedPathFuture;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.base.utils.logging.LogCategory;
import cz.cuni.amis.pogamut.base3d.worldview.object.ILocated;
import cz.cuni.amis.pogamut.base3d.worldview.object.Location;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.DrawStayingDebugLines;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.GameInfo;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPoint;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.NavPointNeighbourLink;
import cz.cuni.amis.pogamut.ut2004.factory.guice.remoteagent.UT2004ServerFactory;
import cz.cuni.amis.pogamut.ut2004.factory.guice.remoteagent.UT2004ServerModule;
import cz.cuni.amis.pogamut.ut2004.server.impl.UT2004Server;
import cz.cuni.amis.pogamut.ut2004.utils.LinkFlag;
import cz.cuni.amis.pogamut.ut2004.utils.UT2004ServerRunner;
import cz.cuni.amis.utils.ExceptionToString;
import cz.cuni.amis.utils.exception.PogamutException;
import java.io.*;
import java.util.*;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.vecmath.Vector2d;
import math.geom2d.Point2D;
import math.geom2d.Shape2D;
import math.geom2d.line.Line2D;
import math.geom2d.line.StraightLine2D;
import math.geom3d.Point3D;
import math.geom3d.Vector3D;
import math.geom3d.line.StraightLine3D;
import math.geom3d.plane.Plane3D;

/**
 * LevelGeometryModule is wrapping the load/save logic for the {@link LevelGeometry}.
 * 
 * @author Jakub Tomek
 * @author Jakub Gemrot aka Jimmy
 */
public class LevelGeometryModule {
	
	private IWorldView worldView;	
	private Logger log;
	
	private Random random;
		
	//
	// STATE
	//
	
	private boolean loaded = false;
	private GameInfo loadedForMap = null;
	
	//
	// NAVMESH DATA STRUCTURES
	//
	
    private LevelGeometry levelGeometry;
	private IWorldObjectEventListener<GameInfo, IWorldObjectEvent<GameInfo>> gameInfoListener = new IWorldObjectEventListener<GameInfo, IWorldObjectEvent<GameInfo>>() {
		@Override
		public void notify(IWorldObjectEvent<GameInfo> event) {
			load(event.getObject());
		}
	};
       
    public LevelGeometryModule(IWorldView worldView, IAgentLogger logger) {
    	if (logger == null) {
    		log = new LogCategory("LevelGeometry");
    	} else {
    		log = logger.getCategory("LevelGeometry");
    	}
    	random = new Random();
    	
    	worldView.addObjectListener(GameInfo.class, WorldObjectFirstEncounteredEvent.class, gameInfoListener);
    	
    	GameInfo info = worldView.getSingle(GameInfo.class);
    	if (info != null) {
    		load(info);
    	}
    }
    
    private void clear() {
    	log.warning("LevelGeometry has been cleared...");
    	
        levelGeometry = null;
        
        loaded = false;
        loadedForMap = null;
    }
    	
    private boolean loadLevelGeometry(String mapName) {
    	// try to read it from processed file        
        String levelGeometryFileName = NavMeshConstants.processedLevelGeometryDir + "\\" + mapName + ".processed";
        File levelGeometryFile = new File(levelGeometryFileName);
        
        log.info("Loading LevelGeometry data for '" + mapName + "'...");
        
        try {
        	if (!levelGeometryFile.exists()) {
        		log.warning("Processed LevelGeometry does not exist at: " + levelGeometryFile.getAbsolutePath());
        	} else {
	            ObjectInputStream in = new ObjectInputStream(new FileInputStream(levelGeometryFile));
	            levelGeometry = (LevelGeometry) in.readObject();  
	            return true;
        	}
        } catch (Exception e) {
        	log.severe(ExceptionToString.process("Previously saved level geometry file could not have been restored.", e));
        }        

        try { 
            levelGeometry = new LevelGeometry(log);
            if (levelGeometry.load(mapName)) {
            	log.info("Level geometry read and built ok.");
            	
	            // save levelgeom for next time
	            try {
	                log.warning("Writing level geometry to a file at: " + levelGeometryFile.getAbsolutePath());
	                levelGeometryFile.getParentFile().mkdirs();
	                ObjectOutputStream out = new ObjectOutputStream(new FileOutputStream(levelGeometryFile));
	                out.writeObject(levelGeometry); 
	                log.info("Level geometry written ok.");
	            } catch (Exception e) {
	            	log.severe(ExceptionToString.process("Exception during writing level geom to a file.", e));
	            }  
	            
	            return true;
            }
            log.severe("COULD NOT INITIALIZE FOR MAP: " + mapName);
            levelGeometry = null;
            return false;
        } catch(Exception e) {
        	log.severe(ExceptionToString.process("Unable to load level geometry files.", e));
            levelGeometry = null;
            return false;
        }        
    }
    
    private void load(GameInfo info) {
    	if (info == null) {
    		log.severe("Could not load for 'null' GameInfo!");
    		return;
    	}
        if (loaded) {
        	if (loadedForMap == null) {
        		// WTF?
        		// => clear
        		clear();
        		// => and reload
        	} else {
        		if (loadedForMap.getLevel().equals(info.getLevel())) {
        			// ALREADY INITIALIZED FOR THE SAME LEVEL
        			return;
        		}
        	}
        }
        
        // LOAD THE LEVEL GEOMETRY FOR THE MAP FROM 'info'
        String mapName = info.getLevel();        
        log.warning("Loading LevelGeometry for '" + mapName + "'...");
    	
        if (loadLevelGeometry(mapName)) {	            
	        loaded = true;
	        loadedForMap = info;
	        return;
        } 
        
        loaded = false;
        loadedForMap = null;
    }
    
    /**
     * Tells, whether the module has been correctly initialized and it is possible to call {@link #getLevelGeometry()} method.  
     * @return
     */
    public boolean isInitialized() {
    	return this.levelGeometry != null;
    }
    
    /**
     * Returns {@link LevelGeometry} object for currently running map.
     * 
     * Non-null iff {@link #isInitialized()}.
     * 
     * @return 
     */
    public LevelGeometry getLevelGeometry() {
        return this.levelGeometry;
    }
    
}
