package cz.cuni.amis.pogamut.ut2004.agent.navigation.stuckdetector;

import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.agent.navigation.IStuckDetector;
import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;

/**
 * Simple {@link IStuckDetector} that watches whether the bot moves at all.
 * <p><p>
 * You may define a period of time (timeout) in which the bot should move on, i.e.,
 * if the bot won't move a bit in a specified amount of time, it will report a stuck.
 * 
 * @author Jimmy
 */
public class UT2004TimeStuckDetector implements IStuckDetector {

	/**
	 * Default distance that is regarded as "bot did not move a bit".
	 */
	private static final double NO_MOVEMENT_SIZE = 10;

	/**
	 * Default timeout after which the detector reports stuck if the bot did not move.
	 */
	private static double DEFAULT_TIMEOUT = 3.0;
	
	/**
	 * Default timeout when the bot is waiting for something...
	 */
	private static double DEFAULT_WAITING_TIMEOUT = 10.0;
	
	/**
	 * Owner of the detector.
	 */
	private UT2004Bot bot;
	
	/**
	 * Timeout used by the detector, when the bot is not waiting, initialized in the constructor.
	 */
	private double timeout;
	
	/**
	 * Timeout used by the detector when the bot is waiting, initialized in the constructor.
	 */
	private double waitingTimeout;
	
	private boolean botWaiting = false;
	
	/**
	 * Listener watching for the {@link Self} message. Recalls {@link UT2004TimeStuckDetector#eventSelf(IWorldObjectEvent)}.
	 * 
	 * @author Jimmy
	 */
	private class SelfListener implements IWorldObjectListener<Self> {

		public SelfListener(IWorldView worldView) {
			worldView.addObjectListener(Self.class, this);
		}
		
		@Override
		public void notify(IWorldObjectEvent<Self> event) {
			eventSelf(event);
		}
		
	};
	
	/**
	 * Listener that triggers {@link UT2004TimeStuckDetector#eventSelf(IWorldObjectEvent)}.
	 */
	private SelfListener selfListener;
	
	/**
	 * Last time when the bot has moved (its velocity had been greater than {@link UT2004TimeStuckDetector#NO_MOVEMENT_SIZE}.
	 */
	private double lastMovementTime = Double.NEGATIVE_INFINITY;

	/**
	 * Whether we should report that the bot has stuck.
	 */
	private boolean stuck = false;

	private Logger log;

	public UT2004TimeStuckDetector(UT2004Bot bot) {
		this(bot, DEFAULT_TIMEOUT, DEFAULT_WAITING_TIMEOUT);
	}
	
	public UT2004TimeStuckDetector(UT2004Bot bot, double timeout, double waitingTimeout) {
		if (this.log == null) {
			this.log = bot.getLogger().getCategory(this.getClass().getSimpleName());
		}
		this.bot = bot;
		this.timeout = timeout;
		this.waitingTimeout = waitingTimeout;
		selfListener = new SelfListener(bot.getWorldView());
	}
	
	public void eventSelf(IWorldObjectEvent<Self> event) {
		if (event.getObject().getVelocity().size() > NO_MOVEMENT_SIZE || lastMovementTime == Double.NEGATIVE_INFINITY) {
			lastMovementTime = event.getObject().getSimTime();
		}
		if (botWaiting) {
			if (event.getObject().getSimTime() - lastMovementTime >= waitingTimeout) {
				stuck = true;
			}
		} else {
			if (event.getObject().getSimTime() - lastMovementTime >= timeout) {
				stuck = true;
			}
		}
	}
	
	@Override
	public void setBotWaiting(boolean state) {
		botWaiting = state;
	}
	
	@Override
	public boolean isStuck() {
		return stuck;
	}

	@Override
	public void reset() {
		lastMovementTime = Double.NEGATIVE_INFINITY;
		stuck = false;
	}

}
