package cz.cuni.amis.pogamut.udk.utils;

import java.util.List;

import cz.cuni.amis.pogamut.base.agent.IAgentId;
import cz.cuni.amis.pogamut.base.agent.impl.AgentId;
import cz.cuni.amis.pogamut.base.agent.params.IAgentParameters;
import cz.cuni.amis.pogamut.base.agent.utils.runner.impl.AgentRunner;
import cz.cuni.amis.pogamut.base.communication.connection.impl.socket.SocketConnectionAddress;
import cz.cuni.amis.pogamut.base.factory.IAgentFactory;
import cz.cuni.amis.pogamut.base.utils.Pogamut;
import cz.cuni.amis.pogamut.base.utils.PogamutPlatform;
import cz.cuni.amis.pogamut.base.utils.PogamutProperty;
import cz.cuni.amis.pogamut.udk.agent.params.UDKAgentParameters;
import cz.cuni.amis.pogamut.udk.bot.IUDKBot;
import cz.cuni.amis.pogamut.udk.bot.IUDKBotController;
import cz.cuni.amis.pogamut.udk.bot.impl.UDKBot;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKBotFactory;
import cz.cuni.amis.pogamut.udk.factory.guice.remoteagent.UDKBotModule;
import cz.cuni.amis.utils.NullCheck;
import cz.cuni.amis.utils.exception.PogamutException;

/**
 * Class used for creating, connecting and starting servers with default settings that are taken from the properties.
 * <p><p>
 * The address where the instances will connect are defined either in the constructor
 * or taken from the properties of the {@link PogamutPlatform}.
 * <p><p>
 * For more information about the class see {@link AgentRunner}.
 * 
 * @author ik
 * @author Jimmy
 *
 * @param <BOT>
 * @param <PARAMS>
 */
public class UDKBotRunner<BOT extends IUDKBot, PARAMS extends UDKAgentParameters> extends AgentRunner<BOT, PARAMS> {

	/**
	 * Default host where the instances are going to be connected as defaults, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
    protected String host;
    
    /**
	 * Default port where the instances are going to be connected as defaults, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
    protected int port;
    
    /**
	 * Default name that will serve as a basis for {@link IAgentId}, see {@link IAgentParameters#assignDefaults(IAgentParameters)}.
	 */
	protected String name;

	/** 
	 * Construct the runner + specify all defaults.
	 * 
	 * @param factory to be used for creating new {@link IUDKBot} instances
	 * @param name default name that serve as a basis for {@link IAgentId}
	 * @param host default host where the instances are going to be connected
	 * @param port default port where the instances are going to be connected
	 */
	public UDKBotRunner(IAgentFactory<BOT, PARAMS> factory, String name, String host, int port) {
        super(factory);
        this.name = name;
        this.port = port;
        this.host = host;
    }

	/**
	 * Construct the runner + specify the default name, host:port will be taken from the Pogamut platform properties.
	 * 
	 * @param factory factory to be used for creating new {@link IUDKBot} instances
	 * @param log used to log stuff
	 * @param name default name that serve as a basis for {@link IAgentId}
	 */
    public UDKBotRunner(IAgentFactory<BOT, PARAMS> factory, String name) {
        this(
        	factory, 
        	name, 
        	Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_HOST.getKey()), 
        	Pogamut.getPlatform().getIntProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_PORT.getKey())
        );
    }
    
    /**
     * Construct the runner without specifying anything as default. Default name for bots will be "UDKBot"
     * and host:port will be taken from the Pogamut platform properties.
     * 
     * @param factory factory to be used for creating new {@link IUDKBot} instances
     */
    public UDKBotRunner(IAgentFactory<BOT, PARAMS> factory) {
        this(factory, "UDKBot");
    }
    
    /** 
	 * Construct the runner + specify all defaults.
	 * 
	 * @param module Guice module that is going to be used by the {@link UDKBotFactory}
	 * @param name default name that serve as a basis for {@link IAgentId}
	 * @param host default host where the instances are going to be connected
	 * @param port default port where the instances are going to be connected
	 */
	public UDKBotRunner(UDKBotModule module, String name, String host, int port) {
        this(new UDKBotFactory<BOT, PARAMS>(module), name, host, port);
    }

	/**
	 * Construct the runner + specify the default name, host:port will be taken from the Pogamut platform properties.
	 * 
	 * @param module Guice module that is going to be used by the {@link UDKBotFactory}
	 * @param name default name that serve as a basis for {@link IAgentId}
	 */
    public UDKBotRunner(UDKBotModule module, String name) {
        this(
        	module, 
        	name, 
        	Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_HOST.getKey()), 
        	Pogamut.getPlatform().getIntProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_PORT.getKey())
        );
    }
    
    /**
     * Construct the runner without specifying anything as default. Default name for bots will be "UDKBot"
     * and host:port will be taken from the Pogamut platform properties.
     * 
	 * @param module Guice module that is going to be used by the {@link UDKBotFactory}
	 */
    public UDKBotRunner(UDKBotModule module) {
        this(module, "UDKBot");
    }
    
    /** 
	 * Construct the runner + specify all defaults.
	 * 
	 * @param botControllerClass controller that will be used to instantiate {@link UDKBotModule}, i.e., it will control the {@link UDKBot} instance
	 * @param name default name that serve as a basis for {@link IAgentId}
	 * @param host default host where the instances are going to be connected
	 * @param port default port where the instances are going to be connected
	 */
	public UDKBotRunner(Class<? extends IUDKBotController> botControllerClass, String name, String host, int port) {
        this(new UDKBotModule(botControllerClass), name, host, port);
    }

	/**
	 * Construct the runner + specify the default name, host:port will be taken from the Pogamut platform properties.
	 * 
	 * @param botControllerClass controller that will be used to instantiate {@link UDKBotModule}, i.e., it will control the {@link UDKBot} instance
	 * @param name default name that serve as a basis for {@link IAgentId}
	 */
    public UDKBotRunner(Class<? extends IUDKBotController> botControllerClass, String name) {
        this(
        	new UDKBotModule(botControllerClass), 
        	name, 
        	Pogamut.getPlatform().getProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_HOST.getKey()), 
        	Pogamut.getPlatform().getIntProperty(PogamutUDKProperty.POGAMUT_UDK_BOT_PORT.getKey())
        );
    }
    
    /**
     * Construct the runner without specifying anything as default. Default name for bots will be "UDKBot"
     * and host:port will be taken from the Pogamut platform properties.
     * 
	 * @param botControllerClass controller that will be used to instantiate {@link UDKBotModule}, i.e., it will control the {@link UDKBot} instance
	 */
    public UDKBotRunner(Class<? extends IUDKBotController> botControllerClass) {
        this(new UDKBotModule(botControllerClass), "UDKBot");
    }
    
    @Override
    public BOT startAgent() throws PogamutException {
    	return super.startAgent();
    }
    
    @Override
    public List<BOT> startAgents(int count) throws PogamutException {
    	return super.startAgents(count);
    }
    
    @Override
    public List<BOT> startAgents(PARAMS... agentParameters) throws PogamutException {
    	return super.startAgents(agentParameters);
    };
    
    /**
     * Returns name that is going to be used to form new {@link IAgentId} of the bots.
     *     
     * @return name used for the newly started bots
     */
    public String getName() {
		return name;
	}

    /**
     * Sets name that is going to be used to form new {@link IAgentId} of the bots.
     * <p><p>
     * If null is passed, generic "UDKBot" will be set.
     *     
     * @param name name used for the newly started bots
     * @return this instance
     */
	public UDKBotRunner<BOT, PARAMS> setName(String name) {
		if (name == null) name = "UDKBot";
		this.name = name;
		return this;
	}

	/**
     * Returns host, where newly launched bots will be connected to.
     * 
     * @return host running GBUDK server
     */
    public String getHost() {
		return host;
	}

    /**
     * Sets host, where newly launched bots will be connected to.
     * 
     * @param host host running GBUDK server (can't be null)
     * @return this instance
     */
	public UDKBotRunner<BOT, PARAMS> setHost(String host) {
		this.host = host;
		NullCheck.check(this.host, "host");
		return this;
	}

	/**
     * Returns port, where newly launched bots will be connected to.
     * 
     * @return port at the host where GBUDK server is listening for bot connections
     */
	public int getPort() {
		return port;
	}

	/**
     * Sets port, where newly launched bots will be connected to.
     * 
     * @param port at the host where GBUDK server is listening for bot connections
     * @return this instance
     */
	public UDKBotRunner<BOT, PARAMS> setPort(int port) {
		this.port = port;
		return this;
	}

    /**
     * Provides default parameters that is, {@link IAgentId} using {@link UDKBotRunner#name} and {@link SocketConnectionAddress}
     * using {@link UDKBotRunner#host} and {@link UDKBotRunner#port}.
     */
	@Override
	protected IAgentParameters newDefaultAgentParameters() {
		return new UDKAgentParameters().setAgentId(new AgentId(name)).setWorldAddress(new SocketConnectionAddress(host, port));
	}
	
}
