package cz.cuni.amis.pogamut.emohawk.agent.module.sensomotoric;

import java.util.HashMap;

import cz.cuni.amis.pogamut.base.agent.module.SensomotoricModule;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.unreal.communication.messages.UnrealId;
import cz.cuni.amis.pogamut.ut2004.bot.impl.UT2004Bot;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbcommands.SetEmoticon;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Player;
import cz.cuni.amis.pogamut.ut2004.communication.messages.gbinfomessages.Self;

/**
 * Wraps emoticon control.
 * @author Knight
 */
public class Emoticons extends SensomotoricModule<UT2004Bot> {
    
    /** Here we store pointer to our self object */
    private Self self = null;

    /** default duration of emoticons in seconds */
    private static double DEFAULT_DURATION = 5;

    /** default bubble for emoticons */
    private static EmoticonBubbleType DEFAULT_BUBBLE = EmoticonBubbleType.BUBBLE_THOUGHT_CENTER;
    /** We use this to track the times players changed any of their emoticons and the emoticons themselves */
    HashMap<UnrealId, EmoticonInfo> playersEmoticons = new HashMap();

    /**
     * Returns all 3 emoticons as currently displayed by you (displayed from LEFT-to-RIGHT).
     * 
     * 0 emoticons displayed: returns []
     * 
     * 1 emoticon displayed: returns [CENTER]
     * 
     * 2 emoticons displayed: returns [LEFT, RIGHT]
     * 
     * 3 emoticons displayed: returns [LEFT, CENTER, RIGHT]
     * 
     * @return array of [left, center, right] emoticon being displayed, NONE emoticons are omitted!
     */
    public EmoticonType[] getCurrentEmoticonTypes() {
    	return getCurrentEmoticon().getEmoticonTypes();
    }
    
    /**
     * Returns current bot emoticon.
     * @return
     */
    public Emoticon getCurrentEmoticon() {
        if (self == null) return new Emoticon(EmoticonType.NONE, EmoticonBubbleType.NONE);
        return new Emoticon(EmoticonType.get(self.getEmotLeft()), EmoticonType.get(self.getEmotCenter()), EmoticonType.get(self.getEmotRight()), EmoticonBubbleType.get(self.getBubble()));
    }

    /**
     * Gets the current emoticon of input player from input Player object.
     * @param player
     * @return
     */
    public Emoticon getPlayerEmoticon(Player player) {
        return new Emoticon(EmoticonType.get(player.getEmotLeft()), EmoticonType.get(player.getEmotCenter()), EmoticonType.get(player.getEmotRight()), EmoticonBubbleType.get(player.getBubble()));
    }

    /**
     * Returns emoticon info for player of input id. Not only you get emoticon, but
     * also the SimTime long when the emoticon was set! If no value, blank emoticon
     * and 0 time will be returned.
     * @param playerId
     * @return
     */
    public EmoticonInfo getPlayerEmoticonInfo(UnrealId playerId) {
        if (playersEmoticons.containsKey(playerId))
            return playersEmoticons.get(playerId);
        else 
            return new EmoticonInfo(new Emoticon(EmoticonType.NONE, EmoticonType.NONE, EmoticonType.NONE, EmoticonBubbleType.NONE), 0);
    }

    /**
     * Sets emoticon for the bot for input duration (seconds).
     * @param emoticon
     */
    public void setEmoticon(Emoticon emoticon) {
        if (emoticon.getEmoticonCount() == 3)
            setTripleEmoticon(emoticon.getLeftEmoticon(), emoticon.getCenterEmoticon(), emoticon.getRightEmoticon(), DEFAULT_DURATION, emoticon.getBubble());
        else if (emoticon.getEmoticonCount() == 2)
            setDoubleEmoticon(emoticon.getLeftEmoticon(), emoticon.getRightEmoticon(), DEFAULT_DURATION, emoticon.getBubble());
        else
            setCenterEmoticonType(emoticon.getCenterEmoticon(), DEFAULT_DURATION, emoticon.getBubble());
    }

    /**
     * Sets emoticon for the bot for input duration (seconds).
     * @param emoticon
     * @param duration
     */
    public void setEmoticon(Emoticon emoticon, double duration) {
        if (emoticon.getEmoticonCount() == 3)
            setTripleEmoticon(emoticon.getLeftEmoticon(), emoticon.getCenterEmoticon(), emoticon.getRightEmoticon(), duration, emoticon.getBubble());
        else if (emoticon.getEmoticonCount() == 2)
            setDoubleEmoticon(emoticon.getLeftEmoticon(), emoticon.getRightEmoticon(), duration, emoticon.getBubble());
        else
            setCenterEmoticonType(emoticon.getCenterEmoticon(), duration, emoticon.getBubble());
    }

    /**
     * Returns current bot left emoticon.
     * @return
     */
    public EmoticonType getCurrentLeftEmoticonType() {
        if (self == null) return EmoticonType.NONE;
        /* TODO - uncomment when new messages genereted
        if (EmoticonType.has(self.getEmotLeft()))
            return EmoticonType.get(self.getEmotLeft());
        */
        return EmoticonType.NONE;
    }

    /**
     * Returns current bot center emoticon.
     * @return
     */
    public EmoticonType getCurrentCenterEmoticonType() {
        if (self == null) return EmoticonType.NONE;
       
        if (EmoticonType.has(self.getEmotCenter()))
            return EmoticonType.get(self.getEmotCenter());
        return EmoticonType.NONE;
    }

    /**
     * Returns current bot right emoticon.
     * @return
     */
    public EmoticonType getCurrentRightEmoticonType() {
        if (self == null) return EmoticonType.NONE;
        if (EmoticonType.has(self.getEmotRight()))
            return EmoticonType.get(self.getEmotRight());

        return EmoticonType.NONE;
    }

    /**
     * Returns current bot bubble.
     * @return
     */
    public EmoticonBubbleType getCurrentBubbleType() {
        if (self == null) return EmoticonBubbleType.NONE;
        if (EmoticonBubbleType.has(self.getBubble())) {
            return EmoticonBubbleType.get(self.getBubble());
        }
        return EmoticonBubbleType.NONE;
    }
    
    /**
     * May set multiple emoticons at once.
     * 
     * Can be invoked with 0 / 1 / 2 / 3 emoticon types.
     * 
     * 0 emoticons - stops displaying all emoticons
     * 
     * 1 emoticon - display emoticon as "center" one.
     * 
     * 2 emoticons - display emoticons as "left" and "center" ones respectively.
     * 
     * 3 emoticons - display emoticons as "left" and "center" and "right" ones respectively.
     * 
     * @param duration
     * @param bubble
     * @param emoticons
     */
    public void setEmoticons(double duration, EmoticonBubbleType bubble, EmoticonType... emoticons) {
    	clearEmoticons();
    	if (emoticons == null || emoticons.length == 0) return;
    	switch(emoticons.length) {
    	case 1:
    		setCenterEmoticonType(emoticons[0], duration, bubble);
    		return;
    	case 2:
    		setDoubleEmoticon(emoticons[0], emoticons[1], duration, bubble);
    		return;
    	case 3:
    		setTripleEmoticon(emoticons[0], emoticons[1], emoticons[3], duration, bubble);
    		return;
    	default:
    		log.warning("Unsupported number of emoticons to show (" + emoticons.length + "), cannot show more than 3, discarding the rest.");
    		setTripleEmoticon(emoticons[0], emoticons[1], emoticons[3], duration, bubble);
    		return;
    	}    	
    }

    /**
     * Stops displaying all emoticons.
     */
    public void clearEmoticons() {
    	// TODO!!!
    }
    
    /**
     * Sets emoticon in the center with default bubble and default duration.
     * @param type
     */
    public void setCenterEmoticonType(EmoticonType type) {
        setCenterEmoticonType(type, DEFAULT_DURATION, DEFAULT_BUBBLE);
    }

    /**
     * Sets emoticon in the center for input duration (seconds).
     * @param type
     * @param duration
     */
    public void setCenterEmoticonType(EmoticonType type, double duration) {
        setCenterEmoticonType(type, duration, DEFAULT_BUBBLE);
    }

    /**
     * Sets emoticon in the center for input duration (seconds) with input bubble.
     * @param centerEmoticon
     * @param duration
     * @param bubble
     */
    public void setCenterEmoticonType(EmoticonType centerEmoticon, double duration, EmoticonBubbleType bubble) {
        this.act.act(new SetEmoticon(centerEmoticon.id, 1, null, 0, null, 0, bubble.id, 1, duration));
    }

    /**
     * Sets double emoticon - left and right for the default duration with default bubble.
     * @param left
     * @param right
     */
    public void setDoubleEmoticon(EmoticonType left, EmoticonType right) {
        setDoubleEmoticon(left, right, DEFAULT_DURATION);
    }

    /**
     * Sets double emoticon - left and right for input duration with default bubble.
     * @param left
     * @param right
     * @param duration
     */
    public void setDoubleEmoticon(EmoticonType left, EmoticonType right, double duration) {
        setDoubleEmoticon(left, right, duration, DEFAULT_BUBBLE);
    }

    /**
     * Sets double emoticon - left and right for input duration with input bubble.
     * @param left
     * @param right
     * @param duration
     * @param bubble
     */
    public void setDoubleEmoticon(EmoticonType left, EmoticonType right, double duration, EmoticonBubbleType bubble) {
        this.act.act(new SetEmoticon(null, 0, left.id, 1, right.id, 1, bubble.id, 1, duration));
    }

    /**
     * Sets triple emoticon - left, center and right for default duration with default bubble.
     * @param left
     * @param center
     * @param right
     */
    public void setTripleEmoticon(EmoticonType left, EmoticonType center, EmoticonType right) {
        setTripleEmoticon(left, center, right, DEFAULT_DURATION);
    }

    /**
     * Sets triple emoticon - left, center and right for input duration with default bubble.
     * @param left
     * @param center
     * @param right
     * @param duration
     */
    public void setTripleEmoticon(EmoticonType left, EmoticonType center, EmoticonType right, double duration) {
        setTripleEmoticon(left, center, right, duration, DEFAULT_BUBBLE);
    }

    /**
     * Sets triple emoticon - left, center and right for input duration with input bubble.
     * @param left
     * @param center
     * @param right
     * @param duration
     * @param bubble
     */
    public void setTripleEmoticon(EmoticonType left, EmoticonType center, EmoticonType right, double duration, EmoticonBubbleType bubble) {
        this.act.act(new SetEmoticon(center.id, 1, left.id, 1, right.id, 1, bubble.id, 1, duration));
    }

    /**
     * Listener for Self object - stores Self object into our link.
     */
    IWorldObjectListener<Self> storeSelfListener = new IWorldObjectListener<Self>() {
        @Override
        public void notify(IWorldObjectEvent<Self> event) {
            if (self == null) {
                self = event.getObject();
                worldView.removeObjectListener(Self.class, this);
                storeSelfListener = null;
            }
        }
    };

    /**
     * Player object updated listener. Here we update playersEmoticons map and generate
     * new emoticon worldView events.
     */
    IWorldObjectEventListener<Player, WorldObjectUpdatedEvent<Player>> playersListener = new IWorldObjectEventListener<Player, WorldObjectUpdatedEvent<Player>>() {
        @Override
        public void notify(WorldObjectUpdatedEvent<Player> event) {
            Player upPl = event.getObject();
            Emoticon actualEmot = getPlayerEmoticon(upPl);
            if (playersEmoticons.containsKey(upPl.getId())) {                
                if (!playersEmoticons.get(upPl.getId()).getEmoticon().equals(actualEmot) ) {
                    playersEmoticons.put(upPl.getId(), new EmoticonInfo(actualEmot, event.getSimTime()));
                    //raise new event in the worldView - new emoticon arrived!
                    worldView.notify(new NewEmoticonEvent(upPl.getId(), actualEmot, event.getSimTime()));
                }            
            } else {
                playersEmoticons.put(upPl.getId(), new EmoticonInfo(actualEmot, event.getSimTime()));
            }
        }
    };

    /**
     * Default constructor.
     * @param agent
     */
    public Emoticons(UT2004Bot agent) {
        super(agent);
        agent.getWorldView().addObjectListener(Self.class, storeSelfListener);
        agent.getWorldView().addObjectListener(Player.class, WorldObjectUpdatedEvent.class, playersListener);
    }

}
