package cz.cuni.amis.pogamut.defcon.communication.worldview.modules.managers.buildings;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.SortedMap;
import java.util.TreeMap;

import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectDestroyedEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.event.WorldObjectFirstEncounteredEvent;
import cz.cuni.amis.pogamut.defcon.agent.impl.DefConAgentLogicController;
import cz.cuni.amis.pogamut.defcon.agent.impl.ILogicUpdateListener;
import cz.cuni.amis.pogamut.defcon.ai.buildingai.IBuildingAI;
import cz.cuni.amis.pogamut.defcon.base3d.worldview.object.DefConLocation;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.DefConCommand;
import cz.cuni.amis.pogamut.defcon.communication.messages.commands.PlaceStructure;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.AirBase;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.DefConUnitObject;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.Radar;
import cz.cuni.amis.pogamut.defcon.communication.messages.infos.Silo;
import cz.cuni.amis.pogamut.defcon.consts.UnitType;

/**
 * Manages all known buildings. Be they either own or enemies. Also takes care
 * of updating of all you buildings AIs.
 * 
 * @author Radek 'Black_Hand' Pibil
 * 
 */
public class BuildingsManager {

	protected final ArrayList<BuildingWithAI<?>> ownBuildings = new ArrayList<BuildingWithAI<?>>();
	protected final SortedMap<Integer, List<DefConUnitObject<?>>> enemyBuildings = new TreeMap<Integer, List<DefConUnitObject<?>>>();
	protected final LinkedList<PlaceStructure> placementQueue = new LinkedList<PlaceStructure>();

	protected int placementsCounter = 0;

	protected static final int PLACEMENT_PER_TICK = 2;

	protected static final long PLACEMENT_TIMEOUT = 1000;

	protected long lastPlacement;

	protected final ILogicUpdateListener logicListener = new ILogicUpdateListener() {

		@Override
		public void update() {
			int counter = 0;

			if (!placementQueue.isEmpty()) {

				do {

					if (counter > PLACEMENT_PER_TICK)
						break;

					PlaceStructure placement = placementQueue.pollFirst();

					act(placement);

					++counter;
				} while (!placementQueue.isEmpty());
			}

			if (placementsCounter != ownBuildings.size()
					&& System.currentTimeMillis() - lastPlacement > PLACEMENT_TIMEOUT) {
				placementsCounter = ownBuildings.size();
			}

			for (BuildingWithAI<?> building : ownBuildings) {
				IBuildingAI ai = building.getAI();

				if (ai != null && !building.getBuilding().getDestroyed()) {
					ai.update();
				}
			}
		}
	};

	protected DefConAgentLogicController<?> logic;

	protected IWorldObjectListener<DefConUnitObject<?>> buildingListener =
			new IWorldObjectListener<DefConUnitObject<?>>() {

				@Override
				public void notify(IWorldObjectEvent<DefConUnitObject<?>> event) {
					if (event instanceof WorldObjectFirstEncounteredEvent<?>) {
						addBuilding(event.getObject());
					}
					if (event instanceof WorldObjectDestroyedEvent<?>) {
						removeBuilding(event.getObject());
					}
				}
			};

	/**
	 * Building and AI pair.
	 * 
	 * @author Radek 'Black_Hand' Pibil
	 * 
	 * @param <T>
	 */
	public class BuildingWithAI<T extends DefConUnitObject<?>> {
		T building;
		IBuildingAI ai;

		public BuildingWithAI(T building, IBuildingAI ai) {
			this.building = building;
			this.ai = ai;
		}

		public final IBuildingAI getAI() {
			return ai;
		}

		public final void setAI(IBuildingAI ai) {
			this.ai = ai;
		}

		public final T getBuilding() {
			return building;
		}
	}

	public BuildingsManager(DefConAgentLogicController<?> logic) {

		this.logic = logic;

		for (int teamId : logic.getGameInfo().getEnemyTeamIds()) {
			enemyBuildings.put(teamId, new ArrayList<DefConUnitObject<?>>());
		}

		logic.getWorldView().addObjectListener(AirBase.class, buildingListener);
		logic.getWorldView().addObjectListener(Silo.class, buildingListener);
		logic.getWorldView().addObjectListener(Radar.class, buildingListener);
		logic.addGameLogicListener(logicListener);

	}

	protected void addBuilding(DefConUnitObject<?> building) {
		if (building.getTeamId() == logic.getGameInfo().getOwnTeamId()) {
			addOwnBuilding(building);
		} else {
			addEnemyBuilding(building);
		}
	}

	protected void removeBuilding(DefConUnitObject<?> building) {
		if (building.getTeamId() == logic.getGameInfo().getOwnTeamId()) {
			removeOwnBuilding(building);
		} else {
			removeEnemyBuilding(building);
		}
	}

	protected void addOwnBuilding(DefConUnitObject<?> building) {
		if (!ownBuildings.contains(building)) {
			ownBuildings.add(new BuildingWithAI<DefConUnitObject<?>>(building,
					null));
			lastPlacement = System.currentTimeMillis();
		}
	}

	protected void removeOwnBuilding(DefConUnitObject<?> building) {
		Iterator<BuildingWithAI<?>> ownBuildingsIterator = ownBuildings
				.iterator();

		while (ownBuildingsIterator.hasNext()) {
			BuildingWithAI<?> ownBuilding = ownBuildingsIterator.next();

			if (ownBuilding.building != building)
				continue;

			ownBuildingsIterator.remove();

			if (ownBuilding.getAI() != null) {
				ownBuilding.getAI().dispose();
				ownBuilding.setAI(null);
				return;
			}

		}
	}

	protected void addEnemyBuilding(DefConUnitObject<?> building) {

		int enemyId = building.getTeamId();
		List<DefConUnitObject<?>> singleEnemyBuildings = enemyBuildings
				.get(enemyId);

		if (!singleEnemyBuildings.contains(building))
			singleEnemyBuildings.add(building);
	}

	protected void removeEnemyBuilding(DefConUnitObject<?> building) {

		int enemyId = building.getTeamId();
		List<DefConUnitObject<?>> singleEnemyBuildings = enemyBuildings
				.get(enemyId);

		if (singleEnemyBuildings.contains(building))
			singleEnemyBuildings.remove(building);

	}

	/**
	 * Returns a list of all own buildings with their AIs.
	 * 
	 * @return
	 */
	public List<BuildingWithAI<?>> getOwnBuildings() {
		return Collections.unmodifiableList(ownBuildings);
	}

	/**
	 * Returns a list of all enemy buildings.
	 * 
	 * @return
	 */
	public SortedMap<Integer, List<DefConUnitObject<?>>> getEnemyBuildings() {
		return Collections.unmodifiableSortedMap(enemyBuildings);
	}

	/**
	 * Returns the list of enemy buildings of the enemy of given id.
	 * 
	 * @return
	 */
	public List<DefConUnitObject<?>> getSingleEnemyBuildings(int enemyId) {
		return Collections.unmodifiableList(enemyBuildings.get(enemyId));
	}

	public static boolean isBuilding(DefConUnitObject<?> unitObject) {
		return unitObject instanceof AirBase ||
				unitObject instanceof Radar ||
				unitObject instanceof Silo;
	}

	public static boolean isBuilding(UnitType type) {
		return type == UnitType.AIR_BASE || type == UnitType.RADAR
				|| type == UnitType.SILO;
	}

	/**
	 * Tries to place buildings on all given buildings location.
	 * 
	 * @param locations
	 * @param type
	 */
	public void placeBuildings(List<DefConLocation> locations, UnitType type) {
		if (!BuildingsManager.isBuilding(type))
			return;

		for (DefConLocation location : locations) {
			placementQueue.add(new PlaceStructure(type.id, location));
		}
	}

	/**
	 * Assigns AI to the given building.
	 * 
	 * @param building
	 * @param ai
	 * @return
	 */
	public boolean assignAI(DefConUnitObject<?> building, IBuildingAI ai) {
		for (BuildingWithAI<?> buildingWithAI : ownBuildings) {
			if (buildingWithAI.equals(building)) {
				buildingWithAI.setAI(ai);
				return true;
			}
		}

		return false;
	}

	protected void act(DefConCommand command) {
		++placementsCounter;
		lastPlacement = System.currentTimeMillis();
		logic.act(command);
	}

	/**
	 * Returns true if all attempts to place buildings were processed.
	 * 
	 * @return
	 */
	public boolean isFinished() {
		return placementQueue.isEmpty()
				&& placementsCounter == ownBuildings.size();
	}
}
