/* 
 * DEFCON API Java Bot
 * This is the java extension to the DEFCON AI API.
 * Please see the project homepage for more general information about the api:
 * http://www.doc.ic.ac.uk/~rb1006/projects:api and http://www.introversion.co.uk/defcon/bots/
 * 
 * By default, a JDK compiler compliance level of java version 1.6 is expected by the API. 
 * By giving the command line option javaversion="1.4", java 1.4 compliance can be activated.
 * 
 * The functions called by the api are initialise, update, and addEvent.
 * 
 * Feel free to modify and change the code as you see fit.
 * 
 * Author: Robin Baumgarten (see website above for contact details)
 * February 2009
 * 
 */

package javabot;

import java.util.LinkedList;
import java.util.List;


/**
 * Based on DefCon JavaBot v0.9
 * 
 * @author Robin Baumgarten
 * @author Jimmy
 * @author Radek 'Black_Hand' Pibil
 */
public class JBot {
	// the following declarations are functions implemented in the C++ API of DEFCON,
	// they have been exposed by the Java Native Interface. For more information on what
	// each function does, please refer to the function list at the project homepage:
	// http://www.doc.ic.ac.uk/~rb1006/projects:api:documentation
	
	/**
	 * Current Defcon Stage, game starts with 5
	 * @return int Defcon stage
	 */
    public static native int GetDefcon();
    /**
     * Current Game Time, measured in seconds. Each tick, the game progresses by 0.1 sec * GameSpeed
     * @return game time
     */
    public static native float GetGameTime();
    /**
     * Amount of update cycles (ticks) passed since game start
     * @return number of game ticks
     */
    public static native int GetGameTick();
    /**
     * Current speed-up factor of the game over the real time passed. Usually has values
     * from 0 (paused), 1 (real time), 5, 10, 20, see enum GAMESPEED (GameSpeed class)
     * @return
     */
    public static native int GetGameSpeed();
    /**
     * Time remaining in game, if victory timer was started. Test this with 
     * IsVictoryTimerStarted
     * @return
     */
    public static native float GetVictoryTimer();
    /**
     * True iff the victory-timer has been started
     * @return
     */
    public static native boolean IsVictoryTimerActive();
    /**
     * Value of certain option
     * @param name
     * @return
     */
    public static native int GetOptionValue( String name );
    /**
     * Array of City Ids. The amount of cities does not change during a game.
     * @return
     */
    public static native int[] GetCityIds();
    /**
     * Population (in millions)
     * @param cityId
     * @return
     */
    public static native int GetCityPopulation( int cityId );
    /**
     * Remaining population of given team
     * @param teamId
     * @return
     */
    public static native int GetRemainingPopulation( int teamId );
    /**
     * True if the given coordinates belong to the given Team. If seaArea is set to true, then
     * Coordinates must be on sea area, otherwise land. If teamId = -1, then function 
     * returns if coordinates are land or sea terrain respectively. Note that there can be 
     * coordinates which are neither land nor sea
     * @param teamId
     * @param longitude
     * @param latitude
     * @param seaArea
     * @return
     */
    public static native boolean IsValidTerritory( int teamId, float longitude, float latitude, boolean seaArea );
    /**
     * True if given coordinates are on the border. Compare "data/earth/coastlines.bmp".
     * @param longitude
     * @param latitude
     * @return
     */
    public static native boolean IsBorder( float longitude, float latitude );
    /**
     * Territory Id of territory at given coordinates.
     * @param longitude
     * @param latitude
     * @return
     */
    public static native int GetTerritoryId( float longitude, float latitude );
    /**
     * Own team id.
     * @return
     */
    public static native int GetOwnTeamId();
    /**
     * List of Team Ids in the game.
     * @return
     */
    public static native int[] GetTeamIds();
    /**
    * Number of territories for given team, usually 1.
    */        
    public static native int GetTeamTerritoriesCount( int teamId );
    /**
     * 
     * Territory Ids of territories that the given team owns. The enum TERRITORY_* relates 
     * the ids to starting positions
     * @param teamId
     * @return territories belonging to the team
     */
    public static native int[] GetTeamTerritories( int teamId );
    /**
     * Id of alliance. Each team belongs to exactly one alliance.
     * @param teamId
     * @return
     */
    public static native int GetAllianceId( int teamId );
    /**
     * Currently requested game speed of given team.
     * @param teamId
     * @return
     */
    public static native int GetDesiredGameSpeed( int teamId );
    /**
     * Sum of enemy kills of the given team (for scoring).
     * @param teamId
     * @return
     */
    public static native int GetEnemyKills( int teamId );
    /**
     * Sum of friendly deaths (deaths in allied populations) of the given team.
     * @param teamId
     * @return
     */
    public static native int GetFriendlyDeaths( int teamId );
    /**
     * Sum of collateral damage deaths (deaths in own population) of the given team.
     * @param teamId
     * @return
     */
    public static native int GetCollateralDamage( int teamId );
    /**
     * Name of the given team.
     * @param teamId
     * @return
     */
    public static native String GetTeamName( int teamId );
    /**
     * True iff the first team is sharing its radar with the second team.
     * @param teamId1
     * @param teamId2
     * @return
     */
    public static native boolean IsSharingRadar( int teamId1, int teamId2 );
    /**
     * True iff the first team is in cease fire mode with the second team.
     * @param teamId1
     * @param teamId2
     * @return
     */
    public static native boolean IsCeaseFire( int teamId1, int teamId2 );
    /**
     * Sends requests to the alliance members to join alliance. Replies are handled by the
     * event system.
     * @param allianceId
     */
    public static native void RequestAlliance( int allianceId );
    /**
     * Send request to cease fire with given team.
     * @param teamId
     */
    public static native void RequestCeaseFire( int teamId );
    /**
     * Send request to share radar with given team.
     * @param teamId
     */
    public static native void RequestShareRadar( int teamId );
    /**
     * Send request to change game speed to given speed. Must be one of the values 
     * specified in GAMESPEED.
     * @param requestedSpeedIdentifier
     */
    public static native void RequestGameSpeed( int requestedSpeedIdentifier );
    /**
     * All visible unit ids.
     * @return
     */
    public static native int[] GetAllUnits();
    /**
     * All own unit ids.
     * @return
     */
    public static native int[] GetOwnUnits();
    /**
     * All visible units of a given team.
     * @param teamId
     * @return
     */
    public static native int[] GetTeamUnits( int teamId );
    /**
     * Data about all visible units, contained in the struct unitData (see enums). This 
     * function is for convenience only, as all data can be gathered through other functions, 
     * too.
     * @return
     */
    private static native float[] _GetAllUnitData();
    /**
     * Type of unit, event or team, specified in enum TYPE_*, EVENT_* or TEAM_TYPE_*.
     * @param id
     * @return
     */
    public static native int GetType( int id );
    /**
     * Team Id of given unit.
     * @param id
     * @return
     */
    public static native int GetTeamId( int id );
    /**
     * Own fleet ids.
     * @return
     */
    public static native int[] GetOwnFleets();
    /**
     * Fleet ids of given team. Only fleets ids with visible members are returned.
     * @param teamId
     * @return
     */
    public static native int[] GetFleets( int teamId );
    /**
     * Ids of ships in given fleet.
     * @param _fleetId
     * @return
     */
    public static native int[] GetFleetMembers( int _fleetId );
    /**
     * Id of fleet of given unit.
     * @param unitId
     * @return
     */
    public static native int GetFleetId( int unitId );
    
    ///**
    // * Ids of all visible shots (projectiles like gunfire and depth charges).
    // * @return
    // */    
    //public static native int[] GetShots();
    ///**
    // * UnitId of originator of given shot, if visible at shooting time
    // * @param shotId
    // * @return
    // */
    //public static native int GetShotOrigin(int shotId);
    ///**
    // * Location where shot has been first seen.
    // * @param shotId
    // * @return
    // */
    //public static native int[] GetShotOriginLocation(int shotId);
    /**
     * State of unit, specified in enum STATE_*.
     * @param unitId
     * @return
    */
    public static native int GetCurrentState( int unitId );
    ///**
    // * Number of activations of current state in given unit, i.e., number of nukes in a sub or 
    // * silo, number of planes available in a carrier or airbase
    // */
    // TODO: MISSING!
    // public static native int GetCurrentStateCount( int unitId );
    /**
     * Number of activations of given state in given unit
     * @param unitId
     * @param stateId
     * @return
     */
    public static native int GetStateCount( int unitId, int stateId );
    /**
     * Time until current state is active.
     * @param unitId
     * @return
     */
    public static native float GetStateTimer( int unitId );
    /**
     * Array of unitIds of currently queued actions, for example nukes in a silo or planes on a 
     * carrier
     * @param unitId
     * @return
     */
    public static native int[] GetActionQueue( int unitId ); 
    /**
     * Current target id. -1 if no target is set or target is location. If track of target is lost, the 
     * last known location is used instead
     * @param unitId
     * @return
     */
    public static native int GetCurrentTargetId( int unitId );
    /**
     * Current target location. (0,0) if no target location is set.
     * @param unitId
     * @return
     */
    public static native float[] GetMovementTargetLocation( int unitId );
    /**
     * Number of available nukes.
     * @param _unitId
     * @return
     */
    public static native int GetNukeSupply( int _unitId );
    /**
     * Target of the nuke carried by given bomber.
     * @param _unitId
     * @return
     */
    public static native float[] GetBomberNukeTarget( int _unitId );
    /**
     * True iff given unit is automatically retaliating an earlier attack.
     * @param _unitId
     * @return
     */
    public static native boolean IsRetaliating( int _unitId );
    /**
     * True iff given unit is visible to given team. In full information mode, visibility 
     * information about other teams is available. In limited information mode, only visible 
     * units are accessible.
     * @param _unitId
     * @param _byTeamId
     * @return
     */
    public static native boolean IsVisible( int _unitId, int _byTeamId );
    /**
     * Set state of given unit. See STATE_*
     * @param unitId
     * @param stateId
     * @return
     */
    public static native void SetState( int unitId, int stateId );
    /**
     * TODO: do the movement? same as SetTargetLocation?
     * @param unitId
     * @param longitude
     * @param latitude
     */
    public static native void SetMovementTarget( int unitId, float longitude, float latitude );
    /**
     * Set target location for given unit. If target id is also given, the id overrides the 
     * location.
     * TODO: different name in documentation SetTargetLocation(unitId, longitude, latitude) ?
     * TODO: also should be used for SetTargetId() (missing...)
     * @param _unitId
     * @param _targetUnitId
     * @param _longitude
     * @param _latitude
     */
    public static native void SetActionTarget( int _unitId, int _targetUnitId, float _longitude, float _latitude );
    /**
     * TODO: what's this?
     * @param _unitId
     * @param _targetUnitId
     */
    public static native void SetLandingTarget( int _unitId, int _targetUnitId );
    /**
     * Longitude of given unit, city, or event.
     * @param id
     * @return
     */
    public static native float GetLongitude( int id );
    /**
     * Latitude of given unit, city, or event.
     * @param id
     * @return
     */
    public static native float GetLatitude( int id );
    /**
     * Movement direction of given unit, in longitude and latitude parts. The vector has the 
     * length of the unit speed (see also SPEED_*).
     * @param unitId
     * @return
     */
    public static native float[] GetVelocity( int unitId );
    /**
     * Remaining range of unit. If unlimited, -1 is returned.
     * @param unitId
     * @return
     */
    public static native float GetRange( int unitId );
    /**
     * Amount of remaining units of given type that can be placed.
     * @param typeId
     * @return
     */
    public static native int GetRemainingUnits( int typeId );
    /**
     * True iff given location is valid for placement of given type. For fleets use 
  	 * getFleetMemberOffset to get offset from fleet center.
     * @param longitude
     * @param latitude
     * @param typeId
     * @return
     */
    public static native boolean IsValidPlacementLocation( float longitude, float latitude, int typeId );
    /**
     * Offset of ship number memberId from center of fleet, given fleet has memberCount 
     * ships
     * @param memberCount
     * @param memberId
     * @return
     */
    public static native float[] GetFleetMemberOffset( int memberCount, int memberId );
    /**
     * Tries to place a given structure to the given coordinates. Use IsValidStructureLocation 
     * to test if valid.
     * @param typeId
     * @param longitude
     * @param latitude
     */
    public static native void PlaceStructure( int typeId, float longitude, float latitude );
    /**
     * Tries to place a given amount of battlecruisers, carriers and subs into a fleet at the 
     * given location. Use IsValidFleedLocation to test.
     * @param longitude
     * @param latitude
     * @param typeShip1
     */
    public static native void PlaceFleet( float longitude, float latitude, int[] shipTypes);
    /**
     * Credits available for placement (if in variable unit mode).
     * @return
     */
    public static native int GetUnitCredits();
    /**
     * Value of given unit type (if in variable unit mode).
     * @param _typeId
     * @return
     */
    public static native int GetUnitValue( int _typeId );
    /**
     * TODO: documentation?
     * @param _voteId
     * @param _vote
     */
    public static native void SendVote( int _voteId, int _vote );
    /**
     * Sends a chat message.
     * TODO: documentation for second param?
     * @param chatMessage
     * @param receiverId
     */
    public static native void SendChatMessage( String chatMessage, int receiverId );
    /**
     * Distance in game between given coordinates.
     * @param longitude1
     * @param latitude1
     * @param longitude2
     * @param latitude2
     * @return
     */
    public static native float GetDistance( float longitude1, float latitude1, float longitude2, float latitude2 );
    /**
     * Distance in game between given coordinates on sea (performs pathfinding).
     * @param longitude1
     * @param latitude1
     * @param longitude2
     * @param latitude2
     * @return
     */
    public static native float GetSailDistance( float longitude1, float latitude1, float longitude2, float latitude2 );
    /**
     * Prints a line in the debug console in the specified color
     * @param entry
     */
    public static native void DebugLog(String entry);
    /**
     * Prints a line in the debug console in the specified color.
     * @param entry
     * @param objectId
     */
    public static native void DebugLog(String entry, int objectId);
    /**
     * Prints a line in the debug console.
     * TODO: param tags? objectId?
     * @param entry
     * @param objectId
     * @param tags
     */
    public static native void DebugLog(String entry, int objectId, String tags);
    /**
     * Prints a line in the debug console in the specified color.
     * @param entry
     * @param objectId
     * @param tags
     * @param colorR
     * @param colorG
     * @param colorB
     */
    public static native void DebugLog(String entry, int objectId, String tags, int colorR, int colorG, int colorB);
    /**
     * Prints a line in the debug console in the specified color.
     * @param entry
     * @param objectId
     * @param tags
     * @param colorR
     * @param colorG
     * @param colorB
     * @param colorAlpha
     */
    public static native void DebugLog(String entry, int objectId, String tags, int colorR, int colorG, int colorB, int colorAlpha);
    /**
     * True if the game is currently replayed (Timeline has been clicked).
     * TODO: e.g. game has been paused?
     * @return
     */
    public static native boolean DebugIsReplayingGame();
    /**
     * Draws a line on the whiteboard.
     * @param longitude1
     * @param latitude1
     * @param longitude2
     * @param latitude2
     */
    public static native void WhiteboardDraw(float longitude1, float latitude1, float longitude2, float latitude2);
    /**
     * Clears the whiteboard.
     */
    public static native void WhiteboardClear();
    /**
     * CommandIds of all commands that have been executed in previous cycle.
     * @return
     */
    public static native int[] GetSuccessfulCommands();
    
    ///**
    // * Moves the camera to a given location
    // */
    // TODO: MISSING!
    // public static native void DebugMoveCamera( float longitude, float latitude, float zoom );
    
    /**
     * Outputs a string to console
     * @param logLine
     */
    public static native void WriteToConsole(String logLine);
    
    private static String[][] commandLineOptions;

    public static class UnitData
    {
    	public int m_objectId;
    	public int m_type;
    	public int m_teamId;
    	public int m_currentState;
    	public boolean m_visible;
    	public float m_longitude;
    	public float m_latitude;

		@Override
		public String toString() {
			StringBuilder builder = new StringBuilder();

			builder.append("objectId: ");
			builder.append(m_objectId);
			builder.append("; type: ");
			builder.append(m_type);
			builder.append("; teamId: ");
			builder.append(m_teamId);
			builder.append("; currentState: ");
			builder.append(m_currentState);
			builder.append("; visible: ");
			builder.append(m_visible);
			builder.append("; longitude: ");
			builder.append(m_longitude);
			builder.append("; latitude: ");
			builder.append(m_latitude);

			return builder.toString();
		}
    }
    
	public static class FleetData {
		public int m_fleetId;
		public int m_teamId;
		public boolean m_visible;
		public float m_longitude;
		public float m_latitude;
		public int[] m_fleetMembers;

		@Override
		public String toString() {
			StringBuilder builder = new StringBuilder();

			builder.append("objectId: ");
			builder.append(m_fleetId);
			builder.append("; teamId: ");
			builder.append(m_teamId);
			builder.append("; visible: ");
			builder.append(m_visible);
			builder.append("; longitude: ");
			builder.append(m_longitude);
			builder.append("; latitude: ");
			builder.append(m_latitude);
			builder.append("; fleetMembers: ");
			builder.append(m_fleetMembers);

			return builder.toString();
		}

		public FleetData(int m_fleetId, int m_teamId, boolean m_visible,
				float m_longitude, float m_latitude, int[] m_fleetMembers) {
			this.m_fleetId = m_fleetId;
			this.m_teamId = m_teamId;
			this.m_visible = m_visible;
			this.m_longitude = m_longitude;
			this.m_latitude = m_latitude;
			this.m_fleetMembers = m_fleetMembers;
		}
	}

    public static class EventData
    {
    	public int m_eventType;
    	public int m_causeObjectId;
    	public int m_targetObjectId;
		public int m_unitType;
		public float m_longitude;
		public float m_latitude;

		@Override
		public String toString() {
			StringBuilder builder = new StringBuilder();

			builder.append("eventType: ");
			builder.append(m_eventType);
			builder.append("; causeObjectId: ");
			builder.append(m_causeObjectId);
			builder.append("; targetObjectId: ");
			builder.append(m_targetObjectId);
			builder.append("; unitType: ");
			builder.append(m_unitType);
			builder.append("; longitude: ");
			builder.append(m_longitude);
			builder.append("; latitude: ");
			builder.append(m_latitude);

			return builder.toString();
		}
    }
    
    public static final int STATE_AIRBASEFIGHTERLAUNCH = 0;
    public static final int STATE_AIRBASEBOMBERLAUNCH = 1;

    public static final int STATE_BATTLESHIPATTACK = 0;

    public static final int STATE_BOMBERATTACK = 0;    
    public static final int STATE_BOMBERNUKE = 1;      
    public static final int STATE_BOMBERINQUEUE = 2;   

    public static final int STATE_CARRIERFIGHTERLAUNCH = 0;
    public static final int STATE_CARRIERBOMBERLAUNCH = 1;
    public static final int STATE_CARRIERANTISUB = 2;  

    public static final int STATE_FIGHTERATTACK = 0;   
    public static final int STATE_FIGHTERINQUEUE = 2;    //2 instead of 1 makes handling easier when looking for STATE_s of items in queues

    public static final int STATE_NUKEONTARGET = 0;    
    public static final int STATE_NUKEDISARM = 1;      

    public static final int STATE_RADARACTIVE = 0;     

    public static final int STATE_SILONUKE = 0;        
    public static final int STATE_SILOAIRDEFENSE = 1;  

    public static final int STATE_SUBPASSIVESONAR = 0; 
    public static final int STATE_SUBACTIVESONAR = 1;  
    public static final int STATE_SUBNUKE = 2;         

    public static final int CHATCHANNEL_PUBLIC      = 100;
    public static final int CHATCHANNEL_ALLIANCE    = 101;
    public static final int CHATCHANNEL_SPECTATORS  = 102;

    // These were enums in C++, but to keep the JNI interface clean have been
    // converted to static ints here. Thus the somewhat inconsistent naming scheme 
    // to the above constants. Feel free to use enums instead, with an adequate
    // conversion of the return values from the API functions.    
    public static final int TerritoryNorthAmerica = 0;
    public static final int TerritorySouthAmerica = 1;
    public static final int TerritoryEurope = 2;
    public static final int TerritoryRussia = 3;
    public static final int TerritorySouthAsia = 4;
    public static final int TerritoryAfrica = 5;
    public static final int NumTerritories = 6;
    
    public static final int EventPingSub            = 0;
    public static final int EventPingCarrier        = 1;
    public static final int EventNukeLaunchSilo     = 2;
    public static final int EventNukeLaunchSub      = 3;
    public static final int EventHit                = 4;
    public static final int EventDestroyed          = 5;
    public static final int EventPingDetection      = 6;
    public static final int EventCeasedFire         = 7;
    public static final int EventUnceasedFire       = 8;
    public static final int EventSharedRadar        = 9;
    public static final int EventUnsharedRadar      = 10;
    public static final int EventNewVote            = 11;
    public static final int EventTeamVoted          = 12;
    public static final int EventTeamRetractedVote  = 13;
    public static final int EventVoteFinishedYes    = 14;
    public static final int EventVoteFinishedNo     = 15;
    
    public static final int TypeInvalid      = 0;
    public static final int TypeCity         = 1;
    public static final int TypeSilo         = 2;
    public static final int TypeRadarStation = 3;
    public static final int TypeNuke         = 4;
    public static final int TypeExplosion    = 5;
    public static final int TypeSub          = 6;
    public static final int TypeBattleShip   = 7;
    public static final int TypeAirBase      = 8;
    public static final int TypeFighter      = 9;
    public static final int TypeBomber       = 10;
    public static final int TypeCarrier      = 11;
    public static final int TypeTornado      = 12;
    public static final int TypeSaucer       = 13;
    public static final int TypeFleet        = 14;
    public static final int TypeGunshot      = 15;
    public static final int TypeQueueItem    = 16;
    public static final int NumObjectTypes   = 17;
    
    public static final int VoteTypeInvalid      = 0;
    public static final int VoteTypeJoinAlliance = 1;
    public static final int VoteTypeKickPlayer   = 2;
    public static final int VoteTypeLeaveAlliance = 3;
    
    public static final int VoteUnknown = 0;
    public static final int VoteYes     = 1;
    public static final int VoteNo      = 2;
    public static final int VoteAbstain = 3;
 
    /**
     * Data about all visible units, contained in the struct unitData (see enums). This 
     * function is for convenience only, as all data can be gathered through other functions, 
     * too.
     * <p><p>
     * Parse the unit data into a proper object array.
     * @return
     */
    public static List<UnitData> GetAllUnitData()
    {
    	float[] data = _GetAllUnitData();
    	LinkedList<UnitData> javaData = new LinkedList<UnitData>();
    	int counter = 0;
    	int limit = (int)Math.floor(data.length / 7);
    	for (int i = 0; i < limit; i++)
    	{
    		UnitData unit_data = new UnitData();
    		unit_data.m_objectId = (int) data[counter++];
    		unit_data.m_type = (int) data[counter++];
    		unit_data.m_teamId = (int) data[counter++];
    		unit_data.m_currentState = (int) data[counter++];
    		unit_data.m_visible = Math.round( data[counter++] ) != 0;
    		unit_data.m_longitude = data[counter++];
    		unit_data.m_latitude = data[counter++];
    		

    		javaData.add(unit_data);
		}


		return javaData;
	}

	/**
	 * Data about all visible fleets, contained in the struct fleetData (see
	 * enums). This function is for convenience only, as all data can be
	 * gathered through other functions, too.
	 * <p>
	 * <p>
	 * Parse the unit data into a proper object array.
	 * 
	 * @return
	 */
	public static List<FleetData> GetAllFleetData() {
    	
    	LinkedList<FleetData> javaData = new LinkedList<FleetData>();
    	
		for (int teamId : GetTeamIds()) {
			int[] fleets = GetFleets(teamId);
			
			for (int fleetId : fleets) {
				
				int[] fleetMembers = GetFleetMembers(fleetId);
				float[] fleetOffset = GetFleetMemberOffset(fleetMembers.length, 0);
				float fleetLongitude = GetLongitude(fleetMembers[0])
						- fleetOffset[0];
				float fleetLatitude = GetLatitude(fleetMembers[0])
						- fleetOffset[1];

				FleetData fleet_data = new FleetData(fleetId, teamId, true,
						fleetLongitude, fleetLatitude,
						fleetMembers);
				
				javaData.add(fleet_data);
			}
    	}
    	
		return javaData;
    }
    
    // update is called every tick of the game
    public static boolean update()
    {
    	try {
    		return PogamutJBotSupport.update();
    	} catch (Exception e) {
    		PogamutJBotSupport.logInitException(e);
    		return false;
    	}
    }
        
    // this is called when the dll is initialised, usually in the host/join dialog of the game
    public static boolean initialise(String[][] _commandLineOptions)
    {    
    	//saveCommandLineOptions(_commandLineOptions);
    	commandLineOptions = _commandLineOptions;
    	Boolean returnvalue = false;
    	try {
    		returnvalue = PogamutJBotSupport.initialise(commandLineOptions);
    	} catch (Exception e) {
    		PogamutJBotSupport.writeToConsole("FAIL!!!");
    		PogamutJBotSupport.writeToConsole(e.getMessage());
    		PogamutJBotSupport.logInitException(e);
    		return false;
    	}
    	WriteToConsole("Initialise finished");
    	return returnvalue;
    }
    /*
    private static void saveCommandLineOptions(String[][] commandLineOptions)
    {
    	JBot.commandLineOptions = new String[commandLineOptions.length][2];
    	for (int i = 0; i < commandLineOptions.length; ++i) {
    		String[] parameter = new String[2];
    		parameter[0] = commandLineOptions[i][0];
    		parameter[1] = commandLineOptions[i][1];
    		JBot.commandLineOptions[i] = parameter;
    	}
    }
    */
    // Whenever an event occurs, this function is called by the dll. This should be used to collect these events only,
    // with the main processing taking place in the update() function.
    public static void addEvent(int _eventType, int _causeObjectId, int _targetObjectId,
    		int _unitType, float _longitude, float _latitude)
    {
		try {
    		EventData data = new EventData();
    		data.m_eventType = _eventType;
    		data.m_causeObjectId = _causeObjectId;
    		data.m_targetObjectId = _targetObjectId;
			// unitType is screwed up
    		data.m_unitType = _unitType;
    		data.m_longitude = _longitude;
    		data.m_latitude = _latitude;

    		PogamutJBotSupport.addEvent(data);
    	} catch (Exception e) {
    		PogamutJBotSupport.logInitException(e);
    		return;
    	}
    }
    
}