package cz.cuni.amis.pogamut.multi.communication.worldview.impl;

import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;

import cz.cuni.amis.pogamut.base.communication.translator.event.IWorldChangeEvent;
import cz.cuni.amis.pogamut.base.component.bus.exception.ComponentNotRunningException;
import cz.cuni.amis.pogamut.base.component.controller.ComponentDependencies;
import cz.cuni.amis.pogamut.base.component.lifecyclebus.ILifecycleBus;
import cz.cuni.amis.pogamut.base.utils.logging.IAgentLogger;
import cz.cuni.amis.pogamut.multi.agent.ITeamedAgentId;
import cz.cuni.amis.pogamut.multi.communication.messages.SharedBatchBeginEvent;
import cz.cuni.amis.pogamut.multi.communication.messages.SharedBatchFinishedEvent;
import cz.cuni.amis.pogamut.multi.communication.translator.event.ICompositeWorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.multi.communication.translator.event.ILocalWorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.multi.communication.translator.event.ISharedWorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.multi.communication.translator.event.IStaticWorldObjectUpdatedEvent;
import cz.cuni.amis.pogamut.multi.communication.worldview.ISharedWorldView;
import cz.cuni.amis.pogamut.multi.utils.timekey.TimeKey;
import cz.cuni.amis.utils.exception.PogamutInterruptedException;

public abstract class BatchAwareLocalWorldView extends VisionLocalWorldView {

	public BatchAwareLocalWorldView(ComponentDependencies dependencies,
			ILifecycleBus bus, IAgentLogger logger,
			ISharedWorldView parentWorldView, ITeamedAgentId agentId) {
		super(dependencies, bus, logger, parentWorldView, agentId);
	}

	/**
	 * Queue of all incoming batches ready to be processed.
	 */
	private Queue<List<IWorldChangeEvent>> batches = new LinkedBlockingQueue<List<IWorldChangeEvent>>();
	
	/**
	 * The current(incomplete) batch of events waiting to be processed.
	 */
	private List<IWorldChangeEvent> currentBatch = new LinkedList<IWorldChangeEvent>();
	
	protected abstract boolean isBatchBeginEvent( IWorldChangeEvent event );
	
	protected abstract boolean isBatchEndEvent( IWorldChangeEvent event );
	
	/**
	 * This means, that lock was requested by a thread and it is waiting for finishing the batch.
	 */
	private boolean lockRequested = false;
	
	private boolean lockFinished = false;
	
	
	/**
	 * This means, that we are waiting for shared worldView to process all events from current batch (between current begin event and current end event)
	 */
	private boolean waitingForSharedBatch = false; 
	
	/**
	 * The lock for a time is set when the sharedWorldView has been sent a BatchBeginMessage, but it has not yet confirmed that
	 * all events for the specified time have been processed.
	 */
	private Set<Long> sharedWVLocks = Collections.synchronizedSet( new HashSet<Long>(4));
	
	private boolean timeKeyIncreased = false;
	
	private Object objectMutex = new Object();	
	private CountDownLatch latch = new CountDownLatch(1);
	
	/**
	 * These keys are currently locked (shadowCopies are held)
	 */
	private List<Long> lockedTimes = new LinkedList<Long>();
	
	/**
	 * Notifies sharedWorldView that a beginEvent has been recieved with with the specified time and the sharedWorldView should notify
	 * this worldView back, when all events for the time have been processed.
	 * @param time
	 */
	protected synchronized void notifySharedBegin( long time )
	{
		log.finer("Notifying sharedWorldView with SharedBegin event of time : " + time);
		sharedWorldView.notify( new SharedBatchBeginEvent(time, this.agentId) );
	}
	
	/**
	 * This method is called when the SharedBatchFinishedEvent is recieved from the sharedWorldView, notifying us that
	 * all sharedEvents for the specified time have been processed and it is safe to run logic on the time.
	 * @param time
	 */
	protected synchronized void sharedBatchFinished( long time )
	{
		synchronized( lockedTimes )
		{
			log.finer("SharedBatchFinishedEvent recieved from the SharedWorldView for time " + time );
			if ( !lockFinished )
			{
				
				setCurrentTime( TimeKey.get(time) );
				timeKeyIncreased = true;
				List<Long> newLocks = new LinkedList<Long>();
				for ( Long t : lockedTimes )
				{
					if ( t < time)
					{
						unlockTime(t);
					}
					else
					{
						newLocks.add(t);
					}
				}
				lockedTimes = newLocks;
				latch.countDown();
				
			}		
		}
	}
	
	public boolean isLocked()
	{
		return lockFinished;
	}
	
	/**
	 * Must be called before starting logic.
	 */
	public void lock()
	{
		log.fine("Locking BatchAwareLocalWorldView");
		
		synchronized ( objectMutex )
		{
			if (!isRunning()) throw new ComponentNotRunningException("Can't lock() world view is not running!", log, this);
			if ( !lockFinished  )
			{
				lockRequested = true;
			}
			else
			{
				return;
			}			
		}
		
		try {
			latch.await();
		} catch (InterruptedException e) {
			throw new PogamutInterruptedException("Interrupted while waiting to acquire lock()!", e, this);
		}
		
		lockFinished = true;
		
		log.fine("BatchAwareLocalWorldView locked.");
	}

	/**
	 * Called after the logic has finished.
	 */
	public void unlock()
	{
		synchronized( objectMutex )
		{
			if (!isRunning()) throw new ComponentNotRunningException("Can't unlock() world view is not running!", log, this);
			LinkedList<Long> newLocks = new LinkedList<Long>();
			for ( Long t : lockedTimes )
			{
				if ( t < currentTimeKey.getTime() )
				{
					this.unlockTime( t );
				}
				else
				{
					newLocks.add(t);
				}
			}
			lockedTimes = newLocks;
			lockFinished = false;
			lockRequested = false;
			latch = new CountDownLatch(1);
			log.fine("BatchAwareLocalWorldView unlocked");
		}
	}
	
	//TODO fix behavior when we are recieving events for different times together (filter and process lower time first)
	
	private boolean timeKeySet = false;
	
	@Override
	public synchronized void notify(IWorldChangeEvent event)
	{
		log.finest( "BatchAwareLocalWorldView notify : " + event);
		
		if (!timeKeySet)
		{
			this.currentTimeKey = TimeKey.get( event.getSimTime() );
			timeKeySet = true;
		}
		
		//if the event updates a shared part of a WorldObject, notify sharedWorldView
    	if (!( event instanceof ILocalWorldObjectUpdatedEvent))
        {
    		if ( event instanceof ICompositeWorldObjectUpdatedEvent)
    		{
    			IWorldChangeEvent partEvent = ((ICompositeWorldObjectUpdatedEvent)event).getSharedEvent();
    			if (partEvent != null) //shared part
    			{
    				log.finest("Notyfying sharedWV " + partEvent.toString() + ")");
    				sharedWorldView.notify(partEvent);	
    			}
    			partEvent = ((ICompositeWorldObjectUpdatedEvent)event).getStaticEvent();
				if ( partEvent != null) //static part
				{
					log.finest("Notyfying sharedWV " + partEvent.toString() + ")");
					sharedWorldView.notify(partEvent);
				}
    		}
        	//shared or static event will not modify LocalObjects, no need to process it beyond notifying sharedWorldView
    		else if ( (event instanceof ISharedWorldObjectUpdatedEvent) || (event instanceof IStaticWorldObjectUpdatedEvent) )
        	{
    			log.finest("Notyfying sharedWV " + event.toString() + ")");
    			sharedWorldView.notify(event);
        		return;
        	}
        }
    	
    	synchronized(objectMutex)
    	{
	    	if ( isBatchBeginEvent(event) )
	    	{
	    		if ( currentTimeKey == null )
	    		{
	    			currentTimeKey = TimeKey.get(event.getSimTime());
	    		}
		    	lockTime( event.getSimTime());
		    	this.lockedTimes.add( event.getSimTime() );
		    	notifySharedBegin( event.getSimTime() );
		    	
	    	}
	    	else if ( isBatchEndEvent(event) )
	    	{
	    		log.finer("Notifying sharedWorldView with EndEvent of time " + event.getSimTime() + " : " + event);
	    		sharedWorldView.notify(event);
	    		super.notify(event);
	    	}    	
	    	else if ( event instanceof SharedBatchFinishedEvent )
	    	{
	    		sharedBatchFinished(event.getSimTime());
	    	}
	    	else
	    	{
	    		super.notify( event );
	    	}
    	}
	}
	
	@Override
	protected void stop() {
		super.stop();
		synchronized(objectMutex) {
			while (latch != null && latch.getCount() > 0) latch.countDown();
			while (lockedTimes != null && lockedTimes.size() > 0) {
				long time = lockedTimes.get(0);
				unlockTime(lockedTimes.get(0));
				if (lockedTimes.get(0) == time) lockedTimes.remove(0);
			}
		}
	}
	
	@Override
	protected void kill() {
		super.kill();
		synchronized(objectMutex) {
			while (latch != null && latch.getCount() > 0) latch.countDown();
			while (lockedTimes != null && lockedTimes.size() > 0) {
				try {
					long time = lockedTimes.get(0);
					unlockTime(lockedTimes.get(0));
					if (lockedTimes.get(0) == time) lockedTimes.remove(0);
				} catch (Exception e) {					
				}
			}
		}
	}
	
}
