package cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.logging.Logger;

import cz.cuni.amis.pogamut.base.communication.worldview.IWorldView;
import cz.cuni.amis.pogamut.base.communication.worldview.event.IWorldEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.IListenerRegistrator;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.ListenerLevel;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.exception.ListenerMethodParametersException;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.exception.MissingConstructorException;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.annotation.exception.MoreThanOneListenerLevelAnnotationException;
import cz.cuni.amis.pogamut.base.communication.worldview.listener.exception.ListenersAlreadyRegisteredException;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEvent;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectEventListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.IWorldObjectListener;
import cz.cuni.amis.pogamut.base.communication.worldview.object.WorldObjectId;
import cz.cuni.amis.utils.ClassUtils;
import cz.cuni.amis.utils.Lazy;
import cz.cuni.amis.utils.NullCheck;
import cz.cuni.amis.utils.exception.PogamutException;
import cz.cuni.amis.utils.maps.LazyMap;
import java.util.logging.Level;

/**
 * The registrator that is driven by annotations on the class it introspects.
 * <p><p>
 * WARNING: the inheritance does not work here! Only the top class object is introspected.
 * 
 * 
 * @author Jimmy
 */
public class AnnotationListenerRegistrator implements IListenerRegistrator{

	/**
	 * Returns a new {@link WorldObjectId} for the given 'annotation'.
	 * @param method
	 * @param annotation
	 * @return id instance
	 */
	public static WorldObjectId getId(Method method, ObjectListener annotation) {
		try {
			return annotation.idClass().getConstructor(String.class).newInstance(annotation.objectId());
		} catch (Exception e) {
			throw new MissingConstructorException(method, annotation.idClass(), AnnotationListenerRegistrator.class);
		}
	}
	
	/**
	 * Returns a new {@link WorldObjectId} for the given 'annotation'.
	 * @param method
	 * @param annotation
	 * @return id instance
	 */
	public static WorldObjectId getId(Method method, ObjectEventListener annotation) {
		try {
			return annotation.idClass().getConstructor(String.class).newInstance(annotation.objectId());
		} catch (Exception e) {
			throw new MissingConstructorException(method, annotation.idClass(), AnnotationListenerRegistrator.class);
		}
	}
	
	/**
	 * Returns listener level that is gained from the method's annotation.
	 * @param method
	 * @return listener level of the method
	 */
	public static ListenerLevel getListenerLevel(Method method) {
		ListenerLevel level = null;
		if (method.isAnnotationPresent(EventListener.class)) {
			level = ListenerLevel.A;
		}
		if (method.isAnnotationPresent(ObjectClassListener.class)) {
			if (level != null) throw new MoreThanOneListenerLevelAnnotationException(method, AnnotationListenerRegistrator.class);
			level = ListenerLevel.B;
		}
		if (method.isAnnotationPresent(ObjectClassEventListener.class)) {
			if (level != null) throw new MoreThanOneListenerLevelAnnotationException(method, AnnotationListenerRegistrator.class);
			level = ListenerLevel.C;
		}
		if (method.isAnnotationPresent(ObjectListener.class)) {
			if (level != null) throw new MoreThanOneListenerLevelAnnotationException(method, AnnotationListenerRegistrator.class);
			level = ListenerLevel.D;
		}
		if (method.isAnnotationPresent(ObjectEventListener.class)) {
			if (level != null) throw new MoreThanOneListenerLevelAnnotationException(method, AnnotationListenerRegistrator.class);
			level = ListenerLevel.E;
		}
		return level;
	}

	/**
	 * Level A listener that can be hooked to the world view (for more info 
	 * about listeners see {@link IWorldView}).
	 * 
	 * @author Jimmy
	 */
	private class LevelAListener implements IWorldEventListener {

		Method method;
		
		public LevelAListener(Method method) {
			// check the method signature
			if (method.getParameterTypes().length != 1 ||
			    !method.getParameterTypes()[0].isAssignableFrom(method.getAnnotation(EventListener.class).eventClass())) { 
				throw new ListenerMethodParametersException(method, method.getAnnotation(EventListener.class), AnnotationListenerRegistrator.this);
			}
			this.method = method; 
		}
		
		public EventListener getAnnotation() {
			return method.getAnnotation(EventListener.class);
		}
		
		public Class getEventClass() {
			return getAnnotation().eventClass();
		}
		
		@Override
		public void notify(Object event) {
			try {
				method.setAccessible(true);
				method.invoke(obj, event);
				method.setAccessible(false);
			} catch (Exception e) {
				throw new PogamutException("Could not invoke LevelA listener " + ClassUtils.getMethodSignature(method) + " with parameter of class " + event.getClass() + ".", e, log, this);
			}
		}
		
	}
	
	private class LevelBListener implements IWorldObjectListener {

		Method method;
		
		public LevelBListener(Method method) {
			// check the method signature
			if (method.getParameterTypes().length != 1 ||
			    !method.getParameterTypes()[0].isAssignableFrom(IWorldObjectEvent.class)) { 
				throw new ListenerMethodParametersException(method, method.getAnnotation(ObjectClassListener.class), AnnotationListenerRegistrator.this);
			}
			this.method = method; 
		}
		
		public ObjectClassListener getAnnotation() {
			return method.getAnnotation(ObjectClassListener.class);
		}
		
		public Class getObjectClass() {
			return getAnnotation().objectClass();
		}
		
		@Override
		public void notify(Object event) {
			try {
				method.setAccessible(true);
				method.invoke(obj, event);
				method.setAccessible(false);
			} catch (Exception e) {
				throw new PogamutException("Could not invoke LevelB listener " + ClassUtils.getMethodSignature(method) + " with parameter of class " + event.getClass() + ".", e, log, this);
			}
		}
		
	}
	
	private class LevelCListener implements IWorldObjectEventListener {

		Method method;
		
		public LevelCListener(Method method) {
			// check the method signature
			if (method.getParameterTypes().length != 1 ||
			    !method.getParameterTypes()[0].isAssignableFrom(method.getAnnotation(ObjectClassEventListener.class).eventClass())) { 
				throw new ListenerMethodParametersException(method, method.getAnnotation(ObjectClassEventListener.class), AnnotationListenerRegistrator.this);
			}
			this.method = method; 
		}
		
		public ObjectClassEventListener getAnnotation() {
			return method.getAnnotation(ObjectClassEventListener.class);
		}
		
		public Class getEventClass() {
			return getAnnotation().eventClass();
		}
		
		public Class getObjectClass() {
			return getAnnotation().objectClass();
		}
		
		@Override
		public void notify(Object event) {
			try {
				method.setAccessible(true);
				method.invoke(obj, event);
				method.setAccessible(false);
			} catch (Exception e) {
				throw new PogamutException("Could not invoke LevelC listener " + ClassUtils.getMethodSignature(method) + " with parameter of class " + event.getClass() + ".", e, log, this);
			}
		}
		
	}
	
	private class LevelDListener implements IWorldObjectListener {

		Method method;
		
		WorldObjectId objectId;
		
		public LevelDListener(Method method) {
			// check the method signature
			if (method.getParameterTypes().length != 1 ||
			    !method.getParameterTypes()[0].isAssignableFrom(IWorldObjectEvent.class)) { 
				throw new ListenerMethodParametersException(method, method.getAnnotation(ObjectListener.class), AnnotationListenerRegistrator.this);
			}
			this.method = method;
			objectId = getId(method, getAnnotation());
		}
		
		public ObjectListener getAnnotation() {
			return method.getAnnotation(ObjectListener.class);
		}
		
		public WorldObjectId getObjectId() {
			return objectId;
		}
		
		@Override
		public void notify(Object event) {
			try {
				method.setAccessible(true);
				method.invoke(obj, event);
				method.setAccessible(false);
			} catch (Exception e) {
				throw new PogamutException("Could not invoke LevelD listener " + ClassUtils.getMethodSignature(method) + " with parameter of class " + event.getClass() + ".", e, log, this);
			}
		}
		
	}
	
	private class LevelEListener implements IWorldObjectEventListener {

		Method method;
		
		WorldObjectId objectId;
		
		public LevelEListener(Method method) {
			// check the method signature
			if (method.getParameterTypes().length != 1 ||
			    !method.getParameterTypes()[0].isAssignableFrom(method.getAnnotation(ObjectEventListener.class).eventClass())) { 
				throw new ListenerMethodParametersException(method, method.getAnnotation(ObjectEventListener.class), AnnotationListenerRegistrator.this);
			}
			this.method = method; 
			objectId = getId(method, getAnnotation());
		}
		
		public ObjectEventListener getAnnotation() {
			return method.getAnnotation(ObjectEventListener.class);
		}
		
		public WorldObjectId getObjectId() {
			return objectId;
		}
		
		public Class getEventClass() {
			return getAnnotation().eventClass();
		}
		
		@Override
		public void notify(Object event) {
			try {
				method.setAccessible(true);
				method.invoke(obj, event);
				method.setAccessible(false);
			} catch (Exception e) {
				throw new PogamutException("Could not invoke LevelE listener " + ClassUtils.getMethodSignature(method) + " with parameter of class " + event.getClass() + ".", e, log, this);
			}
		}
		
	}
	
	private IWorldView worldView;
	private Object obj;
	private boolean listenersRegistered = false;
	
	private Lazy<List<Method>> methods = new Lazy<List<Method>>() {

		@Override
		protected List<Method> create() {
			return probeMethods();
		}
		
	};
	
	private Map<ListenerLevel, List<IWorldEventListener>> listeners = new LazyMap<ListenerLevel, List<IWorldEventListener>>() {

		@Override
		protected List<IWorldEventListener> create(ListenerLevel key) {
			return new ArrayList<IWorldEventListener>();
		}
		
	};
	
	private Logger log;
	
	public AnnotationListenerRegistrator(Object obj, IWorldView worldView, Logger log) {
		this.worldView = worldView;
		NullCheck.check(this.worldView, "worldView");
		this.obj = obj;
		NullCheck.check(this.obj, "obj");
		this.log = log;
		NullCheck.check(this.log, "log");
	}
	
	/**
	 * Check the {@link AnnotationListenerRegistrator#obj} methods for the listener annotations. 
	 * @return
	 */
	private List<Method> probeMethods() {
		List<Method> methods = new ArrayList<Method>();
		for (Method method : obj.getClass().getDeclaredMethods()) {
			if (getListenerLevel(method) != null) methods.add(method);
		}
		return methods;
	}
	
	/**
	 * Introspect all object's methods and register various listeners based on
	 * {@link EventListener}, etc... annotations.
	 * 
	 * @param obj
	 */
	@Override
	public synchronized void addListeners() throws ListenersAlreadyRegisteredException {
		if (listenersRegistered) throw new ListenersAlreadyRegisteredException(this);
		if (log.isLoggable(Level.FINER)) log.finer(obj + " -> " + worldView + ": Registering listeners.");
		for (Method method : this.methods.getVal()) {
			switch(getListenerLevel(method)){
			case A:
				if (log.isLoggable(Level.FINE)) log.fine(obj + " -> " + worldView + ": Registering level A listener for " + ClassUtils.getMethodSignature(method));
				LevelAListener listenerA = new LevelAListener(method);
				worldView.addEventListener(listenerA.getEventClass(), listenerA);
				listeners.get(ListenerLevel.A).add(listenerA);
				break;
			case B:
				if (log.isLoggable(Level.FINE)) log.fine(obj + " -> " + worldView + ": Registering level B listener for " + ClassUtils.getMethodSignature(method));
				LevelBListener listenerB = new LevelBListener(method);
				worldView.addObjectListener(listenerB.getObjectClass(), listenerB);
				listeners.get(ListenerLevel.B).add(listenerB);
				break;
			case C:
				if (log.isLoggable(Level.FINE)) log.fine(obj + " -> " + worldView + ": Registering level C listener for " + ClassUtils.getMethodSignature(method));
				LevelCListener listenerC = new LevelCListener(method);
				worldView.addObjectListener(listenerC.getObjectClass(), listenerC.getEventClass(), listenerC);
				listeners.get(ListenerLevel.C).add(listenerC);
				break;
			case D:
				if (log.isLoggable(Level.FINE)) log.fine(obj + " -> " + worldView + ": Registering level D listener for " + ClassUtils.getMethodSignature(method));
				LevelDListener listenerD = new LevelDListener(method);
				worldView.addObjectListener(listenerD.getObjectId(), listenerD);
				listeners.get(ListenerLevel.D).add(listenerD);
				break;
			case E:
				if (log.isLoggable(Level.FINE)) log.fine(obj + " -> " + worldView + ": Registering level E listener for " + ClassUtils.getMethodSignature(method));
				LevelEListener listenerE = new LevelEListener(method);
				worldView.addObjectListener(listenerE.getObjectId(), listenerE.getEventClass(), listenerE);
				listeners.get(ListenerLevel.E).add(listenerE);
				break;
			}	
		}
		if (log.isLoggable(Level.INFO)) log.info(obj + " -> " + worldView + ": Registered " + listeners.size() + " listeners.");
	}
	
	public int getListenersCount() {
		return listeners.get(ListenerLevel.A).size() + listeners.get(ListenerLevel.B).size() + listeners.get(ListenerLevel.C).size() + listeners.get(ListenerLevel.D).size() + listeners.get(ListenerLevel.E).size();
	}

	@Override
	public synchronized void removeListeners() {
		if (!listenersRegistered) return;
		if (log.isLoggable(Level.FINER)) log.finer(obj + " -> " + worldView + ": Removing " + getListenersCount()  + " listeners.");
		for (IWorldEventListener l : listeners.get(ListenerLevel.A)) {
			LevelAListener listenerA = (LevelAListener) l;
			worldView.removeEventListener(listenerA.getEventClass(), listenerA);
		}
		for (IWorldEventListener l : listeners.get(ListenerLevel.B)) {
			LevelBListener listenerB = (LevelBListener) l;
			worldView.removeObjectListener(listenerB.getObjectClass(), listenerB);
		}
		for (IWorldEventListener l : listeners.get(ListenerLevel.C)) {
			LevelCListener listenerC = (LevelCListener) l;
			worldView.removeObjectListener(listenerC.getObjectClass(), listenerC.getEventClass(), listenerC);
		}
		for (IWorldEventListener l : listeners.get(ListenerLevel.D)) {
			LevelDListener listenerD = (LevelDListener) l;
			worldView.removeObjectListener(listenerD.getObjectId(), listenerD);
		}
		for (IWorldEventListener l : listeners.get(ListenerLevel.E)) {
			LevelEListener listenerE = (LevelEListener) l;
			worldView.removeObjectListener(listenerE.getObjectId(), listenerE.getEventClass(), listenerE);
		}
		if (log.isLoggable(Level.INFO)) log.info(obj + " -> " + worldView + ": Listeners removed.");
	}
	
}
