package scheduler;

import java.io.File;
import java.io.FileInputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Unmarshaller;
import utils.TimeUtils;

/**
 *
 * @author kadlr3am
 */
public class DefaultProbability implements ProbabilityDistribution {

    Random random = new Random();
    
    Map<IState, List<SInterval>> stateIntervalTable = null;
    
    public Map<IState, Double> getStateTransitionsProbabilities(double time, int day, IState previousState) {

        Map<IState, Double> resultMap = new HashMap<IState, Double>();

        if (stateIntervalTable.get(previousState) != null) {
            for(SInterval interval : stateIntervalTable.get(previousState)) {
                if(interval.isApplicable(time, day)) {
                    for (StateTransition sTran : interval.getTransitions()){
                        resultMap.put(sTran.getTargetState(), sTran.getProbability());
                    }
                }
            }
        }

        return resultMap;
    }

    public IState computeNextState(double time, int day, IState previousState) {
        Map<IState, Double> transitionProbabilities = getStateTransitionsProbabilities(time, day, previousState);

              
       /* System.out.print("\n Time:" + time + " Day:" + day);
        for (IState state : transitionProbabilities.keySet()) {
            System.out.print("\n Set:" + state.getItention() + transitionProbabilities.get(state));
        }*/
        
        double randomValue = random.nextDouble();
        
        double currentValue = 0;

        for (IState state : transitionProbabilities.keySet()) {
            currentValue += transitionProbabilities.get(state);
            if (randomValue <= currentValue) {
                return state;
            }
        }

        //if there was new state generated based on this weekday transitions, universal transitions are tried.
        transitionProbabilities = getStateTransitionsProbabilities(time, 0, previousState);
        for (IState state : transitionProbabilities.keySet()) {
            currentValue += transitionProbabilities.get(state);
            if (randomValue <= currentValue) {
                return state;
            }
        }

        //We have not returned anything yet - either transition not defined or probability
        //values have been wrongly assigned. We will return state according to default transition.
        if (!previousState.getItention().equals("Default")) //infinite recursion testing
            return computeNextState(time, day, new State("Default"));
        else
            throw new RuntimeException("Transition not defined!");
    }

    public void addStateTransition(int from, int to, int day, IState fromState, IState toState, double probability){

        List<SInterval> stateIntervals = stateIntervalTable.get(fromState);
        if (stateIntervals == null) {
            stateIntervals = new ArrayList<SInterval>();
            stateIntervalTable.put(fromState, stateIntervals);
        }

        SInterval intervalToAddTo = null;
        for (SInterval interv : stateIntervals) {
            if (interv.getFrom() == from && interv.getTo() == to && interv.getDay() == day) {
                intervalToAddTo = interv;
                break;
            }
        }
        if (intervalToAddTo != null) {
            for (StateTransition sTran : intervalToAddTo.getTransitions()){
                if (sTran.getTargetState().getItention().equals(toState.getItention())){
                    throw new RuntimeException("State transition already there!");                    
                }
            }
            StateTransition stateTrans = new StateTransition(new State(toState.getItention()), probability);
            intervalToAddTo.getTransitions().add(stateTrans);                        
        } else {
            StateTransition stateTrans = new StateTransition(new State(toState.getItention()), probability);
            ArrayList<StateTransition> newStateTransitionList = new ArrayList<StateTransition>();
            newStateTransitionList.add(stateTrans);
            intervalToAddTo = new SInterval(newStateTransitionList,from,to,day);
            stateIntervals.add(intervalToAddTo);
        }
    }

    public void setExampleTransitions() {
        stateIntervalTable = new HashMap<IState, List<SInterval>>();
        random.setSeed(TimeUtils.getTimeInMilis());

        //test inicialization
        addStateTransition(7, 20, 0, new State("ISleep"), new State("IWork"), 1);
        addStateTransition(7, 20, 0, new State("IWork"), new State("IWork"), 0.5);
        addStateTransition(7, 20, 0, new State("IWork"), new State("IEntertainment"), 0.5);
        addStateTransition(7, 20, 0, new State("IEntertainment"), new State("IEntertainment"), 0.5);
        addStateTransition(7, 20, 0, new State("Default"), new State("IWork"), 1);

        addStateTransition(20, 22, 0, new State("Default"), new State("IEntertainment"), 1);
        addStateTransition(20, 22, 0, new State("IWork"), new State("IStudy"), 0.8);
        addStateTransition(20, 22, 0, new State("IWork"), new State("IEntertainment"), 0.2);
        addStateTransition(20, 22, 0, new State("IStudy"), new State("IStudy"), 0.8);
        addStateTransition(20, 22, 0, new State("IStudy"), new State("IEntertainment"), 0.2);

        addStateTransition(22, 24, 0, new State("Default"), new State("IGoToPub"), 1);

        addStateTransition(0, 7, 0, new State("ISleep"), new State("ISleep"), 1);
        addStateTransition(0, 7, 0, new State("Default"), new State("ISleep"), 1);

    }

    public DefaultProbability(Map<IState, List<SInterval>> stateIntervalTable) {
        
        this.stateIntervalTable = stateIntervalTable;
        random.setSeed(TimeUtils.getTimeInMilis());
    }

    public DefaultProbability(String directory, String planningFilename) {
                
        this.stateIntervalTable = new HashMap<IState, List<SInterval>>();
        random.setSeed(TimeUtils.getTimeInMilis());
        
        ArrayList<ProbabilisticSchedulerEntry> schedulerEntries = loadTransitionEntries(directory, planningFilename);

        for (ProbabilisticSchedulerEntry entry : schedulerEntries) {
            for (ProbabilisticSchedulerTransitionEntry transitionEntry : entry.transitions){
                addStateTransition(entry.from, entry.to, entry.day, new State(transitionEntry.fromIntention),
                        new State(transitionEntry.toIntention), transitionEntry.probability);
            }
        }
    }

    /**
     * loads exml file with ScheduleEntries
     * @param directory - location of the file
     * @param filename - file name
     * @return - list of schedule entries
     */
    private ArrayList<ProbabilisticSchedulerEntry> loadTransitionEntries(String directory, String filename) {

        File file = new File(directory + filename);
        ProbabilisticSchedulerEntryListWrapper wrapper = null;
        try {
            FileInputStream reader = new FileInputStream(file);
            JAXBContext context = JAXBContext.newInstance(ProbabilisticSchedulerEntryListWrapper.class);
            Unmarshaller u = context.createUnmarshaller();
            wrapper = (ProbabilisticSchedulerEntryListWrapper) u.unmarshal(reader);
        } catch (Exception e) {
            System.err.println("Error in loading " + filename + "\n" + e);
        }

        return wrapper.list;
    }

    public IState computeNextState(double counter, int day, String name) {
        return this.computeNextState(counter, day, new State(name));
    }
}
