package decisionMakingSystem;

import cz.cuni.amis.utils.exception.PogamutException;
import bot.Bot;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.Serializable;
import utils.Interval;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Random;
import java.util.logging.Logger;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import utils.ScheduleEntryListWrapper;
import utils.TimeUtils;

// those enum has to be changed and used:)
enum BasicNeeds {
  SLEEP, EAT, URINATE
}

enum PlannedActivities {
  SWIM, FRISBEE, SCHOOL, STUDY, WORK, SEE_A_MOVIE, SEE_A_PLAY
}

enum FreeTimeActivities {
  IEntertainment, IStareAtTV
} // ICleanUp, IStudy
enum UnexpectedTricks {
  PARTY, SKI
}

/**
 * Scheduler creates everyday plans according to the information given in the planning.xml.
 * 
 * It first schedules thinks like satisfying basic biological needs - eating, hygiene, etc.
 * then some scheduled actions - like go to school on monday, thursday, friday,
 * go to a training of frisbee three evenings etc.
 * And after all it ends up with using the rest of the place for any planable action
 * 
 * @author Ondrej
 */
public class Scheduler implements Serializable {

  /** all available intentions - loaded at the beginning */
  protected ArrayList<Intention> allIntentions = null;
  /** just a mapping between the ArrayList and intentions, so we can access it a bit faster;) */
  protected HashMap<String, Intention> nameToIntention = null;
  /** log */
  protected transient Logger log = null;
  /** current day from the start of the simulation, which is by default 1.9. 200X */
  private int day = 0;
  /** current week from the start of the simulation */
  private int week = 0;
  /** day of the week - 1:Monday, 7:Sunday */
  private int dayOfWeek = 0;
  /** list of intervals, when a bot has some time during the day */
  protected ArrayList<Interval> freeTime = null;
  /** free part of the day - 0 morning, 1 afternoon, 2 evening */
  protected ArrayList<Boolean> freePartsOfTheDay = null;
  /** a day schedule, which is than returned */
  protected ArrayList<Intention> daySchedule = null;
  /** entries from the planning.xml */
  protected ArrayList<ScheduleEntry> activityScheduling = null;
  /**
   * entries from the extraPlanning.xml
   * which contains planning of less usual things like a weekend in the
   */
  protected ArrayList<ScheduleEntry> extraScheduling;
  /** */
  protected Random rnd = null;
  /** time (in minutes) when agent wakes up */
  protected int agentsAlarm = 400;

  /**
   *
   * @param intentions
   * @param log
   * @param directory
   */
  public Scheduler(ArrayList<Intention> intentions, Logger log, String directory, String planningFilename) {
    this.allIntentions = intentions;
    // initialize name to intention
    this.nameToIntention = new HashMap<String, Intention>();
    for (Intention intention : intentions) {
      nameToIntention.put(intention.getName(), intention);
    }
    this.log = log;
    freePartsOfTheDay = new ArrayList<Boolean>();
    freeTime = new ArrayList<Interval>();
    daySchedule = new ArrayList<Intention>();
    // load XML file planning.xml
    activityScheduling = loadPlanning(directory, planningFilename);
    // extraScheduling = loadPlanning(directory, "ExtraActivities.xml");
    rnd = new Random(System.currentTimeMillis());
  }

  public Scheduler(){};

  /**
   * modifies the activation intervals of intentions stored in intentions
   * => new day timetable
   */
  public ArrayList<Intention> scheduleNewDay(int counter) {
    for (Intention intention : nameToIntention.values()) // reinitialize the activation intervals
    {
      intention.setActivationIntervals(null);
    }
    daySchedule.clear();
    day = counter / GlobalParameters.LENGHT_OF_A_DAY;
//    week = day / 7;
    week = 0;
    dayOfWeek = day % 7 + 1; // 1 - monday, ... 7 - sunday
    freeTime.clear();
    freePartsOfTheDay.clear();
    freePartsOfTheDay.add(true); // morning
    freePartsOfTheDay.add(true); // afternoon
    freePartsOfTheDay.add(true); // evening
    // basic needs -> food, sleep - based on biorhytms?
    scheduleBasicActions(counter);
    // scheduled actions
    scheduleNormalActions();
    // surprising actions
    // go skiing, or I dont know - based on the month:)
    scheduleSpecialActions();
    // to fill the rest of the time
    scheduleOtherActions();
    return daySchedule;
  }

  protected void setLog(Logger log) {
    this.log = log;
  }

  private String printFreeTime() {
    String result = "";
    for (Interval interval : freeTime) {
      result += interval.toString() + " \n";
    }
    return result;
  }

  /**
   * schedules basic actions like sleeping, eating, brushing teeth
   */
  private void scheduleBasicActions(int start) {
    rnd.setSeed(System.currentTimeMillis());
    // ask biorythms for the time of the waking up => should reflect, when he went to bed, if it is saturday, etc.
    int dayBegins = start + TimeUtils.minutesToTicksOfLogic(agentsAlarm + rnd.nextInt(40)); // 7 a.m. it will be soon given by biorythms
    int dayEnds = start + TimeUtils.minutesToTicksOfLogic(1400 + rnd.nextInt(40)); // 11 p.m. should be determined by biorythms as well

    Intention sleep, eat, hygiene;
    sleep = nameToIntention.get("ISleep");
    eat = nameToIntention.get("IEat");
    hygiene = nameToIntention.get("IHygiene");
    ArrayList<Interval> activationIntervals = new ArrayList<Interval>();

    // first sleeping
    activationIntervals.add(new Interval(start, dayBegins, 30));
//    activationIntervals.add(new Interval(dayEnds, start + GlobalParameters.LENGHT_OF_A_DAY, 30));
    sleep.setActivationIntervals(activationIntervals);
    // hygiene - dayBegins + 15 minutes, dayEnd - 35 minutes
    activationIntervals = new ArrayList<Interval>();
    activationIntervals.add(new Interval(dayBegins + TimeUtils.minutesToTicksOfLogic(5), dayBegins + TimeUtils.minutesToTicksOfLogic(15), 55));
    activationIntervals.add(new Interval(dayEnds - TimeUtils.minutesToTicksOfLogic(45), dayEnds - TimeUtils.minutesToTicksOfLogic(35), 55));
    hygiene.setActivationIntervals(activationIntervals);
    // eating - about 20 minutes breakfast, 40 minutes lunch, dinner
    activationIntervals = new ArrayList<Interval>();
    activationIntervals.add(new Interval(dayBegins + TimeUtils.minutesToTicksOfLogic(20), dayBegins + TimeUtils.minutesToTicksOfLogic(40), 60)); // breakfast

    int timeOfLunch = start + GlobalParameters.LENGHT_OF_A_DAY / 2 + TimeUtils.minutesToTicksOfLogic(rnd.nextInt(60) - 30);
    activationIntervals.add(new Interval(timeOfLunch, timeOfLunch + TimeUtils.minutesToTicksOfLogic(40), 60)); // lunch

    int timeOfDinner = start + GlobalParameters.LENGHT_OF_A_DAY / 2 + GlobalParameters.LENGHT_OF_A_DAY / 4 + TimeUtils.minutesToTicksOfLogic(rnd.nextInt(60) - 90);
    activationIntervals.add(new Interval(timeOfDinner, timeOfDinner + TimeUtils.minutesToTicksOfLogic(40), 60)); // dinner
    eat.setActivationIntervals(activationIntervals);

    // initialize free time intervals according to the time between meals
    freeTime.add(new Interval(dayBegins + TimeUtils.minutesToTicksOfLogic(60), timeOfLunch - TimeUtils.minutesToTicksOfLogic(15), 10)); // morning
    freeTime.add(new Interval(timeOfLunch + TimeUtils.minutesToTicksOfLogic(60), timeOfDinner - TimeUtils.minutesToTicksOfLogic(25), 10)); // afternoon
    freeTime.add(new Interval(timeOfDinner + TimeUtils.minutesToTicksOfLogic(55), dayEnds - TimeUtils.minutesToTicksOfLogic(60), 10)); // evening
    // add those three intentions to the day schedule
    daySchedule.add(sleep);
    daySchedule.add(eat);
    daySchedule.add(hygiene);
    log.config("after BASIC ACTION: \n" + printIntentions(daySchedule));
  }

  /**
   * so far very simple, got three intervals of time and am assigning activities to those intervals
   * it schedules actions like school, trainings...
   *
   * anyway there are still many issues concerning planning;).
   */
  private void scheduleNormalActions() {
    Intention intention = null;
    ArrayList<Interval> activationIntervals = null;
    boolean toAdd = false;
    int currentDay = this.dayOfWeek; // normal settings for a week planning
    // int currentDay = 1; // everyday is a monday (bad luck for a guy:))
    for (ScheduleEntry entry : this.activityScheduling) { // etries at the beginning got higher possibilities
      intention = null;
      toAdd = false;
      activationIntervals = null;
      for (int i = 0; i < 3; i++) { // morning, afternoon, evening
        if (entry.probabilityOfPerformance(currentDay, i + 1, this.week) >= rnd.nextDouble()) { // TODO:
          if (activationIntervals == null) {
            activationIntervals = new ArrayList<Interval>();
          }
          intention = this.nameToIntention.get(entry.intention);
          toAdd = true;
          this.addInterval(i, intention, activationIntervals);
        }
      }
      if (toAdd) {
        intention.setActivationIntervals(activationIntervals);
        daySchedule.add(intention);
      }
    }
    log.config("after NORMAL ACTIONS: \n" + printIntentions(daySchedule));
  }

  /**
   * add interval is a function which:
   * - takes free time interval on the given index
   * - adds activation interval to the activationIntervals depending on the intention's time limit
   * - creates new free time interval as a remaining time after the addition
   * - put new interval to the freeTime
   * @param index - index in the free time
   * @param intention - intention, which will get the activationIntervals
   * @param activationIntervals - set of activation intervals
   */
  private void addInterval(int index, Intention intention, ArrayList<Interval> activationIntervals) {
    Interval actInterval = null;
    Interval newFreeInterval = null;
    int temp = 0, activation;
    temp = freeTime.get(index).getLeftSide();
    activation = (int) ((rnd.nextGaussian() + 1) * 20) + 25;
    activation = (activation < 20 ? 20 : activation);

    actInterval = new Interval(temp, temp + intention.getTimeLimit(), activation);
    newFreeInterval = new Interval(temp + intention.getTimeLimit() +
            TimeUtils.minutesToTicksOfLogic(5), freeTime.get(index).getRightSide(), 0);
    activationIntervals.add(actInterval);

    freeTime.remove(index);
    freeTime.add(index, newFreeInterval);
  }

  /**
   * other actions are scheduled into the remaining free time, they are chosen from the enum FreeTimeActivities
   * so it can be any of Entertainment, Study, CleanUp
   */
  private void scheduleOtherActions() {
    // play boardgames, relax, free time, ...?
    Intention intention = null;
    Interval interval = null;
    String name = null;
    ArrayList<Interval> activationIntervals = new ArrayList<Interval>();
    log.config("Free time: \n" + printFreeTime());
    rnd.setSeed(System.currentTimeMillis());
    for (int i = 0; i < freeTime.size(); i++) {
      interval = freeTime.get(i);
      while (interval.getIntervalSize() > TimeUtils.minutesToTicksOfLogic(60)) { // when there is more than an hour to do something
        for (FreeTimeActivities trick : FreeTimeActivities.values()) {
          name = trick.toString();
          if (rnd.nextDouble() > 0.7) // arbitrary value to give some chance to either entertainement, studing or some sleep
          {
            break;
          }
        }
        intention = this.nameToIntention.get(name);
        intention.restrictTimeLimits(interval.getIntervalSize());
        if (intention.getActivationIntervals() == null) { // completely new intentions
          this.addInterval(i, intention, activationIntervals);
          intention.setActivationIntervals(activationIntervals);
        } else { // already added intentions -> as it already has activationIntervals, we can't just replace them!
          this.addInterval(i, intention, intention.getActivationIntervals());
        }
        interval = freeTime.get(i); // as it was changed but I don't load it and it holds a pointer on the previous version...
        if (!daySchedule.contains(intention)) {
          this.daySchedule.add(intention);
        }
      }
    }
    log.info("after OTHER ACTIONS: \n" + printIntentions(daySchedule));
  }

  private void scheduleSpecialActions() {
    // go skiing, parties etc.
    }

  /**
   * creates an XML file with a list of ScheduleEntry
   * such a file than can be used as a part of the schedules accessible to the agent
   */
  private static void createFirstXMLEntry() {
    ScheduleEntryListWrapper list = new ScheduleEntryListWrapper();
    list.list.add(ScheduleEntry.getExample());
    list.list.add(ScheduleEntry.getExample());
    try {
      File file = new File("Plan.xml");
      FileOutputStream outputFile = new FileOutputStream(file);
      JAXBContext context = JAXBContext.newInstance(ScheduleEntryListWrapper.class);
      Marshaller m = context.createMarshaller();
      m.marshal(list, outputFile);
    } catch (Exception e) {
      System.err.println("Error in creating Plan.xml! " + e);
    }
  }

  /**
   * loads exml file with ScheduleEntries
   * @param directory - location of the file
   * @param filename - file name
   * @return - list of schedule entries
   */
  private ArrayList<ScheduleEntry> loadPlanning(String directory, String filename) {
    File file = new File(directory + "config\\" + filename);
    ScheduleEntryListWrapper wrapper = null;
    try {
      FileInputStream reader = new FileInputStream(file);
      JAXBContext context = JAXBContext.newInstance(ScheduleEntryListWrapper.class);
      Unmarshaller u = context.createUnmarshaller();
      wrapper = (ScheduleEntryListWrapper) u.unmarshal(reader);
    } catch (Exception e) {
      System.err.println("Error in loading " + filename + "\n" + e);
      log.severe("Error in loading " + filename + "\n" + e);
    }
    return wrapper.list;
  }

  private static String printIntentions(List<Intention> intentions) {
    String result = "";
    for (Intention intention : intentions) {
      result += intention.getName() + " " + intention.getActivationIntervals();
      result += "\n";
    }
    return result;
  }

  /**
   * Changes the plan of the agent.
   * @param directory
   * @param planningFilename
   */
  public void changeLifestyle(String directory, String planningFilename, int alarm) {
    activityScheduling.clear();
    activityScheduling = loadPlanning(directory, planningFilename);
    agentsAlarm = alarm;
  }
}