package decisionMakingSystem;

import java.io.Serializable;
import java.util.ArrayList;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlRootElement;
import utils.Interval;
/**
 * This structure is used to store the schedule entries, 
 * example: when we want agent to go to school for 
 * few months we add to the planing.xml a record with:<br>
 * intention = IStudy<br>
 * weeks = list of intervals of weeks -> (0-12) will tell him to go there for first 3 months <br>
 * executionTimes - [(1, 1, 1.0) ; (2, 2, 0.5) ; (4, 3, 0)] will say, that bot studies every monday morning, 
 *     tuesday afternoon on fifty percent and is not studying thursday evenings 
 *     (though this is implicit and doesn't have to be included)
 * 
 * @author Ondrej
 */
@XmlRootElement
public class ScheduleEntry implements Serializable {
    /**
     * stores probability for the day of a week and a part of a day
     * example: (4,2,0.5) there is a 50% chance, that bot will do the activity on thursday afternoon
     */
    @XmlElement
    private ArrayList<DayPartProbability> executionTimes = null;
    @XmlElement
    /** intervals of weeks when he is doing the activity */
    private ArrayList<Interval> weeks = null;
    @XmlElement
    /** name of the intention */
    public String intention = null; 
    
    /**
     * Return probability of the execution in the given time
     * @param day - day of the week
     * @param part - part of the day -> 0: all day, 1: morning, 2: afternoon, 3: evening
     * @param week - week from the beginning of the simulation
     * @return
     */
    public double probabilityOfPerformance(int day, int part, int week) {
        boolean result = false;
        for (Interval interval : weeks) {
            if (interval.isInInterval(week)) {
                result = true;
                break;
            }
        }
        if (!result)
            return 0;
        for (DayPartProbability execTime : executionTimes) {
            if (execTime.corresponds(day, part))
                return execTime.getProbability();
        }
        return 0;
    }
    
    ScheduleEntry(String intention, ArrayList<Interval> weeks, ArrayList<DayPartProbability> executionTimes) {
        this.intention = intention;
        this.weeks = weeks;
        this.executionTimes = executionTimes;
    }
    
    ScheduleEntry(){};
    
    @Override
    public String toString() {
        String result = "Intention: " + this.intention;
        result += "Execution times: " + this.executionTimes;
        result += "Weeks: " + this.weeks;
        return result;
    }
    
    /**
     * @return example of the ScheduleEntry
     */
    static ScheduleEntry getExample() {
        ScheduleEntry result = new ScheduleEntry();
        result.intention = "ISleep";
        DayPartProbability temp = new DayPartProbability(4,2,0.9);
        result.executionTimes = new ArrayList<DayPartProbability>();
        result.executionTimes.add(temp);
        temp = new DayPartProbability(7,1,0.3);
        result.executionTimes.add(temp);
        result.weeks = new ArrayList<Interval>();
        result.weeks.add(new Interval(2, 5, 0));
        return result;
    }    
}