package decisionMakingSystem;


import atomicActions.*;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.HashSet;
import java.util.logging.Logger;
import javax.xml.bind.annotation.XmlRootElement;
import pogamutEndEvent.PogamutEndEventType;
import utils.DeepCopyCreator;

/**
 * Action is one of the core classes of this decision making modul.
 * It represents the "AND" node of the AND-OR tree
 * it contains a list of affordances which has to be satisfied before executing the action.
 * 
 * Action is satisfied in the following fashion.
 * First it needs to satisfy its intentions (all of them in the defined order), then it 
 * finds the affordances and in the end it performs consecutively atomic actions.
 * 
 * @author Ondrej
 */
@XmlRootElement
public class Action implements Cloneable, Serializable {
  // pointers in the tree structure - intention backward, intentions forward

  /** parental intention which invoked the action */
  public Intention intention = null;
  /** intentions necessary to fulfill the action - for hierarchical actions - in certain order */
  public ArrayList<Intention> intentions = null;
  /** intentions to do - just increase index after fullfiling the current intention */
  public int unfullfiledIntentionsIndex = 0;
  /** written as a part of the agent's name - to show what he's doing*/
  public String name = null;
  /** items that satisfy affordances */
  public final HashMap<AffordanceType, EItem> satisfyingItems = new HashMap<AffordanceType, EItem>();
  /** atomic actions to do in sequence when in the list - moreover I can define the atomic actions to do after fulfilling intentions */
  public ArrayList<AtomicAction> atomicActions = null;    // nakonec mozna bude potreba jen jedna? ale muzu to udelat tak, ze je to bude vykonavat v posloupnosti
  // to mi trochu pritne na sile modelu
  /** index of atomic actions -> set on the actual to perform, increased when action is successfully performed */
  public int atomicActionsIndex = 0;
  /** value of the total activity in the perception field */
  public int activity = 0;
  /** value of the activity of parental intention */
  public int intentionActivity = 0;
  /** if the action is blocked or not */
  public boolean enabled = true;
  // time variables
  /** how long the agent will try to fulfil the action - in number of logic iterations
   * or more precisely in number of executions of its atomic actions
   */
  public int timeLimit = 0;
  /**
   * real time limit - means in the MINUTES, should be given in the action description in the XML
   * on the other hand timeLimit will be overridden by the calculated value from realTimeLimit
   * and the number of iterations for a day
   */
  public double realTimeLimit = 0;
  /** time after which the process will fade out from the short-term memory */
  public int fadeout = 0;
  /** iteration of the logic when the action begins to execute */
  public int start = 0;
  /** how long does the activity takes to execute - for the activities which are not actively represented by activity of agent like sleep*/
  public int duration = 0;
  /** how long is agent trying to do the activity */
  public int time = 0;
  /** state of the action -> success, fail, ... */
  ActionStates state = ActionStates.PRESTATE;
  /** attention give to the action, no clue what does it do so far*/
  public int attention;

  /** specified in XML file - used to determine when to delete node from episodic memory  */
  public int attractivity = 0;

  public Action() {
    this.fadeout = GlobalParameters.FADEOUT_FOR_ACTIONS;
  }
  ;

  public Action(HashSet<AffordanceType> affos) {
    this.fadeout = GlobalParameters.FADEOUT_FOR_ACTIONS;
    Iterator<AffordanceType> it = affos.iterator();
    while (it.hasNext()) {
      satisfyingItems.put(it.next(), null);
    }
  }

  @Override
  public String toString() {
    String result = "ACTION: " + name + " | | ";
    result += "Activity: " + this.activity + " | | " + "Duration: " + this.duration + " | | ";
    result += "Fadeout: " + this.fadeout + " | | " + "Enabled: " + this.enabled + " | | ";
    result += "Attention: " + this.attention + " | | " + "Time: " + this.time + " | | ";
    result += "Time limit: " + this.timeLimit + " | | " + "State: " + this.state + " | | ";
    if (intention != null) {
      result += " Parent intention name: " + this.intention.getName() + " | | ";
    }
    if (intentions != null && !intentions.isEmpty()) {
      result += " Intentions: ";
      for (Intention i : intentions) {
        result += i.getName() + " ";
      }
      result += " INDEX: " + this.unfullfiledIntentionsIndex + " | | ";
    }
    if (atomicActions != null && !atomicActions.isEmpty()) {
      result += " Atomic actions size: " + this.atomicActions.size() + " Actions: ";
      for (AtomicAction act : atomicActions) {
        result += act.type + " ";
      }
      result += " INDEX: " + this.atomicActionsIndex + " | | ";
    }
    if (satisfyingItems != null && !satisfyingItems.isEmpty()) {
      result += " Satisfying affordancese: ";
      for (AffordanceType aff : satisfyingItems.keySet()) {
        if (satisfyingItems.get(aff) != null) {
          result += " aff: " + aff + " item: " + satisfyingItems.get(aff).classOfItem + " | | ";
        } else {
          result += " aff: " + aff + " item: none ";
        }
      }
      result += " | | ";
    }
    return result + "\n";
  }

  /**
   * actually just returns the intention which should be performed now
   * as they are performed in sequential manner
   * @return first unfulfilled intention
   */
  public Intention chooseIntention() {
    Intention result = null;
    if (this.intentions == null) {
      return null;
    }
    if (this.unfullfiledIntentionsIndex < this.intentions.size()) {
      result = this.intentions.get(this.unfullfiledIntentionsIndex);
      result.setActivity(this.activity);
    }
    if (result != null) {
      result.setAncestorAction(this);
    }
    return result;
  }

  /**
   * @return currently executing intention
   */
  public Intention getCurrentIntention() {
    if (this.intentions == null) {
      return null;
    }
    if (this.intentions.size() > this.unfullfiledIntentionsIndex) {
      return this.intentions.get(this.unfullfiledIntentionsIndex);
    } else {
      return null;
    }
  }

  /**
   * @return true if it is spontaneous action - no parental intention
   */
  public boolean isSpontaneous() {
    if (this.intention == null) {
      return true;
    }
    return false;
  }

  /**
   * @return current atomic action to execute
   */
  public AtomicAction getCurrentAtomicAction() {
    if (atomicActions == null) {
      return null;
    }
    if (this.atomicActions.size() > this.atomicActionsIndex) {
      return this.atomicActions.get(this.atomicActionsIndex);
    }
    return null;
  }

  /**
   * move to next atomic action - just rises index
   */
  public void moveToNextAtomicAction() {
    this.atomicActionsIndex += 1;
  }

  /**
   * executes an atomic action
   * if there is not any -> successful execution of the action -> TERMINATED
   * if there is one -> execute it
   *      this can be done for a long time (as the atomic actions fails or succeeds depending on the environment)
   *      it can fails by timeout as well... timeLimit
   * if it succeeds -> move to next action
   * if it fails -> all action failed
   */
  public AtomicAction executeAtomicActions(Logger log) {
    if (this == null || this.state == null || this.state.equals(ActionStates.FAILED) || this.state.equals(ActionStates.TERMINATED)) {
      log.warning("Execute atomic actions called on null, failed or terminated action:");
      return null;
    }
    AtomicAction act = this.getCurrentAtomicAction();
    if (act != null) { // there is some atomic action to do
      // check if agent is on the right place given by Satisfying items - e.g. for swimming he has to be in the pool
      if (act.checkIfOnThePlace()) {
        // execute the action
        act.execute();
      }
    }
    if (act != null && act.succeeded()) {// I still got some action to do and previous finished successfuly
      this.moveToNextAtomicAction();
      if (this.getCurrentAtomicAction() == null) { // there are no other actions
        this.state = ActionStates.TERMINATED;
      }
    } else if (act != null && act.failed()) { // atomic action failed, so all the action failed
      this.state = ActionStates.FAILED;// fail
      if (!act.type.toString().contains("SEARCH_")) // will not write search atomic actions of Want...
      {
        log.info("Atomic action " + act.type + " failed");
      }
    }
    if (act != null) { 
        act.agent.getDecisions().eventSource.fireEvent(PogamutEndEventType.ATOMIC_ACTION_EXECUTED, act);
        act.agent.getDecisions().eventSource.fireEvent(PogamutEndEventType.VISIBLE_OBJECTS, this);
    }

    return act;
  }

  /**
   * makes hard copy of a subtree starting at action via Serializable, as decision module and agent are transient
   * it sets those manually after acquiring hard copy
   * @param agent - link to agent
   * @param dModule - link to decision module
   * @return hard copy of the tree
   */
  public Action cloneBySerialize(DecisionModuleImpl dModule) {
    Action result = (Action) DeepCopyCreator.copy(this);
    ArrayList<Action> actionsWithAtomicActions = new ArrayList<Action>();
    if (result.atomicActions != null) {
      actionsWithAtomicActions.add(result);
    }
    // as i did the hard copy, I lost all connections on decision module and decision module, so
    // i add those connections to all atomic actions in the subtree
    if (result.intentions != null) // sub intentions -> add their actions with atomic actions for further processing
    {
      for (Intention intent : result.intentions) {
        actionsWithAtomicActions.addAll(BasicIntentionLoader.getActionsWithAtomicActions(intent));
      }
    }
    for (Action action : actionsWithAtomicActions) {
      for (AtomicAction aAction : action.atomicActions) {
        aAction.agent = dModule.agent;
        aAction.dModule = dModule;
      }
    }
    return result;
  }

  @Override
  public boolean equals(Object aThat) {
    if (this == aThat) {
      return true;
    }
    if (!(aThat instanceof Action)) {
      return false;
    }
    Action that = (Action) aThat;
    if (this.timeLimit != that.timeLimit) {
      return false;
    }
    if (this.start != that.start) {
      return false;
    }
    if (this.unfullfiledIntentionsIndex != that.unfullfiledIntentionsIndex) {
      return false;
    }
    if ((this.satisfyingItems == null) && (that.satisfyingItems == null)); else if ((this.satisfyingItems == null) || (this.satisfyingItems == null)) {
      return false;
    } else if (!this.satisfyingItems.equals(that.satisfyingItems)) {
      return false;
    }
    if (!this.name.equals(that.name)) {
      return false;
    }
    return true;
  }

  @Override
  public int hashCode() {
    int hash = 5;
    hash = 7 * hash + (this.name != null ? this.name.hashCode() : 0);
    hash = 7 * hash + (this.satisfyingItems != null ? this.satisfyingItems.hashCode() : 0);
    hash = 7 * hash + this.timeLimit;
    return hash;
  }

  /**
   * @return an example instance of action => to create first XML entry for instance
   */
  public static Action exampleAction() {
    Action result = new Action();
    AtomicAction aAction = new SayHello(result, null);
    AtomicAction aAction2 = new SayHello(result, null);
    result.name = "My little action";
    result.atomicActions = new ArrayList<AtomicAction>();
    result.atomicActions.add(aAction);
    result.atomicActions.add(aAction2);
    result.intentions = new ArrayList<Intention>();
    result.state = ActionStates.PRESTATE;
    Intention intention1 = new Intention();
    Intention intention2 = new Intention();
    result.intentions.add(intention1);
    result.intentions.add(intention2);

    EItem item1 = EItem.exampleItem();
    EItem item2 = new EItem();
    result.satisfyingItems.put(AffordanceType.TO_CLEAN, item1);
    return result;
  }

}