package bot;

import decisionMakingSystem.DMSLogger;
import decisionMakingSystem.GlobalParameters;
import java.io.File;
import java.io.Serializable;
import java.util.logging.Logger;
import utils.FileSaveLoadUtils;
import utils.TimeUtils;

/**
 * A controller for experiments. It has two main responsabilities:
 * <ol>
 * <li> Experiment control - setting properties at the designated time or situation etc.
 * <li> Recording - saving pictures, data like errors, GMT, current settings etc.
 * </ok>
 *
 * Agent is relatively indenpendent on the experiment controller. E.g. it can live
 * without it... but no data would be obtained from the simulation.
 *
 *
 * @author Ondrej
 */
public class ExperimentControllerImpl implements ExperimentController, Serializable {

    /** directs to the path for saving JPG shots of memory */
    private String networkImagePath;
    /** directs to the path for saving neural memory to a file */
    private String memoryPath;
    /** directs to the path for saving GMT visualizer to a file */
    private String gmtVizPath;
    /** DMS path */
    private String decisionsPath;
    /** Experiment controller path */
    private String experimentControllerPath;
    /** Connections viewer path */
    private String connectionsPath;
    /** counts saved files */
    private int dayCounter = 0;
    private int hourCounter;
    private ActionManager actionManager = new ActionManager();
    /** how many iteratios bot runs around the map just exploring? */
    private long explorationPeriod = 0;
    /** experiment counter */
    private static String experimentCounter = "";
    private String directory;
    private String experimentDir;
    /** internal counter of ticks of logic */
    private long internalCounter = 0;
    /** special logger which is provided by a DMS and logs agent actions to a file */
    private transient DMSLogger dmsLogger = null;
    /** to log experiment results */
    private transient FileLog experimentLogger = null;
    private transient FileLog extraLogger = null;
    private transient Logger log;
    private transient Bot agent;
    private int conceptNodesLearned = 8;
    private int totalDays = 36;

    public ExperimentControllerImpl(Logger log, String directory, Bot agent) {
        this.log = log;
        this.directory = directory;
        this.agent = agent;
        this.totalDays = agent.lifeTime;
    }

    /**
     * Loads the controller from file on the specified path and injection and preparation of
     * the object (there are several transient objects...).
     */
    static ExperimentController loadFromFile(String path, Logger log, Bot agent) {
        ExperimentControllerImpl exp = (ExperimentControllerImpl) FileSaveLoadUtils.loadObject(path, log);
        exp.log = log;
        exp.agent = agent;
        exp.dmsLogger = new DMSLogger(log, exp.experimentDir + "agent" + experimentCounter + ".log");
        exp.experimentLogger = new DMSLogger(log, exp.experimentDir + "experiment" + experimentCounter + ".log");
        return exp;
    }

    /**
     * Main method for the experiment control.
     *
     * It usually performs daily update of agent and recording of the current situation.
     */
    @Override
    public void controlExperiment() {
        saveToFiles();
        hourCounter = 0;
        actionManager.doEveryDayActions(dayCounter);
        dayCounter++;
//    controlExperimentGMT();
//    controlExperimentTwoPlans();
        controlExperimentLongLife();
    }

    @Override
    public void prepareExperiment() {
        experimentCounter = TimeUtils.now("yyyy-MM-dd_HH.mm.ss.SSS");
        experimentDir = directory + "experiment" + experimentCounter + File.separator;
        File td = new File(experimentDir);
        if (td.mkdir()) {
            log.config("Directory for the experiment was prepared.");
        } else {
            log.fine("Experiment's directory not created! Existing?");
        }
        dmsLogger = new DMSLogger(log, experimentDir + "_agent" + experimentCounter + ".log");
        experimentLogger = new DMSLogger(log, experimentDir + "_experiment" + experimentCounter + ".log");
        extraLogger = new DMSLogger(log, experimentDir + "_extraLog" + experimentCounter + ".log");

        networkImagePath = experimentDir + "Image";
        memoryPath = experimentDir + "Memory";
        gmtVizPath = experimentDir + "GMT";
        decisionsPath = experimentDir + "DMS";
        experimentControllerPath = experimentDir + "ExpController";
        connectionsPath = experimentDir + "Connections";


        //QueryConMemoryAction conMemAction = new QueryConMemoryAction(memory, getExperimentLogger());
        //conMemAction.prepareLoggers(log, experimentDir, experimentCounter);
        // actionManager.addDayAction(conMemAction);

        // choose a lifestyle

        //agent.getDMS().changeLifestyle(directory, EpisodicMemoryParameters.agentMasterPlan, EpisodicMemoryParameters.alarmClock);
        //EpisodicMemoryParameters.writeParameters(experimentLogger);
    }

    @Override
    public void hourUpdate() {
        actionManager.doEveryHourActions(gmtVizPath, dayCounter, hourCounter);
        hourCounter++;
    }

    /**
     * @return the experimentLogger
     */
    @Override
    public FileLog getExperimentLogger() {
        return experimentLogger;
    }

    private void controlExperimentLongLife() {
        int day = (int) ((internalCounter - explorationPeriod) / GlobalParameters.LENGHT_OF_A_DAY);
        //actionManager.addHourAction(new LogGMTAction(agent.getNetworkNFViz()));
        if (day == conceptNodesLearned) {
            getExperimentLogger().logToFile("Slowing down learning rate.");
            //NeuralMemory.setAlpha(NeuralMemory.getAlpha() / 10);
            //NeuralMemory.setBeta(NeuralMemory.getBeta() / 10);
//      NeuralMemory.setDelta(0.05);
        }
        if (day == totalDays) {
            agent.agent.kill();
           // agent.terminate(); //agent.stopAgentSoft();
        }
    }

    /**
     * An experiment for two agents with different lifestyles - when they learn their lifestyles,
     * they change to another lifestyle -> mutual for both and do some queries.
     */
    private void controlExperimentTwoPlans() {
        int day = (int) ((internalCounter - explorationPeriod) / GlobalParameters.LENGHT_OF_A_DAY);
        if (day == conceptNodesLearned) {
//      getExperimentLogger().logToFile("Slowing down learning rate.");
//      NeuralMemory.setAlpha(NeuralMemory.getAlpha() / 10);
//      NeuralMemory.setBeta(NeuralMemory.getBeta() / 10);
            getExperimentLogger().logToFile("Changing to a holidays lifestyle!");
            agent.getDMS().changeLifestyle(directory, "Holidays.xml", 580); // a millionaire who will wake up at 9 a.m.
        }
        if (day == conceptNodesLearned + 12) {
            agent.agent.kill();
           // agent.terminate(); //agent.stopAgentSoft();
        }
    }

    /*
    private int loadExperimentCounter() {
        return (Integer) FileSaveLoadUtils.loadObject(directory + "config\\experimentCounter.cfg", log);
    }

    private void saveExperimentCounter() {
        Integer toSave = new Integer(experimentCounter);
        FileSaveLoadUtils.saveObject(toSave, directory + "config\\experimentCounter.cfg", log);
    }*/

    private void saveToFile(String fileName) {
        FileSaveLoadUtils.saveObject(this, fileName, log);
    }

    private void saveToFiles() {
        saveToFile(fileCounterPrefix(experimentControllerPath));
        //agent.getNetworkNFViz().saveToJPG(fileCounterPrefix(networkImagePath));
        //agent.getGmtNFViz().saveToJPG(fileCounterPrefix(gmtVizPath));
        agent.getDMS().saveToFile(fileCounterPrefix(decisionsPath));
        //agent.getConMemory().saveMemory(fileCounterPrefix(memoryPath), log);
        //agent.getConnectionViewer().saveToJPG(fileCounterPrefix(connectionsPath));
        //agent.getGraphCreator().saveToJPG(fileCounterPrefix(queryDeviationPath));
    }

    /**
     * creates a filename in the shape: path0001 for a fileCounter equals to 1
     * @param path
     * @param fileCounter
     * @return
     */
    public String fileCounterPrefix(String path) {
        return FileSaveLoadUtils.fileCounterPrefix(path, dayCounter, hourCounter);
    }

    @Override
    public void increaseInternalCounter(double time) {
        internalCounter = Math.round(time+1);
    }

    @Override
    public long getInternalCounter() {
        return internalCounter;
    }

    @Override
    public DMSLogger getDMSLogger() {
        return this.dmsLogger;
    }

    /**
     * @return the extraLogger
     */
    public FileLog getExtraLogger() {
        return extraLogger;
    }

    @Override
    public void cleanUp() {
        getDMSLogger().cleanUp();
        getExperimentLogger().cleanUp();
        getExtraLogger().cleanUp();
        actionManager.cleanUp();
    }
}
