package cz.cuni.amis.clear2d.engine.controls;

import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import cz.cuni.amis.clear2d.engine.iface.IUpdatable;

public class Keyboard implements KeyListener, IUpdatable {

	private Map<Integer, KeyState> states = new HashMap<Integer, KeyState>();
	
	private Map<Integer, List<KeyState>> reports = new HashMap<Integer, List<KeyState>>();
	
	private Map<Integer, KeyState> newStates = new HashMap<Integer, KeyState>();
	
	// ================
	// Keyboard Getters
	// ================
	
	/**
	 * @param keyCode see {@link KeyEvent#getKeyCode()}.
	 * @return
	 */
	public KeyState getState(int keyCode) {
		KeyState state = states.get(keyCode);
		if (state == null) return KeyState.NONE;
		return state;
	}
	
	public KeyState getState(String key) {
		List<Integer[]> alternatives = KeyMap.keyCodes.get(key);
		if (alternatives == null) {
			throw new RuntimeException("Key '" + key + "' not supported! Add it into the static initializer of the KeyMap class!");
		}
		
		KeyState result = KeyState.NONE;
		
		// ITERATE THROUGH ALTERNATIVE (or) COMBINATIONS
		for (Integer[] keyCodes : alternatives) {
			
			// HERE WE STORE KeyState FOR THE SEQUENCE (and) COMBINATION OF 'keyCodes'
			KeyState state = null;
			
			// ITERATE THROUGH KEY CODES
			for (Integer keyCode : keyCodes) {
				KeyState keyCodeState = states.get(keyCode);
				if (state == null) state = keyCodeState;
				else 
					if (keyCodeState != null) state = state.andCombination[keyCodeState.id];
			}
			
			// COMBINE keyCodes state WITH result
			if (state != null) result = result.orCombination[state.id];
		}
		
		return result;		
	}
	
	public KeyState getState(Key key) {
		KeyState keyState = getState(key.key);
		
		if (key.ctrl) keyState = keyState.andCombination[getState(KeyEvent.VK_CONTROL).id];
		if (key.alt) keyState = keyState.andCombination[getState(KeyEvent.VK_ALT).id];
		if (key.shift) keyState = keyState.andCombination[getState(KeyEvent.VK_SHIFT).id];
		
		return keyState;
	}
	
	public boolean isPressed(String key) {
		KeyState state = getState(key);
		return state == KeyState.PRESSED || state == KeyState.HOLD;
	}
	
	public boolean isHeld(String key) {
		return getState(key) == KeyState.HOLD;
	}
	
	public boolean isReleased(String key) {
		KeyState state = getState(key);
		return state == KeyState.RELEASED || state == KeyState.NONE;
	}
	
	public boolean isNone(String key) {
		return getState(key) == KeyState.NONE;
	}
	
	public boolean isPressed(Key key) {
		KeyState state = getState(key);
		return state == KeyState.PRESSED || state == KeyState.HOLD;
	}
	
	public boolean isHeld(Key key) {
		return getState(key) == KeyState.HOLD;
	}
	
	public boolean isReleased(Key key) {
		KeyState state = getState(key);
		return state == KeyState.RELEASED || state == KeyState.NONE;
	}
	
	public boolean isNone(Key key) {
		return getState(key) == KeyState.NONE;
	}
	
	// ====================================
	// KeyListener Interface Implementation
	// ====================================
	
	@Override
	public void keyPressed(KeyEvent e) {
		synchronized(reports) {
			List<KeyState> states = ensure(e.getKeyCode());			
			states.add(KeyState.PRESSED);
		}
	}

	@Override
	public void keyReleased(KeyEvent e) {
		synchronized(reports) {
			List<KeyState> states = ensure(e.getKeyCode());			
			states.add(KeyState.RELEASED);
		}
	}

	@Override
	public void keyTyped(KeyEvent e) {
	}
	
	private List<KeyState> ensure(int keyCode) {
		List<KeyState> states = reports.get(keyCode);
		if (states == null) {
			states = new ArrayList<KeyState>(2);
			reports.put(keyCode, states);
		}
		return states;
	}
	
	// ================
	// Keyboard updates
	// ================

	@Override
	public void update() {
		newStates.clear();
		
		synchronized(reports) {
			// CHECK REPORTS
			for (Map.Entry<Integer, List<KeyState>> entry : reports.entrySet()) {
				int keyCode = entry.getKey();
				List<KeyState> states = entry.getValue();
				
				if (states == null || states.size() == 0) continue;
				
				KeyState newState = states.remove(0);
				
				KeyState currState = this.states.get(keyCode);
				if (currState == null) currState = KeyState.NONE;
				
				KeyState nextState = currState.next[newState.id];
				if (currState != nextState) {
					newStates.put(keyCode, nextState);					
				}
			}
		}
			
		synchronized(states) {
			// CHECK STATES
			for (Map.Entry<Integer, KeyState> entry : states.entrySet()) {
				int keyCode = entry.getKey();
				KeyState currState = entry.getValue();
				if (newStates.containsKey(currState)) continue;
				KeyState nextState = currState.next[currState.id];
				if (currState != nextState) {
					newStates.put(keyCode, nextState);
				}
			}
		}
		
		// ALL NEW STATES ARE IN 'newState'
		// => propagate
		synchronized(states) {	
			for (Map.Entry<Integer, KeyState> entry : states.entrySet()) {
				int keyCode = entry.getKey();
				KeyState newState = entry.getValue();
				
				if (newState == KeyState.NONE) {
					states.remove(keyCode);					
				} else {
					KeyState oldState = states.put(keyCode, newState);
					stateChanged(keyCode, oldState, newState);
				}
			}
			
		}
	}

	private void stateChanged(int keyCode, KeyState oldState, KeyState newState) {
		
	}

}
