/*
 * Copyright (C) 2013 Martin Cerny
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package cz.cuni.amis.experiments.impl.metrics;

/**
 * A metric that measures time spent in a given task.
 * Thread safe.
 * @author Martin Cerny
 */
public class TimeMeasuringMetric extends AbstractMetric{

    private long timeConsumed = 0;
    
    private long unaccountedPeriodStart = 0;
    
    private int numRunningInstances = 0;
    
    /**
     * Whether multiple instances of this task are separate processses.
     * If true multiple executions of this task are counted as if they were separate processes - this means
     * two running instances consume two time units per second.
     * Otherwise the time is counted from first instance start to last instance end without any
     * multiplications.  The latter case is useful, if there is some recursive call where each method
     * wants to make sure, the time is counted for its execution.
     */
    private boolean countEachInstanceSeparately;

    /**
     * Creates new time measuring that counts each instance separately
     * @param name 
     */
    public TimeMeasuringMetric(String name){
        this(name, true);
    }
    
    public TimeMeasuringMetric(String name, boolean countEachInstanceSeparately) {
        super(name);
        this.countEachInstanceSeparately = countEachInstanceSeparately;
    }
    
    public synchronized void taskStarted(){
        if(!active){
            return;
        }

        accountLastPeriod();

        numRunningInstances++;
    }

    private void accountLastPeriod() {
        long currentTime = System.currentTimeMillis();
        
        //Account for the last time period
        if(numRunningInstances > 0){
            long currentPeriodLength = currentTime - unaccountedPeriodStart;
            if(countEachInstanceSeparately){
                //count the last period for each task separately
                //note that it works also for numRunningInstances == 0
                timeConsumed += currentPeriodLength * numRunningInstances;
            } else {
                //if the task was running, count the time once no matter the number of instances
                timeConsumed += currentPeriodLength;
            }
        }
        unaccountedPeriodStart = currentTime;
    }
    
    public synchronized void taskFinished(){
        if(!active){
            return;
        }

        if(numRunningInstances <= 0){
            throw new IllegalStateException("Cannot finish a task that has not started. Task:" + getName());
        }

        accountLastPeriod();
        numRunningInstances--;
    }

    
    @Override
    public synchronized Long getValue() {
        //if there are some unfinished tasks, count them
        if(active){
            accountLastPeriod();
        }
        return timeConsumed;
    }

    /**
     * Forces all running instances to be counted as finished. 
     * May be called during abnormal experiment termination.
     */
    @Override
    public void stopMeasurement() {
        if(!active){
            return;
        }
        accountLastPeriod();
        numRunningInstances = 0;
        super.stopMeasurement();
    }

    
    
    
}
