/*
 * Copyright (C) 2013 AMIS research group, Faculty of Mathematics and Physics, Charles University in Prague, Czech Republic
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
package cz.cuni.amis.experiments.impl;

import cz.cuni.amis.experiments.ELogType;
import cz.cuni.amis.experiments.ExperimentException;
import cz.cuni.amis.experiments.IBareLoggingOutput;
import cz.cuni.amis.experiments.ILogCentral;
import cz.cuni.amis.experiments.ILogIdentifier;
import cz.cuni.amis.experiments.ILoggingHeaders;
import cz.cuni.amis.experiments.ILoggingOutput;
import cz.cuni.amis.utils.PairKey;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.apache.log4j.Logger;

/**
 *
 * @author Martin Cerny
 */
public abstract class DefaultLogCentral implements ILogCentral {
    
    private final Logger logger = Logger.getLogger(DefaultLogCentral.class);
    
    protected Map<PairKey<ILogIdentifier, ELogType>, OutputData> registeredOutputs;

    
    
    private boolean active = false;
    
    protected void activeCheck(String opName){
        if(!active){
            throw new ExperimentException("Cannot perform " + opName + " on inactive log central");
        }
    }
    
    @Override
    public synchronized void close() {
        activeCheck("close()");
        active = false;
        for(OutputData outputData : registeredOutputs.values()){
            if(!outputData.requestors.isEmpty()){
                logger.warn("Closing output:" + outputData.output + " but there are requestors holding this output:" + outputData.requestors);
            }
            try {                
                outputData.output.close();
            } catch (IOException ex){
                throw new ExperimentException("Could not close log output: " + outputData.output, ex);
            }
        }
    }

    @Override
    public synchronized void flush() {
        for(OutputData outputData : registeredOutputs.values()){
            try {
                outputData.output.flush();
            } catch (IOException ex){
                throw new ExperimentException("Could not flush log output: " + outputData.output, ex);
            }
        }
    }
    
    

    @Override
    public synchronized void init() {        
        registeredOutputs = new HashMap<PairKey<ILogIdentifier, ELogType>, OutputData>();
        active = true;
    }


    protected abstract ILoggingOutput createLoggingOutput(ILogIdentifier identifier, ELogType logType) throws IOException;
    
    @Override
    public synchronized IBareLoggingOutput requestLoggingOutput(ILogIdentifier identifier, ELogType logType, Object requestor, ILoggingHeaders... loggingHeaders) {
        if(logger.isDebugEnabled()){
            logger.debug("Requested logging output: " + identifier + " - " + logType);
        }
        ILoggingHeaders concatenatedHeaders = LoggingHeadersConcatenation.concatenate(loggingHeaders);
        PairKey<ILogIdentifier, ELogType> key = new PairKey<ILogIdentifier, ELogType>(identifier, logType);
        if(!registeredOutputs.containsKey(key)){
            try {
                ILoggingOutput output = createLoggingOutput(identifier, logType);
                output.init(concatenatedHeaders);                
                OutputData data = new OutputData(output, concatenatedHeaders, requestor);
                registeredOutputs.put(key, data);
                return output;
            } catch (IOException ex){
                throw new ExperimentException("Error creating output for log " + identifier + " - " + logType, ex);
            }
        } else {
            OutputData data = registeredOutputs.get(key);
            if(!data.headers.equals(concatenatedHeaders)){
                throw new ExperimentException("Trying to log with different headers. Log: " + identifier + " - " + logType + " original headers: " + data.headers + " new headers:" + concatenatedHeaders);
            }
            data.requestors.add(requestor);
            return data.output;
        }
    }

    @Override
    public synchronized void releaseLoggingOutput(ILogIdentifier identifier, ELogType logType, Object requestor) {
        if(logger.isDebugEnabled()){
            logger.debug("Releasing logging output: " + identifier + " - " + logType);
        }
        PairKey<ILogIdentifier, ELogType> key = new PairKey<ILogIdentifier, ELogType>(identifier, logType);
        if(!registeredOutputs.containsKey(key)){
            throw new IllegalArgumentException("Cannot release logging output that was never requested. Identifier: " + identifier + " - " + logType);
        }
        registeredOutputs.get(key).requestors.remove(requestor);        
    }
 
    private static class OutputData {
        protected ILoggingOutput output;
        protected Set<Object> requestors;
        protected ILoggingHeaders headers;

        public OutputData(ILoggingOutput output, ILoggingHeaders headers, Object firstRequestor) {
            this.output = output;
            this.headers = headers;
            this.requestors = new HashSet<Object>();
            this.requestors.add(firstRequestor);
        }
        
        
        
    }
}
